// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 19:01:35 $
//
// History:	       $Log: TriangularMaze.java,v $
// History:	       Revision 1.10  2012/06/07 19:01:35  rammi
// History:	       Fixed jdoc ref
// History:
// History:	       Revision 1.9  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.8  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.7  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.6  2005/04/19 20:00:45  rammi
// History:	       no message
// History:	
// History:	       Revision 1.5  2004/11/02 20:53:53  rammi
// History:	       Added keyboard shortcuts etc via actions
// History:	
// History:	       Revision 1.4  2004/10/31 23:30:52  rammi
// History:	       Redraw speed-up
// History:	
// History:	       Revision 1.3  2004/10/31 15:07:05  rammi
// History:	       Changed drawing to be always in BOX_SIZE
// History:	
// History:	       Revision 1.2  2004/10/25 14:54:57  rammi
// History:	       Javadoc changes
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.GeneralPath;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 *  A maze with triangular shape, consisting of triangular cells.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.10 $
 */
public class TriangularMaze
        extends AbstractBasicMaze
{
  /** The type of this maze (for storage). */
  public static final String MAZE_TYPE = "Triangular Maze";

  /** Property key for the settable size property. */
  public static final String PROPERTY_SIZE = "TRIANGLE_MAZE:propertyTriangleSize";

  /** Internally used bit value for direction N. */
  private static final int NORTH = 1;
  /** Internally used bit value for direction S. */
  private static final int SOUTH = 2;
  /** Internally used bit value for direction W. */
  private static final int WEST  = 4;
  /** Internally used bit value for direction E. */
  private static final int EAST  = 8;

  /**
   *  A triangular cell.
   */
  private class TriangularMazeCell extends MazeCell
  {
    /** The id of this cell. */
    private int number;
    /** The connection bitmask. */
    private int connection;
    /** The shape of this cell. */
    private Shape shape;

    /**
     * Constructor.
     * @param num cell number
     */
    TriangularMazeCell(int num)
    {
      this.number = num;
    }

    /**
     *  Forget the connections.
     */
    public void reset()
    {
      connection = 0;
    }

    /**
     * Get the neighbour cells of this one.
     *
     * @return neigbour cells
     */
    public MazeCell[] getNeighbours()
    {
      List<MazeCell> neighbours = new ArrayList<MazeCell>(3);
      int row = (int)Math.sqrt(number);
      int rowStart = row*row;
      int rowEnd   = rowStart+2*row;
      if (number > rowStart) {
        neighbours.add(getCell(number-1));
      }
      if (number < rowEnd) {
        neighbours.add(getCell(number+1));
      }
      if ((number-rowStart)%2 == 1) {
        // peak down
        if (row > 0) {
          neighbours.add(getCell(number-2*row));
        }
      }
      else if (row < size-1) {
        // peak up
        neighbours.add(getCell(number+2*row+2));
      }
      return neighbours.toArray(new MazeCell[neighbours.size()]);
    }

    /**
     * Connect this cell to the given one.
     * The algorithm will also call the inverse method for cell.
     *
     * @param cell cell to connect to
     */
    public void connectTo(MazeCell cell)
    {
      TriangularMazeCell rCell = (TriangularMazeCell)cell;
      int row = (int)Math.sqrt(number);
      int row2 = (int)Math.sqrt(rCell.number);
      if (row == row2) {
        if (number > rCell.number) {
          connection |= WEST;
        }
        else {
          connection |= EAST;
        }
      }
      else {
        if (number > rCell.number) {
          connection |= NORTH;
        }
        else {
          connection |= SOUTH;
        }
      }
    }

    /**
     * Get the connected neighbour cells of this one.
     *
     * @return neigbour cells
     */
    public MazeCell[] getConnectedNeighbours()
    {
      List<MazeCell> neighbours = new ArrayList<MazeCell>(3);
      int row = (int)Math.sqrt(number);
      if ((connection & NORTH) != 0) {
        neighbours.add(getCell(number-2*row));
      }
      if ((connection & SOUTH) != 0) {
        neighbours.add(getCell(number+2*row+2));
      }
      if ((connection & WEST) != 0) {
        neighbours.add(getCell(number-1));
      }
      if ((connection & EAST) != 0) {
        neighbours.add(getCell(number+1));
      }
      return neighbours.toArray(new MazeCell[neighbours.size()]);
    }

    /**
     * Get the connection bitmask.
     * @return connection bitmask
     */
    public int getConnection()
    {
      return connection;
    }

    /**
     *  Get the maze to which this cell belongs.
     *  @return the maze of this cell
     */
    public Maze getMaze()
    {
      return TriangularMaze.this;
    }

    /**
     *  Get the shape of this cell in the current display.
     *  @return maze cell shape
     */
    public Shape getShape()
    {
      return shape;
    }

    /**
     *  Set the shape of this cell in the current display.
     *  @param shape maze cell shape
     */
    public void setShape(Shape shape)
    {
      this.shape = shape;
    }

    /**
     * Get the id of this cell.
     * The id has to be unique for a given geometry of a maze.
     *
     * @return unique id
     * @see Maze#getCellByID(int)
     */
    public int getID()
    {
      return number;
    }
  }


  /** The size of this maze (number of cells with flat side at border) */
  private int size;
  /** The cells of this maze. */
  private TriangularMazeCell[] cells;

  /**
   * Constructor.
   * @param size maze size
   */
  public TriangularMaze(int size)
  {
    initialize(size);
  }

  /**
   * Initialize.
   * @param size maze size
   */
  private void initialize(int size)
  {
    if (size != this.size) {
      int oldSize = this.size;
      this.size = size;
      firePropertyChange(PROPERTY_SIZE, Integer.valueOf(oldSize), Integer.valueOf(size));

      cells = new TriangularMazeCell[size*size];
      for (int i = 0;  i < size*size;  ++i) {
        cells[i] = new TriangularMazeCell(i);
      }
      createShapes();
      setDefaultWayPoints();
    }
  }

  /**
   * Get a deep copy of this maze geometry.
   *
   * @return deep copy
   */
  protected AbstractBasicMaze getGeometryClone()
  {
    return new TriangularMaze(size);
  }

  /**
   *  Get a internally used string describing the maze type.
   *  @return maze type
   */
  public String getMazeType()
  {
    return MAZE_TYPE;
  }

  /**
   *  Resets the internal data.
   *  Should be overwritten, and overwritung methods should call this.
   */
  @Override
  public void reset()
  {
    for (int c = 0;  c < cells.length;  ++c) {
      cells[c].reset();
    }
    super.reset();
  }

  /**
   *  Get the cells of this maze.
   *  @return the cells of this maze
   */
  public MazeCell[] getCells()
  {
    MazeCell[] mCells = new MazeCell[cells.length];
    System.arraycopy(cells, 0, mCells, 0, cells.length);
    return mCells;
  }

  /**
   * Get a cell with a given number.
   * @param nr cell number
   * @return cell with that number
   */
  private TriangularMazeCell getCell(int nr)
  {
    return cells[nr];
  }

  /**
   * Get the cell with the given id.
   *
   * @param id cell id
   * @return the cell with the given id or <code>null</code> if there is no such cell
   * @see MazeCell#getID()
   */
  public MazeCell getCellByID(int id)
  {
    return id >= 0  &&  id < cells.length  ?
            getCell(id)  :
            null;
  }

  /** Outer border shape. */
  private Shape outShape;

  /**
   *  Create the cell shapes.
   */
  private void createShapes()
  {
    final float width  = BOX_SIZE;
    final float height = BOX_SIZE;

    final float deltaX = width/size;
    final float deltaY = height/size;

    outShape = getTriangleShape(0.5f*width, 0, width, height, true);

    for (int row = 0;  row < size;  ++row) {
      float startX = (width-row*deltaX)/2;
      final float startY = row*deltaY;
      boolean up = true;
      for (int c = row*row;  c < (row+1)*(row+1);  ++c, up = !up) {
        final TriangularMazeCell cell = getCell(c);
        cell.setShape(getTriangleShape(startX, startY, deltaX, deltaY, up));

        if (!up) {
          startX += deltaX;
        }
      }
    }
  }

  /**
   * Do the actual drawing.
   * The call to this method is embedded in the the calls to
   * {@link de.caff.maze.MazePainter#startPainting(de.caff.maze.MazePainter.PaintObjectType)} and
   * {@link de.caff.maze.MazePainter#endPaintingMaze()}.
   *
   * @param painter    painter to draw to
   * @param properties access to properties for drawing (colors etc)
   */
  @Override
  protected void doDraw(MazePainter painter, MazePaintPropertiesProvider properties)
  {
    final float width  = BOX_SIZE;
    final float height = BOX_SIZE;

    final float deltaX = width/size;
    final float deltaY = height/size;

    drawBackgroundAndWay(painter, properties);

    painter.setStroke(new BasicStroke(deltaX / 10, BasicStroke.CAP_ROUND, BasicStroke.JOIN_BEVEL));
    if (properties.isShowingCellBorders()) {
      try {
        painter.startPainting(MazePainter.PaintObjectType.CellBorders);
        painter.setPaint(properties.getCellBorderPaint());
        for (int row = 1;  row < size;  ++row) {
          float startX = (width-row*deltaX)/2;
          final float startY = row*deltaY;
          final int rowEnd = (row+1)*(row+1);
          for (int c = row*row+1;  c < rowEnd;  c += 2) {
            int connection = getCell(c).getConnection();
            if ((connection & NORTH) != 0) {
              painter.drawLine(startX, startY, startX + deltaX, startY);
            }
            if ((connection & WEST) != 0) {
              painter.drawLine(startX, startY, startX + deltaX / 2, startY + deltaY);
            }
            if ((connection & EAST) != 0) {
              painter.drawLine(startX + deltaX, startY, startX + deltaX / 2, startY + deltaY);
            }
            startX += deltaX;
          }
        }
      } finally {
        painter.endPainting(MazePainter.PaintObjectType.CellBorders);
      }
    }

    try {
      painter.startPainting(MazePainter.PaintObjectType.InnerWalls);
      painter.setPaint(properties.getInnerWallsPaint());
      for (int row = 1;  row < size;  ++row) {
        float startX = (width-row*deltaX)/2;
        final float startY = row*deltaY;
        final int rowEnd = (row+1)*(row+1);
        for (int c = row*row+1;  c < rowEnd;  c += 2) {
          int connection = getCell(c).getConnection();
          if ((connection & NORTH) == 0) {
            painter.drawLine(startX, startY, startX + deltaX, startY);
          }
          if ((connection & WEST) == 0) {
            painter.drawLine(startX, startY, startX + deltaX / 2, startY + deltaY);
          }
          if ((connection & EAST) == 0) {
            painter.drawLine(startX + deltaX, startY, startX + deltaX / 2, startY + deltaY);
          }
          startX += deltaX;
        }
      }
    } finally {
      painter.endPainting(MazePainter.PaintObjectType.InnerWalls);
    }

    try {
      painter.startPainting(MazePainter.PaintObjectType.OuterWalls);
      painter.setPaint(properties.getOuterWallPaint());
      painter.setStroke(new BasicStroke(deltaX/5));
      painter.draw(outShape);
    } finally {
      painter.endPainting(MazePainter.PaintObjectType.OuterWalls);
    }
  }

  /**
   *  Get the preferred aspect ratio of this maze.
   *  @return aspect ratio (width/height)
   */
  public float getPreferredAspectRatio()
  {
    return 2/(float)Math.sqrt(3);
  }

  /**
   * Get the necessary insets depending on the paint properties.
   * Usually the insets are necessary to allow for the thick border line to be drawn completely.
   *
   * @param properties paint properties
   * @param scaling    scaling used when painting
   * @return insets
   */
  public Insets getInsets(MazePaintPropertiesProvider properties, float scaling)
  {
    float delta = scaling*BOX_SIZE/size;
    int top = (int)Math.ceil(delta/10/Math.sin(Math.toRadians(30)));
    int lr  = (int)Math.ceil(delta/10/Math.sin(Math.toRadians(30) + delta/10/Math.cos(Math.toRadians(30))));
    int bot = (int)Math.ceil(delta/10);
    return new Insets(top, lr, bot, lr);
  }

  /**
   * Create a triangular shape.
   * @param startX x pos
   * @param startY y pos
   * @param deltaX width
   * @param deltaY height
   * @param up     pointing upward?
   * @return triangular shape
   */
  private static Shape getTriangleShape(float startX, float startY, float deltaX, float deltaY, boolean up)
  {
    GeneralPath shape = new GeneralPath();
    shape.moveTo(startX, startY);
    if (up) {
      shape.lineTo(startX-deltaX/2, startY+deltaY);
      shape.lineTo(startX+deltaX/2, startY+deltaY);
    }
    else {
      shape.lineTo(startX+deltaX/2, startY+deltaY);
      shape.lineTo(startX+deltaX, startY);
    }
    shape.closePath();
    return shape;
  }

  /**
   *  Set some useful default way points.
   */
  public void setDefaultWayPoints()
  {
    setWayPoints(getCell(0), getCell(size*size-1));
  }

  /**
   * Set the maze size.
   * @param size maze size (number of triangular cells with flat side at border)
   */
  public void setSize(int size)
  {
    if (this.size != size) {
      initialize(size);
      recreateMaze();
    }
  }

  /**
   *  Property information handling the size of the maze.
   */
  private IntegerDelayedPropertyInformation sizePropertySetter = new IntegerDelayedPropertyInformation(PROPERTY_SIZE, 5, 1000, 1) {
    protected int getMazeValue()
    {
      return size;
    }
  };

  /**
   *  Get the property setters and displays for this maze.
   *
   *  Overriding classes should include this collection first.
   *  @return colletion of property setters
   */
  @Override
  public Collection<PropertyInformation> getPropertyInformations()
  {
    Collection<PropertyInformation> setters  = super.getPropertyInformations();

    setters.add(sizePropertySetter);

    return setters;
  }

  /**
   *  This is called during the call of {@link #setFromSetters()}
   *  and should be used to recreate the geometry of the maze from
   *  the setters of the geometric properties.
   */
  protected void recreateFromDelayedSetters()
  {
    initialize(sizePropertySetter.getValue());
  }

  /**
   *  Get the borders of the maze as a shape.
   *  @return outer border
   */
  protected Shape getOuterBorder()
  {
    return outShape;
  }

  /**
   * Load extra data defining the maze from the system access.
   *
   * @param systemAccess system access
   */
  public void loadPersistentData(DataStorage systemAccess)
  {
    initialize(systemAccess.getInt(PROPERTY_SIZE, size));
    loadSeedWayAndVersion(systemAccess, MAZE_TYPE);
  }

  /**
   * Store extra data defining the maze to the system access.
   *
   * @param systemAccess system access
   */
  public void storePersistentData(DataStorage systemAccess)
  {
    systemAccess.setInt(PROPERTY_SIZE, size);
    storeSeedWayAndVersion(systemAccess, MAZE_TYPE);
  }


  /**
   * Test code.
   * @param args unused
   */
  public static void main(String[] args)
  {
    JFrame frame = new JFrame("TEST");
    final int size = arg2int(args, 0, 150);
    final TriangularMaze maze = new TriangularMaze(size);
    final MazePaintProperties properties = MazePaintProperties.getDrawMazePaintProperties(null);
    maze.createMaze();
    maze.setDefaultWayPoints();
    properties.setShowingSolution(true);
    //maze.setShowingCellBorders(true);
    final MazeCanvas drawarea = new MazeCanvas(maze, properties,
                                              MazePrintProperties.getPrintMazePaintProperties(null));
    drawarea.addMouseListener(new MouseAdapter() {
      @Override public void mouseClicked(MouseEvent e)
      {
        try {
          drawarea.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
          maze.createMaze();
        } finally {
          drawarea.redraw();
          drawarea.setCursor(Cursor.getDefaultCursor());
        }
      }
    });
    frame.getContentPane().add(drawarea);
    frame.setSize(900, 800);

    frame.setVisible(true);

    frame.addWindowListener(new WindowAdapter() {
      @Override public void windowClosing(WindowEvent e)
      {
        System.exit(0);
      }
    });

  }
}
