// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: TemporaryDataStorage.java,v $
// History:	       Revision 1.4  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.3  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.2  2004/10/31 22:07:43  rammi
// History:	       Fixed problem with default ways on first execution
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:55  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import java.awt.*;
import java.util.HashMap;
import java.util.Map;

/**
 * Data storage which has no access to the system.
 * All data it stores is lost when the storage is gone.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @version $Revision: 1.4 $
 */
public class TemporaryDataStorage
        implements DataStorage
{
  /** The persistent data of this session. */
  protected Map<String,Object> persistentData = new HashMap<String,Object>();

  /**
   * Get a persistent string value.
   * A persistent string is a string which should survive the
   * end of the program.
   *
   * @param key          key for the string
   * @param defaultValue default value which is returned if the string is not defined
   * @return the string value defined by the key or the default value
   */
  public String getString(String key, String defaultValue)
  {
    if (persistentData.containsKey(key)) {
      return persistentData.get(key).toString();
    }
    return defaultValue;
  }

  /**
   * Set a persistent string.
   * The string should be stored so it may
   * be read again in this or a later session.
   *
   * @param key   key for the string
   * @param value the string value to store
   */
  public void setString(String key, String value)
  {
    persistentData.put(key, value);
  }

  /**
   * Get a persistent boolean value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the boolean value defined by the key of the default value
   */
  public boolean getBoolean(String key, boolean defaultValue)
  {
    try {
      Boolean b = (Boolean)persistentData.get(key);
      if (b != null) {
        return b.booleanValue();
      }
    } catch (ClassCastException x) {
    }

    return defaultValue;
  }

  /**
   * Set a persistent boolean.
   *
   * @param key   key for the boolean
   * @param value value of the boolean
   */
  public void setBoolean(String key, boolean value)
  {
    persistentData.put(key, value);
  }

  /**
   * Get a persistent color value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the color value deg´fined by the key of the default value
   */
  public Color getColor(String key, Color defaultValue)
  {
    try {
      Color c = (Color)persistentData.get(key);
      if (c != null) {
        return c;
      }
    } catch (ClassCastException x) {
    }
    return defaultValue;
  }

  /**
   * Set a persistent color.
   *
   * @param key   key for the color
   * @param value value of the color
   */
  public void setColor(String key, Color value)
  {
    persistentData.put(key, value);
  }

  /**
   * Get a persistent integer value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the integer value defined by the key of the default value
   */
  public int getInt(String key, int defaultValue)
  {
    try {
      Integer i = (Integer)persistentData.get(key);
      if (i != null) {
        return i.intValue();
      }
    } catch (ClassCastException x) {
    }
    return defaultValue;
  }

  /**
   * Set a persistent integer value.
   *
   * @param key   key for the int
   * @param value value of the int
   */
  public void setInt(String key, int value)
  {
    persistentData.put(key, Integer.valueOf(value));
  }

  /**
   * Get a persistent long integer value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the long integer value defined by the key of the default value
   */
  public long getLong(String key, long defaultValue)
  {
    try {
      Long i = (Long)persistentData.get(key);
      if (i != null) {
        return i.longValue();
      }
    } catch (ClassCastException x) {
    }
    return defaultValue;
  }

  /**
   * Set a persistent long integer value.
   *
   * @param key   key for the long int
   * @param value value of the long int
   */
  public void setLong(String key, long value)
  {
    persistentData.put(key, Long.valueOf(value));
  }

  /**
   * Get a persistent double value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the double value defined by the key of the default value
   */
  public double getDouble(String key, double defaultValue)
  {
    try {
      Double d = (Double)persistentData.get(key);
      if (d != null) {
        return d.doubleValue();
      }
    } catch (ClassCastException x) {
    }
    return defaultValue;
  }

  /**
   * Set a persistent double value.
   *
   * @param key   key for the int
   * @param value value of the int
   */
  public void setDouble(String key, double value)
  {
    persistentData.put(key, Double.valueOf(value));
  }

  /**
   * Is his key defined?
   *
   * @param key key to look for
   * @return <code>true</code> if the key is defined, <code>false</code> otherwise
   */
  public boolean hasKey(String key)
  {
    return persistentData.containsKey(key);
  }

  /**
   * Store all persistent data.
   * This is called when the process is about to shutdown.
   */
  public void storePersistentData()
  {
  }
}
