// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: SystemAccess.java,v $
// History:	       Revision 1.5  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.4  2009/09/24 17:32:20  rammi
// History:	       Finetuning dialog placement
// History:
// History:	       Revision 1.3  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.2  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import de.caff.gimmix.I18n;

import javax.swing.*;
import java.awt.print.Printable;
import java.io.IOException;

/**
 *  Allow access to the underlying system for storage of persisitent
 *  data, file access and printing.
 *
 *  This class abstracts between running as an Application, an Applet or via
 *  Java WebStart.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.5 $
 */
interface SystemAccess extends DataStorage
{
  /**
   *  Just a file extension and the resource for a description of the type it defines.
   */
  public static class FileType
  {
    /** The extensions (not including the dots). */
    private final String[] extensions;
    /** I18n resource for the type descripton. */
    final String resType;

    /**
     *  Constructor.
     *  @param extension extension (w/o dot)
     *  @param resType   i18n resource for the type description
     */
    public FileType(String extension, String resType)
    {
      this.extensions = new String[] { extension };
      this.resType = resType;
    }

    /**
     *  Constructor.
     *  @param extensions extent
     *  @param resType    i18n resource for the type description
     */
    public FileType(String[] extensions, String resType)
    {
      this.extensions = extensions; // not copied because we expect this to be used only in static members
      this.resType = resType;
    }

    /**
     *  Get the extensions.
     *  @return extensions
     */
    public String[] getExtensions()
    {
      String[] ret = new String[extensions.length];
      System.arraycopy(extensions, 0, ret, 0, extensions.length);
      return ret;
    }

    /**
     *  Get the type of this
     *  @return the type
     */
    public String getType()
    {
      return I18n.getString(resType);
    }
  }
  /** The file type of mazes files. */
  public static final FileType MAZE_FILE_TYPE = new FileType("maze", "FILE_TYPE:maze");
  /** The file extensions for images. */
  public static final FileType IMAGE_FILE_TYPE = new FileType("png", "FILE_TYPE:image");

  /**
   *  Print the given printable.
   *  @param printable  printable to be printed
   *  @return <code>true</code>: printing was successful,
   *          <code>false</code>: printing failed or was canceled
   */
  public boolean print(Printable printable);

  /**
   *  Get the contents of a user selected file.
   *  @param filetype type of file to be used
   *  @return byte array or <code>null</code> if the user canceled the operation
   *  @exception java.io.IOException on errors during file access
   */
  public byte[] getUserFileContent(FileType filetype)
          throws IOException;

  /**
   *  Save a byte array into a user selected file.
   *  @param content the content of the file
   *  @param filetype type of file to be used
   *  @return <code>true</code> if the user saved the file, <code>false</code> otherwise
   *  @exception java.io.IOException on errors during file access
   */
  public boolean saveUserFileContent(byte[] content, FileType filetype)
          throws IOException;

  /**
   *  Sets the main frame of the application.
   *  This is useful for centering dialogs.
   *  @param frame application's main frame
   */
  public void setMainFrame(JFrame frame);

}
