// ============================================================================
// File:               $File$
//
// Project:            Maze creator.
//
// Purpose:
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2012  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: SvgMazePainter.java,v $
// History:	       Revision 1.1  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
//=============================================================================
package de.caff.maze;

import java.awt.*;
import java.awt.geom.*;
import java.io.IOException;
import java.io.OutputStream;

/**
 * Painter which outputs the maze into an SVG file.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @version $Revision: 1.1 $
 */
public class SvgMazePainter
        implements MazePainter,
                   Constants
{
  /** File type for DXF files. */
  public static final SystemAccess.FileType FILE_TYPE = new SystemAccess.FileType("svg", "FILE_TYPE:svg");

  private AffineTransform transform;
  private final OutputStream out;
  private Color currentColor = null;
  private Throwable error;

  /**
   * Constructor.
   * @param out output stream
   */
  public SvgMazePainter(OutputStream out)
  {
    this.out = out;
  }

  /**
   * Get any error which happened during usage of the painter.
   * @return error or <code>null</code> if the painter worked nicely
   */
  public Throwable getError()
  {
    return error;
  }

  /**
   *  Output one line into the file.
   *  @param line line to output
   */
  private void outputln(String line)
  {
    output(line);
    output("\n");
  }

  /**
   *  Output something into the file.
   *  @param str text to output
   */
  private void output(String str)
  {
    if (error == null) {
      try {
        out.write(str.getBytes("UTF-8"));
      } catch (IOException e) {
        error = e;
      }
    }
  }

  /**
   *  Write the preamble into the file.
   *  @param width  SVG image width
   *  @param height SVG image height
   */
  private void writePreamble(double width, double height)
  {
    outputln("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");

    outputln("<!DOCTYPE svg PUBLIC '-//W3C//DTD SVG 1.0//EN' 'http://www.w3.org/TR/2001/REC-SVG-20010904/DTD/svg10.dtd'>");
    outputln("<svg style=\"fill-opacity:1; color-rendering:auto; color-interpolation:auto; text-rendering:auto; stroke:black; stroke-linecap:square; stroke-miterlimit:10; shape-rendering:auto; stroke-opacity:1; fill:black; stroke-dasharray:none; font-weight:normal; stroke-width:1; font-family:&apos;sansserif&apos;; font-style:normal; stroke-linejoin:miter; font-size:12; stroke-dashoffset:0; image-rendering:auto;\" xmlns=\"http://www.w3.org/2000/svg\" width=\""+width+"\" xmlns:xlink=\"http://www.w3.org/1999/xlink\" height=\""+height+"\">");
    outputln(String.format("<!-- Generated from irrGardener version %s (see %s) -->", VERSION, WEB_PAGE));
    outputln(" <g>");
    outputln("  <defs id=\"defs1\">");
    outputln("   <clipPath clipPathUnits=\"userSpaceOnUse\" id=\"clipPath\">");
    outputln("     <path d=\"M0 0 L"+width+" 0 L"+width+" "+height+" L0 "+height+" Z\" />");
    outputln("   </clipPath>");
    outputln("  </defs>");
  }

  /**
   *  Write the end of file into the SVG file.
   */
  private void writeEnd()
  {
    if (currentColor != null) {
      outputln("  </g>");
    }
    outputln(" </g>");
    outputln("</svg>");
  }



  /**
   * Start painting the maze.
   *
   * @param maze painted maze
   */
  public void startPaintingMaze(Maze maze)
  {
    transform = AffineTransform.getScaleInstance(1, 1f/maze.getPreferredAspectRatio());
    transform.concatenate(AffineTransform.getTranslateInstance(1, 1));
    writePreamble(1002, 1000/maze.getPreferredAspectRatio()+2);
  }

  /**
   * End painting the maze.
   */
  public void endPaintingMaze()
  {
    writeEnd();
  }

  /**
   * Start painting the given type of paint objects.
   *
   * @param type object type which painting starts
   */
  public void startPainting(PaintObjectType type)
  {
    // ignored
  }

  /**
   * End painting the given type of paint objects.
   *
   * @param type object type which painting has ended
   */
  public void endPainting(PaintObjectType type)
  {
    // ignored
  }

  /**
   * Set a stroke.
   *
   * @param stroke new stroke to use in upcoming drawing commands
   */
  public void setStroke(Stroke stroke)
  {
    // ignored
  }

  /**
   * Set a paint.
   *
   * @param paint paint to use in upcoming drawing commands.
   */
  public void setPaint(Paint paint)
  {
    if (paint instanceof Color) {
      Color color = (Color)paint;
      if (!color.equals(currentColor)) {
        String rgb = "rgb("+color.getRed()+","+color.getGreen()+","+color.getBlue()+")";
        if (currentColor != null) {
          outputln("  </g>");
        }
        outputln("  <g  style=\"fill:"+rgb+"; stroke:"+rgb+"; fill-rule:evenodd;\">");
        currentColor = color;
      }
    }
  }

  /**
   * Draw a line with current paint and stroke..
   *
   * @param startX starting point X coordinate
   * @param startY starting point Y coordinate
   * @param endX   ending point X coordinate
   * @param endY   ending point Y coordinate
   */
  public void drawLine(float startX, float startY, float endX, float endY)
  {
    Point2D start = new Point2D.Double(startX, startY);
    Point2D end   = new Point2D.Double(endX, endY);
    transform.transform(start, start);
    transform.transform(end, end);
    outputln(String.format("    <line x1=\"%s\" y1=\"%s\" x2=\"%s\" y2=\"%s\"/>",
                           Double.toString(start.getX()),
                           Double.toString(start.getY()),
                           Double.toString(end.getX()),
                           Double.toString(end.getY())));
  }

  /**
   * Draw an open arc with current paint and stroke.
   *
   * @param x      x of rectangle enclosing ellipse
   * @param y      y of rectangle enclosing ellipse
   * @param w      width of rectangle enclosing ellipse
   * @param h      height of rectangle enclosing ellipse
   * @param start  start of angle in degrees, from x axis,
   * @param extent extent of angle in degrees
   */
  public void drawArc(float x, float y, float w, float h, float start, float extent)
  {
    draw(new Arc2D.Float(x, y, w, h, start, extent, Arc2D.OPEN));
  }

  private void outputSegment(char type,
                              Point2D ... points)
  {
    output(Character.toString(type));
    Point2D point = new Point2D.Double();
    for (Point2D p : points) {
      transform.transform(p, point);
      output(Double.toString(point.getX()));
      output(" ");
      output(Double.toString(point.getY()));
      output(" ");
    }
  }

  /**
   * Output a shape.
   * @param shape   shape to output
   * @param filled  fill shape?
   */
  private void outputShape(Shape shape, boolean filled)
  {
    output("    <path d=\"");
    double[] coords = new double[6];
    for (PathIterator pit = shape.getPathIterator(null);  !pit.isDone();  pit.next()) {
      switch (pit.currentSegment(coords)) {
      case PathIterator.SEG_MOVETO:
        outputSegment('M', new Point2D.Double(coords[0], coords[1]));
        break;

      case PathIterator.SEG_LINETO:
        outputSegment('L', new Point2D.Double(coords[0], coords[1]));
        break;

      case PathIterator.SEG_QUADTO:
        outputSegment('Q',
                      new Point2D.Double(coords[0], coords[1]),
                      new Point2D.Double(coords[2], coords[3]));
        break;

      case PathIterator.SEG_CUBICTO:
        outputSegment('C',
                      new Point2D.Double(coords[0], coords[1]),
                      new Point2D.Double(coords[2], coords[3]),
                      new Point2D.Double(coords[4], coords[5]));
        break;

      case PathIterator.SEG_CLOSE:
        output("Z ");
        break;
      }
    }
    output("\"");
    if (!filled) {
      output(" style=\"fill:none;\"");
    }
    outputln(" />");
  }

  /**
   * Draw a shape with current paint and stroke.
   *
   * @param shape shape to draw
   */
  public void draw(Shape shape)
  {
    outputShape(shape, false);
  }

  /**
   * Fill a shape with the current paint.
   *
   * @param shape shape to fill
   */
  public void fill(Shape shape)
  {
    outputShape(shape, true);
  }
}
