// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: Stringizer.java,v $
// History:	       Revision 1.3  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.2  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.1.1.1  2004/10/25 14:47:55  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import java.awt.*;

/**
 *  Makes strings from values and vice versa.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.3 $
 */
class Stringizer
{
  /** Used for boolean <code>true</code>. */
  private static final String TRUE_STRING  = "true";
  /** Used for boolean <code>false</code>. */
  private static final String FALSE_STRING = "false";
  /** Used for Color <code>null</code>. */
  private static final String NULL_COLOR   = "#<-null->";
  /**
   *  Convert a color into a string.
   *  @param color  color to convert
   *  @return stringized color value
   *  @see #colorFromString(String, java.awt.Color)
   */
  public static String colorToString(Color color)
  {
    return color != null  ?
            String.format("#%02x%02x%02x%02x", color.getAlpha(), color.getRed(), color.getGreen(), color.getBlue())  :
            NULL_COLOR;
  }

  /**
   *  Convert a string to a color.
   *  @param str          string which defines a color in the form <tt>#aarrggbb</tt>
   *  @param defaultValue color to be used if string does not define a color
   *  @return  color from string or default color
   */
  public static Color colorFromString(String str, Color defaultValue)
  {
    if (str != null  &&  str.startsWith("#")  &&  str.length() == 9) {
      try {
        int alpha = Integer.parseInt(str.substring(1, 3), 16);
        int red   = Integer.parseInt(str.substring(3, 5), 16);
        int green = Integer.parseInt(str.substring(5, 7), 16);
        int blue  = Integer.parseInt(str.substring(7, 9), 16);
        return new Color(red, green, blue, alpha);
      } catch (Exception x) {
      }
    }
    else if (NULL_COLOR.equals(str)) {
      return null;
    }
    return defaultValue;
  }

  /**
   *  Convert a boolean to a string.
   *  @param b  boolean value
   *  @return stringized form of boolean value
   */
  public static String booleanToString(boolean b)
  {
    return b ? TRUE_STRING : FALSE_STRING;
  }

  /**
   *  Convert a string into a boolean value.
   *  @param str           string to convert
   *  @param defaultValue  default value to be used if the string does not define a boolean value
   *  @return value from the string or default value
   */
  public static boolean booleanFromString(String str, boolean defaultValue)
  {
    if (TRUE_STRING.equals(str)) {
      return true;
    }
    else if (FALSE_STRING.equals(str)) {
      return false;
    }
    return defaultValue;
  }

  /**
   *  Convert a int value into a string.
   *  @param value  int value
   *  @return stringized form of int value
   */
  public static String intToString(int value)
  {
    return Integer.toString(value);
  }

  /**
   *  Convert a string into an int value.
   *  @param str          string to convert
   *  @param defaultValue default value to be used if the string does not define an int value
   *  @return value from string or default value
   */
  public static int intFromString(String str, int defaultValue)
  {
    if (str != null) {
      try {
        return Integer.parseInt(str);
      } catch (NumberFormatException e) {
      }
    }
    return defaultValue;
  }

  /**
   *  Convert a long int value into a string.
   *  @param value  long int value
   *  @return stringized form of long int value
   */
  public static String longToString(long value)
  {
    return Long.toString(value);
  }

  /**
   *  Convert a string into a long int value.
   *  @param str          string to convert
   *  @param defaultValue default value to be used if the string does not define a long int value
   *  @return value from string or default value
   */
  public static long longFromString(String str, long defaultValue)
  {
    if (str != null) {
      try {
        return Long.parseLong(str);
      } catch (NumberFormatException e) {
      }
    }
    return defaultValue;
  }

  /**
   *  Convert a double value into a string.
   *  @param value  double value
   *  @return stringized form of double value
   */
  public static String doubleToString(double value)
  {
    return Double.toString(value);
  }

  /**
   *  Convert a string into a double value.
   *  @param str          string to convert
   *  @param defaultValue default value to be used if the string does not define a double value
   *  @return value from string or default value
   */
  public static double doubleFromString(String str, double defaultValue)
  {
    if (str != null) {
      try {
        return Double.parseDouble(str);
      } catch (NumberFormatException e) {
      }
    }
    return defaultValue;
  }
}
