// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 19:01:35 $
//
// History:	       $Log: RectangularMaze.java,v $
// History:	       Revision 1.9  2012/06/07 19:01:35  rammi
// History:	       Fixed jdoc ref
// History:
// History:	       Revision 1.8  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.7  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.6  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.5  2004/11/02 20:53:53  rammi
// History:	       Added keyboard shortcuts etc via actions
// History:	
// History:	       Revision 1.4  2004/10/31 23:30:52  rammi
// History:	       Redraw speed-up
// History:	
// History:	       Revision 1.3  2004/10/31 15:07:05  rammi
// History:	       Changed drawing to be always in BOX_SIZE
// History:	
// History:	       Revision 1.2  2004/10/25 14:54:57  rammi
// History:	       Javadoc changes
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.Line2D;
import java.awt.geom.Point2D;
import java.awt.geom.Rectangle2D;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 *  A rectangular maze with square cells.
 *
 *  This is what I used to draw on checkered paper when I was a child.
 *  Some of my friends loved to solve them and asked for bigger ones,
 *  so one day I made a really large one by using 8 DIN A4 papers with 5mm squares,
 *  resulting in a maze with approximately 18000 cells.
 *  Nobody was able to solve it, until 25 years later my own son took the time to do it.
 *  Drawing took some hours, but solving it took it lot more. Now you can do both
 *  (creating and solving) in mere seconds with this and other programs but I'm not convinced
 *  that this is really a progress.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.9 $
 */
public class RectangularMaze
        extends AbstractBasicMaze
{
  /** The type of this maze (for storage). */
  public static final String MAZE_TYPE = "Rectangular Maze";

  /** Property key for the settable number of horizontal cells property. */
  public static final String PROPERTY_HORIZONTAL_CELLS = "RECT_MAZE:nrHorizontalCells";
  /** Property key for the settable number of vertical cells property. */
  public static final String PROPERTY_VERTICAL_CELLS   = "RECT_MAZE:nrVerticalCells";

  /** Internally used bit value for direction N. */
  private static final int NORTH = 1;
  /** Internally used bit value for direction S. */
  private static final int SOUTH = 2;
  /** Internally used bit value for direction W. */
  private static final int WEST  = 4;
  /** Internally used bit value for direction E. */
  private static final int EAST  = 8;

  /**
   * A square cell.
   */
  private class RectangularMazeCell extends MazeCell
  {
    /** Position x (column). */
    private int x;
    /** Position y (row). */
    private int y;
    /** Connections as a bitmask. */
    private int connection;
    /** The shape of this cell. */
    private Shape shape;

    /**
     * Constructor.
     * @param x  x position (column)
     * @param y  y position (row)
     */
    RectangularMazeCell(int x, int y)
    {
      this.x = x;
      this.y = y;
    }

    public void reset()
    {
      connection = 0;
    }

    /**
     * Get the neighbour cells of this one.
     *
     * @return neigbour cells
     */
    public MazeCell[] getNeighbours()
    {
      List<MazeCell> neighbours = new ArrayList<MazeCell>(4);
      if (x > 0) {
        neighbours.add(getCell(x-1, y));
      }
      if (x < nrHorizontal-1) {
        neighbours.add(getCell(x+1, y));
      }
      if (y > 0) {
        neighbours.add(getCell(x, y-1));
      }
      if (y < nrVertical-1) {
        neighbours.add(getCell(x, y+1));
      }
      return neighbours.toArray(new MazeCell[neighbours.size()]);
    }

    /**
     * Connect this cell to the given one.
     * The algorithm will also call the inverse method for cell.
     *
     * @param cell cell to connect to
     */
    public void connectTo(MazeCell cell)
    {
      RectangularMazeCell rCell = (RectangularMazeCell)cell;
      if (rCell.x == x) {
        if (rCell.y < y) {
          connection |= NORTH;
        }
        else {
          connection |= SOUTH;
        }
      }
      else {
        if (rCell.x < x) {
          connection |= WEST;
        }
        else {
          connection |= EAST;
        }
      }
    }

    /**
     * Get the connected neighbour cells of this one.
     *
     * @return neigbour cells
     */
    public MazeCell[] getConnectedNeighbours()
    {
      List<MazeCell> neighbours = new ArrayList<MazeCell>(4);
      if ((connection & NORTH) != 0) {
        neighbours.add(getCell(x, y-1));
      }
      if ((connection & SOUTH) != 0) {
        neighbours.add(getCell(x, y+1));
      }
      if ((connection & WEST) != 0) {
        neighbours.add(getCell(x-1, y));
      }
      if ((connection & EAST) != 0) {
        neighbours.add(getCell(x+1, y));
      }
      return neighbours.toArray(new MazeCell[neighbours.size()]);
    }

    /**
     * Get the connection bitmask.
     * @return connection bitmask
     */
    public int getConnection()
    {
      return connection;
    }

    /**
     *  Get the maze to which this cell belongs.
     *  @return the maze of this cell
     */
    public Maze getMaze()
    {
      return RectangularMaze.this;
    }

    /**
     *  Get the shape of this cell in the current display.
     *  @return maze cell shape
     */
    public Shape getShape()
    {
      return shape;
    }

    /**
     *  Set the shape of this cell.
     *  @param shape maze cell shape
     */
    public void setShape(Shape shape)
    {
      this.shape = shape;
    }

    /**
     * Get the id of this cell.
     * The id has to be unique for a given geometry of a maze.
     *
     * @return unique id
     * @see Maze#getCellByID(int)
     */
    public int getID()
    {
      return y*nrHorizontal + x;
    }

  }


  /** Number of horizontal cells. */
  private int nrHorizontal;
  /** Number of vertical cells. */
  private int nrVertical;
  /** The cells. */
  private RectangularMazeCell[] cells;

  /**
   * Constructor.
   * @param horizontalCells number of horizontal cells
   * @param verticalCells   number of vertical cells
   */
  public RectangularMaze(int horizontalCells, int verticalCells)
  {
    initialize(horizontalCells, verticalCells);
  }

  /**
   * Initialize.
   * @param horizontalCells number of horizontal cells
   * @param verticalCells   number of vertical cells
   */
  private void initialize(int horizontalCells, int verticalCells)
  {
    boolean differs = horizontalCells != nrHorizontal  ||  verticalCells != nrVertical;

    if (differs) {
      if (this.nrHorizontal != horizontalCells) {
        int oldValue = this.nrHorizontal;
        this.nrHorizontal = horizontalCells;
        firePropertyChange(PROPERTY_HORIZONTAL_CELLS,
                           new Integer(oldValue),
                           new Integer(horizontalCells));
      }
      if (this.nrVertical != verticalCells) {
        int oldValue = this.nrVertical;
        this.nrVertical = verticalCells;
        firePropertyChange(PROPERTY_VERTICAL_CELLS,
                                new Integer(oldValue),
                                new Integer(verticalCells));
      }
      cells = new RectangularMazeCell[nrHorizontal*nrVertical];
      for (int x = 0;  x < nrHorizontal;  ++x) {
        for (int y = 0;  y < nrVertical;  ++y) {
          cells[x*nrVertical+y] = new RectangularMazeCell(x, y);
        }
      }
      createShapes();
      setDefaultWayPoints();
    }
  }

  /**
   *  Resets the internal data.
   *  Should be overwritten, and overwritung methods should call this.
   */
  @Override
  public void reset()
  {
    for (int c = 0;  c < cells.length;  ++c) {
      cells[c].reset();
    }
    super.reset();
  }

  /**
   * Get a deep copy of this maze geometry.
   *
   * @return deep copy
   */
  protected AbstractBasicMaze getGeometryClone()
  {
    return new RectangularMaze(nrHorizontal, nrVertical);
  }

  /**
   *  Get a internally used string describing the maze type.
   *  @return maze type
   */
  public String getMazeType()
  {
    return MAZE_TYPE;
  }

  /**
   *  Get the cells of this maze.
   *  @return the cells of this maze
   */
  public MazeCell[] getCells()
  {
    MazeCell[] mCells = new MazeCell[cells.length];
    System.arraycopy(cells, 0, mCells, 0, cells.length);
    return mCells;
  }

  /**
   * Get the cell at a given row/column combination.
   * @param x  x coord (column)
   * @param y  y coord (row)
   * @return cell at given coordinates
   */
  private RectangularMazeCell getCell(int x, int y)
  {
    return cells[x*nrVertical+y];
  }

  /** The shape of the outer border. */
  private Shape outShape;

  /**
   * Create the shapes.
   */
  private void createShapes()
  {
    final float width  = BOX_SIZE;
    final float height = BOX_SIZE;

    final float deltaX = width/nrHorizontal;
    final float deltaY = height/nrVertical;

    outShape = new Rectangle.Float(0, 0, width, height);

    for (int x = 0;  x < nrHorizontal;  ++x) {
      for (int y = 0;  y < nrVertical;  ++y) {
        final RectangularMazeCell cell = getCell(x, y);
        cell.setShape(new Rectangle2D.Double(x*deltaX, y*deltaY, deltaX, deltaY));
      }
    }

  }

  /**
   * Get the cell with the given id.
   *
   * @param id cell id
   * @return the cell with the given id or <code>null</code> if there is no such cell
   * @see MazeCell#getID()
   */
  public MazeCell getCellByID(int id)
  {
    final int x = id%nrHorizontal;
    final int y = id/nrHorizontal;
    return (y >= 0  &&   y < nrVertical  &&  x >= 0   &&   x < nrHorizontal)  ?
            getCell(x, y)  :
            null;
  }


  /**
   * Do the actual drawing.
   * The call to this method is embedded in the the calls to
   * {@link de.caff.maze.MazePainter#startPaintingMaze(Maze)} and
   * {@link de.caff.maze.MazePainter#endPaintingMaze()}.
   *
   * @param painter    painter to draw to
   * @param properties access to properties for drawing (colors etc)
   */
  @Override
  protected void doDraw(MazePainter painter, MazePaintPropertiesProvider properties)
  {
    final float width  = BOX_SIZE;
    final float height = BOX_SIZE;

    final float deltaX = width/nrHorizontal;
    final float deltaY = height/nrVertical;

    drawBackgroundAndWay(painter, properties);

    painter.setStroke(new BasicStroke(deltaX / 10));
    if (properties.isShowingCellBorders()) {
      painter.startPainting(MazePainter.PaintObjectType.CellBorders);
      try {
        painter.setPaint(properties.getCellBorderPaint());
        for (int x = 0;  x < nrHorizontal;  ++x) {
          for (int y = 0;  y < nrVertical;  ++y) {
            final RectangularMazeCell cell = getCell(x, y);
            final int connection = cell.getConnection();
            if ((connection & NORTH) != 0) {
              painter.drawLine(x * deltaX, y * deltaY, (x + 1) * deltaX, y * deltaY);
            }
            if ((connection & WEST) != 0) {
              painter.drawLine(x * deltaX, y * deltaY, x * deltaX, (y + 1) * deltaY);
            }
          }
        }
      } finally {
        painter.endPainting(MazePainter.PaintObjectType.CellBorders);
      }
    }

    try {
      painter.startPainting(MazePainter.PaintObjectType.InnerWalls);
      painter.setPaint(properties.getInnerWallsPaint());
      for (int x = 0;  x < nrHorizontal;  ++x) {
        for (int y = 0;  y < nrVertical;  ++y) {
          final RectangularMazeCell cell = getCell(x, y);
          final int connection = cell.getConnection();
          if ((connection & NORTH) == 0) {
            painter.draw(new Line2D.Float(x * deltaX, y * deltaY, (x + 1) * deltaX, y * deltaY));
          }
          if ((connection & WEST) == 0) {
            painter.draw(new Line2D.Float(x * deltaX, y * deltaY, x * deltaX, (y + 1) * deltaY));
          }
        }
      }
    } finally {
      painter.endPainting(MazePainter.PaintObjectType.InnerWalls);
    }

    try {
      painter.startPainting(MazePainter.PaintObjectType.OuterWalls);
      painter.setPaint(properties.getOuterWallPaint());
      painter.setStroke(new BasicStroke(deltaX/5));
      painter.draw(outShape);
    } finally {
      painter.endPainting(MazePainter.PaintObjectType.OuterWalls);
    }

  }

  /**
   *  Get the preferred aspect ratio of this maze.
   *  @return aspect ratio (width/height)
   */
  public float getPreferredAspectRatio()
  {
    return nrHorizontal/(float)nrVertical;
  }

  /**
   * Get the necessary insets depending on the paint properties.
   * Usually the insets are necessary to allow for the thick border line to be drawn completely.
   *
   * @param properties paint properties
   * @param scaling    scaling used when painting
   * @return insets
   */
  public Insets getInsets(MazePaintPropertiesProvider properties, float scaling)
  {
    float width = BOX_SIZE/nrHorizontal;
    int margin = (int)Math.ceil(scaling*width/12);
    return new Insets(margin, margin, margin, margin);
  }

  /**
   *  Set some useful default way points.
   */
  public void setDefaultWayPoints()
  {
    setWayPoints(getCell(0, 0), getCell(nrHorizontal-1, nrVertical-1));
  }

  /**
   * Set the number of horizontal cells.
   * @param nrHorizontal number of horizontal cells
   */
  private void setNrHorizontal(int nrHorizontal)
  {
    if (this.nrHorizontal != nrHorizontal) {
      initialize(nrHorizontal, nrVertical);
      recreateMaze();
    }
  }

  /**
   * Set the number of vertical cells.
   * @param nrVertical number of vertical cells
   */
  private void setNrVertical(int nrVertical)
  {
    if (this.nrVertical != nrVertical) {
      initialize(nrHorizontal, nrVertical);
      recreateMaze();
    }
  }

  /**
   *  Property information handling the number of horizontal cells.
   */
  private IntegerDelayedPropertyInformation numberHorizontalPropertySetter = new IntegerDelayedPropertyInformation(PROPERTY_HORIZONTAL_CELLS, 10, 1000, 1) {
    protected int getMazeValue()
    {
      return nrHorizontal;
    }

    protected void setMazeValue(int value)
    {
      setNrHorizontal(value);
    }
  };

  /**
   *  Property information handling the number of vertical cells.
   */
  private IntegerDelayedPropertyInformation numberVerticalPropertySetter = new IntegerDelayedPropertyInformation(PROPERTY_VERTICAL_CELLS, 10, 1000, 1) {
    protected int getMazeValue()
    {
      return nrVertical;
    }

    protected void setMazeValue(int value)
    {
      setNrVertical(value);
    }
  };

    /**
   *  Get the property setters and displays for this maze.
   *
   *  Overriding classes should include this collection first.
   *  @return colletion of property setters
   */
  @Override
  public Collection<PropertyInformation> getPropertyInformations()
  {
    Collection<PropertyInformation> setters  = super.getPropertyInformations();

    setters.add(numberHorizontalPropertySetter);
    setters.add(numberVerticalPropertySetter);

    return setters;
  }

  /**
   *  This is called during the call of {@link #setFromSetters()}
   *  and should be used to recreate the geometry of the maze from
   *  the setters of the geometric properties.
   */
  protected void recreateFromDelayedSetters()
  {
    initialize(numberHorizontalPropertySetter.getValue(),
               numberVerticalPropertySetter.getValue());
  }

  /**
   *  Get the borders of the maze as a shape.
   *  @return outer border
   */
  protected Shape getOuterBorder()
  {
    return outShape;
  }

  /**
   *  Get the cell at a given point. This method uses the second worst algorithm (first checking
   *  the outer border of the maze, and if necessary checking each
   *  cell shape whether it contains the point), so extending
   *  classes are encouraged to use their better understanding of the underlying geometry
   *  to calculate the target cell faster.
   *  @param position  cell position
   *  @return cell at position or <code>null</code> if there is no cell at the given position
   */
  @Override
  public MazeCell getCellAt(Point2D position)
  {
    int col = (int)(position.getX()/(BOX_SIZE/nrHorizontal));
    int row = (int)(position.getY()/(BOX_SIZE/nrVertical));
    if (col >= 0  &&  col < nrHorizontal  &&  row >= 0  &&   row < nrVertical) {
      return getCell(col, row);
    }
    else {
      return null;
    }
  }

  /**
   * Load extra data defining the maze from the system access.
   *
   * @param systemAccess system access
   */
  public void loadPersistentData(DataStorage systemAccess)
  {
    initialize(systemAccess.getInt(PROPERTY_HORIZONTAL_CELLS, nrHorizontal),
               systemAccess.getInt(PROPERTY_VERTICAL_CELLS, nrVertical));
    loadSeedWayAndVersion(systemAccess, MAZE_TYPE);
  }

  /**
   * Store extra data defining the maze to the system access.
   *
   * @param systemAccess system access
   */
  public void storePersistentData(DataStorage systemAccess)
  {
    systemAccess.setInt(PROPERTY_HORIZONTAL_CELLS, nrHorizontal);
    systemAccess.setInt(PROPERTY_VERTICAL_CELLS, nrVertical);
    storeSeedWayAndVersion(systemAccess, MAZE_TYPE);
  }

  /**
   * Test code.
   * @param args unused
   */
  public static void main(String[] args)
  {
    JFrame frame = new JFrame("TEST");
    final int width = 120;
    final int height = 90;
    final RectangularMaze maze = new RectangularMaze(width, height);
    final MazePaintProperties properties = MazePaintProperties.getDrawMazePaintProperties(null);
    final MazeCanvas drawarea = new MazeCanvas(maze, properties,
                                              MazePrintProperties.getPrintMazePaintProperties(null));
    maze.createMaze();
    maze.setDefaultWayPoints();
    properties.setShowingSolution(true);
    properties.setShowingCellBorders(false);
    drawarea.addMouseListener(new MouseAdapter() {
      @Override public void mouseClicked(MouseEvent e)
      {
        try {
          drawarea.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
          maze.createMaze();
        } finally {
          drawarea.redraw();
          drawarea.setCursor(Cursor.getDefaultCursor());
        }
      }
    });
    frame.getContentPane().add(drawarea);
    frame.setSize(800, 600);

    frame.setVisible(true);

    frame.addWindowListener(new WindowAdapter() {
      @Override public void windowClosing(WindowEvent e)
      {
        System.exit(0);
      }
    });

  }
}
