// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 19:01:35 $
//
// History:	       $Log: OctogonalMaze.java,v $
// History:	       Revision 1.6  2012/06/07 19:01:35  rammi
// History:	       Fixed jdoc ref
// History:
// History:	       Revision 1.5  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.4  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.3  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.2  2004/11/02 20:53:53  rammi
// History:	       Added keyboard shortcuts etc via actions
// History:	
// History:	       Revision 1.1  2004/10/31 15:04:58  rammi
// History:	       First version
// History:	
//=============================================================================
package de.caff.maze;

import java.awt.*;
import java.awt.geom.GeneralPath;
import java.util.ArrayList;
import java.util.Collection;

/**
 * A maze with octogonal cells and smaller square cells in a checker board pattern.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @version $Revision: 1.6 $
 */
public class OctogonalMaze
        extends AbstractBasicMaze
{
  /** The type of this maze (for storage). */
  public static final String MAZE_TYPE = "Octogonal Maze";

  /** Property key for the settable number of horizontal cells property. */
  public static final String PROPERTY_HORIZONTAL_CELLS = "OCTAGON_MAZE:nrHorizontalCells";
  /** Property key for the settable number of vertical cells property. */
  public static final String PROPERTY_VERTICAL_CELLS   = "OCTAGON_MAZE:nrVerticalCells";

  /** Internally used bit value for direction W. */
  private static final int WEST = 1;
  /** Internally used bit value for direction E. */
  private static final int EAST = 2;
  /** Internally used bit value for direction N. */
  private static final int NORTH = 4;
  /** Internally used bit value for direction S. */
  private static final int SOUTH = 8;
  /** Internally used bit value for direction NW. */
  private static final int NORTH_WEST  = 16;
  /** Internally used bit value for direction NE. */
  private static final int NORTH_EAST  = 32;
  /** Internally used bit value for direction SW. */
  private static final int SOUTH_WEST  = 64;
  /** Internally used bit value for direction SE. */
  private static final int SOUTH_EAST  = 128;

  /**
   *  An octagonal cell.
   */
  private class OctogonalMazeCell extends MazeCell
  {
    /** Position x. */
    private final int x;
    /** Position y. */
    private final int y;
    /** Connection bit mask. */
    private int connection;
    /** Shape. */
    private Shape shape;

    /**
     * Constructor.
     * @param x  position x
     * @param y  position y
     */
    OctogonalMazeCell(int x, int y)
    {
      this.x = x;
      this.y = y;
    }

    /**
     *  Forget all connections.
     */
    public void reset()
    {
      connection = 0;
    }

    /**
     * Get the neighbour cells of this one.
     *
     * @return neigbour cells
     */
    public MazeCell[] getNeighbours()
    {
      java.util.List<MazeCell> neighbours = new ArrayList<MazeCell>(8);
      if (x > 0) {
        neighbours.add(getOctogonalCell(x-1, y));
        if (y > 0) {
          neighbours.add(getSquareCell(x-1, y-1));
        }
        if (y < nrVertical-1) {
          neighbours.add(getSquareCell(x-1, y));
        }
      }
      if (x < nrHorizontal-1) {
        neighbours.add(getOctogonalCell(x+1, y));
        if (y > 0) {
          neighbours.add(getSquareCell(x, y-1));
        }
        if (y < nrVertical-1) {
          neighbours.add(getSquareCell(x, y));
        }
      }
      if (y > 0) {
        neighbours.add(getOctogonalCell(x, y-1));
      }
      if (y < nrVertical - 1) {
        neighbours.add(getOctogonalCell(x, y+1));
      }
      return neighbours.toArray(new MazeCell[neighbours.size()]);
    }

    /**
     *  Get the maze to which this cell belongs.
     *  @return the maze of this cell
     */
    public Maze getMaze()
    {
      return OctogonalMaze.this;
    }

    /**
     * Get the id of this cell.
     * The id has to be unique for a given geometry of a maze.
     *
     * @return unique id
     * @see Maze#getCellByID(int)
     */
    public int getID()
    {
      return y*nrHorizontal + x;
    }

    /**
     * Connect this cell to the given one.
     * The algorithm will also call the inverse method for cell.
     *
     * @param cell cell to connect to
     */
    public void connectTo(MazeCell cell)
    {
      if (cell instanceof OctogonalMazeCell) {
        OctogonalMazeCell oCell = (OctogonalMazeCell)cell;
        if (oCell.y == y) {
          if (oCell.x < x) {
            connection |= WEST;
          }
          else {
            connection |= EAST;
          }
        }
        else if (oCell.x == x) {
          if (oCell.y < y) {
            connection |= NORTH;
          }
          else {
            connection |= SOUTH;
          }
        }
      }
      else {
        SquareMazeCell sCell = (SquareMazeCell)cell;
        if (sCell.getX() == x) {
          if (sCell.getY() == y) {
            connection |= SOUTH_EAST;
          }
          else {
            connection |= NORTH_EAST;
          }
        }
        else {
          if (sCell.getY() == y) {
            connection |= SOUTH_WEST;
          }
          else {
            connection |= NORTH_WEST;
          }
        }
      }
    }

    /**
     * Get the connected neighbour cells of this one.
     *
     * @return neigbour cells
     */
    public MazeCell[] getConnectedNeighbours()
    {
      java.util.List<MazeCell> neighbours = new ArrayList<MazeCell>(8);
      if ((connection & WEST) != 0) {
        neighbours.add(getOctogonalCell(x-1, y));
      }
      if ((connection & EAST) != 0) {
        neighbours.add(getOctogonalCell(x+1, y));
      }
      if ((connection & NORTH) != 0) {
        neighbours.add(getOctogonalCell(x, y-1));
      }
      if ((connection & SOUTH) != 0) {
        neighbours.add(getOctogonalCell(x, y+1));
      }
      if ((connection & NORTH_WEST) != 0) {
        neighbours.add(getSquareCell(x-1, y-1));
      }
      if ((connection & NORTH_EAST) != 0) {
        neighbours.add(getSquareCell(x, y-1));
      }
      if ((connection & SOUTH_WEST) != 0) {
        neighbours.add(getSquareCell(x-1, y));
      }
      if ((connection & SOUTH_EAST) != 0) {
        neighbours.add(getSquareCell(x, y));
      }
      return neighbours.toArray(new MazeCell[neighbours.size()]);
    }

    /**
     * Get the connection bitmask.
     * @return connection bitmask
     */
    public int getConnection()
    {
      return connection;
    }

    /**
     *  Get the shape of this cell in the current display.
     *  @return maze cell shape
     */
    public Shape getShape()
    {
      return shape;
    }

    /**
     * Set the shape.
     * @param shape new shape
     */
    public void setShape(Shape shape)
    {
      this.shape = shape;
    }
  }

  /**
   *  A square cell.
   */
  private class SquareMazeCell
          extends MazeCell
  {
    /** Position x. */
    private int x;
    /** Position y. */
    private int y;
    /** Connection bitmask. */
    private int connection;
    /** Shape. */
    private Shape shape;

    /**
     * Constructor.
     * @param x  position x
     * @param y  position y
     */
    SquareMazeCell(int x, int y)
    {
      this.x = x;
      this.y = y;
    }

    public void reset()
    {
      connection = 0;
    }

    /**
     * Get the neighbour cells of this one.
     *
     * @return neigbour cells
     */
    public MazeCell[] getNeighbours()
    {
      return new MazeCell[] {
        getOctogonalCell(x, y),
        getOctogonalCell(x, y+1),
        getOctogonalCell(x+1, y),
        getOctogonalCell(x+1, y+1)
      };
    }

    /**
     *  Get the maze to which this cell belongs.
     *  @return the maze of this cell
     */
    public Maze getMaze()
    {
      return OctogonalMaze.this;
    }

    /**
     * Get the id of this cell.
     * The id has to be unique for a given geometry of a maze.
     *
     * @return unique id
     * @see Maze#getCellByID(int)
     */
    public int getID()
    {
      return nrHorizontal*nrVertical + y*(nrHorizontal-1) + x;
    }

    /**
     * Connect this cell to the given one.
     * The algorithm will also call the inverse method for cell.
     *
     * @param cell cell to connect to
     */
    public void connectTo(MazeCell cell)
    {
      OctogonalMazeCell oCell = (OctogonalMazeCell)cell;
      if (oCell.x == x) {
        if (oCell.y == y) {
          connection |= NORTH_WEST;
        }
        else {
          connection |= SOUTH_WEST;
        }
      }
      else {
        if (oCell.y == y) {
          connection |= NORTH_EAST;
        }
        else {
          connection |= SOUTH_EAST;
        }
      }
    }

    /**
     * Get the connected neighbour cells of this one.
     *
     * @return neigbour cells
     */
    public MazeCell[] getConnectedNeighbours()
    {
      java.util.List<MazeCell> neighbours = new ArrayList<MazeCell>(4);
      if ((connection & NORTH_WEST) != 0) {
        neighbours.add(getOctogonalCell(x, y));
      }
      if ((connection & NORTH_EAST) != 0) {
        neighbours.add(getOctogonalCell(x+1, y));
      }
      if ((connection & SOUTH_WEST) != 0) {
        neighbours.add(getOctogonalCell(x, y+1));
      }
      if ((connection & SOUTH_EAST) != 0) {
        neighbours.add(getOctogonalCell(x+1, y+1));
      }
      return neighbours.toArray(new MazeCell[neighbours.size()]);
    }

    /**
     *  Get the connection bitmask.
     *  @return bitmask defining connections
     */
    public int getConnection()
    {
      return connection;
    }

    /**
     *  Get the shape of this cell in the current display.
     *  @return maze cell shape
     */
    public Shape getShape()
    {
      return shape;
    }

    /**
     * Set the shape of this cell.
     * @param shape cell shape
     */
    public void setShape(Shape shape)
    {
      this.shape = shape;
    }

    /**
     * Get the x of the cells position.
     * @return cell column
     */
    public int getX()
    {
      return x;
    }

    /**
     * Get the y of the cells position.
     * @return cell row
     */
    public int getY()
    {
      return y;
    }
  }


  /** Number of horizontal cells. */
  private int nrHorizontal;
  /** Number of vertical cells. */
  private int nrVertical;
  /** The orthogonal cells. */
  private OctogonalMazeCell[] octogonalCells;
  /** The square cells. */
  private SquareMazeCell[]    squareCells;
  /** Predefined numerical factor. */
  private static final float FACTOR1 = (float)(Math.sqrt(0.5)/(1+Math.sqrt(2)));
  /** Predefined numerical factor. */
  private static final float FACTOR2 = (float)((1+Math.sqrt(0.5))/(1+Math.sqrt(2)));

  /**
   *  Constructor.
   *  @param nrHorizontal number of horizontal cells
   *  @param nrVertical   number of vertical cells
   */
  public OctogonalMaze(int nrHorizontal, int nrVertical)
  {
    initialize(nrHorizontal, nrVertical);
  }

  /**
   * Create cells.
   *  @param horizontalCells number of horizontal cells (columns)
   *  @param verticalCells   number of vertical cells   (rows)
   */
  private void initialize(int horizontalCells, int verticalCells)
  {
    if (this.nrHorizontal != horizontalCells  ||
        this.nrVertical   != verticalCells) {
      if (this.nrHorizontal != horizontalCells) {
        int oldValue = this.nrHorizontal;
        this.nrHorizontal = horizontalCells;
        firePropertyChange(PROPERTY_HORIZONTAL_CELLS,
                           new Integer(oldValue),
                           new Integer(horizontalCells));
      }
      if (this.nrVertical != verticalCells) {
        int oldValue = this.nrVertical;
        this.nrVertical = verticalCells;
        firePropertyChange(PROPERTY_VERTICAL_CELLS,
                                new Integer(oldValue),
                                new Integer(verticalCells));
      }
      octogonalCells = new OctogonalMazeCell[nrHorizontal*nrVertical];
      for (int x = 0;  x < nrHorizontal;  ++x) {
        for (int y = 0;  y < nrVertical;  ++y) {
          octogonalCells[y*nrHorizontal+x] = new OctogonalMazeCell(x, y);
        }
      }
      squareCells  = new SquareMazeCell[(nrHorizontal-1)*(nrVertical-1)];
      for (int x = 0;  x < nrHorizontal-1;  ++x) {
        for (int y = 0;  y < nrVertical-1;  ++y) {
          squareCells[y*(nrHorizontal-1)+x] = new SquareMazeCell(x, y);
        }
      }

      createShapes();
      setDefaultWayPoints();
    }
  }

  /**
   * Get a deep copy of this maze geometry.
   *
   * @return deep copy
   */
  protected AbstractBasicMaze getGeometryClone()
  {
    return new HexagonalMaze(nrHorizontal, nrVertical);
  }

  /**
   *  Get a internally used string describing the maze type.
   *  @return maze type
   */
  public String getMazeType()
  {
    return MAZE_TYPE;
  }

  /**
   *  Resets the internal data.
   *  Should be overwritten, and overwritung methods should call this.
   */
  @Override
  public void reset()
  {
    for (OctogonalMazeCell cell: octogonalCells) {
      cell.reset();
    }
    for (SquareMazeCell cell: squareCells) {
      cell.reset();
    }
    super.reset();
  }

  /**
   * Get the cells of this maze.
   * @return the cells of this maze
   */
  public MazeCell[] getCells()
  {
    MazeCell[] cells = new MazeCell[octogonalCells.length+squareCells.length];
    System.arraycopy(octogonalCells, 0, cells, 0, octogonalCells.length);
    System.arraycopy(squareCells, 0, cells, octogonalCells.length, squareCells.length);
    return cells;
  }

  /** The outer border. */
  private Shape outShape;

  /**
   *  Create the cell shapes.
   */
  private void createShapes()
  {
    final float width  = BOX_SIZE;
    final float height = BOX_SIZE;

    final float deltaX = width/nrHorizontal;
    final float deltaY = height/nrVertical;

    outShape = getOuterBounds(deltaX, deltaY);

    for (int y = 0;  y < nrVertical;  ++y) {
      for (int x = 0;  x < nrHorizontal;  ++x) {
        final OctogonalMazeCell cell = getOctogonalCell(x, y);
        cell.setShape(getOctogonalShape(x*deltaX, y*deltaY, deltaX, deltaY));
      }
    }
    for (int y = 1;  y < nrVertical;  ++y) {
      for (int x = 1;  x < nrHorizontal;  ++x) {
        final SquareMazeCell cell = getSquareCell(x-1, y-1);
        cell.setShape(getSquareShape(x*deltaX, y*deltaY, deltaX, deltaY));
      }
    }
  }


  /**
   * Get the cell with the given id.
   *
   * @param id cell id
   * @return the cell with the given id or <code>null</code> if there is no such cell
   * @see MazeCell#getID()
   */
  public MazeCell getCellByID(int id)
  {
    if (id < octogonalCells.length) {
      if (id >= 0) {
        return octogonalCells[id];
      }
    }
    else {
      id -= octogonalCells.length;
      if (id < squareCells.length) {
        return squareCells[id];
      }
    }
    return null;
  }

  /**
   * Do the actual drawing.
   * The call to this method is embedded in the the calls to
   * {@link de.caff.maze.MazePainter#startPaintingMaze(Maze)} and
   * {@link de.caff.maze.MazePainter#endPaintingMaze()}.
   *
   * @param painter    painter to draw to
   * @param properties access to properties for drawing (colors etc)
   */
  @Override
  protected void doDraw(MazePainter painter, MazePaintPropertiesProvider properties)
  {
    final float width  = BOX_SIZE;
    final float height = BOX_SIZE;

    final float deltaX = width/nrHorizontal;
    final float deltaY = height/nrVertical;

    drawBackgroundAndWay(painter, properties);

    painter.setStroke(new BasicStroke(deltaX / 12, BasicStroke.CAP_ROUND, BasicStroke.JOIN_MITER));
    if (properties.isShowingCellBorders()) {
      try {
        painter.startPainting(MazePainter.PaintObjectType.CellBorders);
        painter.setPaint(properties.getCellBorderPaint());
        for (int y = 0;  y < nrVertical;  ++y) {
          final float yOffset = y*deltaY;
          for (int x = 0;  x < nrHorizontal;  ++x) {
            final float xOffset = x*deltaX;
            final OctogonalMazeCell cell = getOctogonalCell(x, y);
            final int connection = cell.getConnection();
            if ((connection & SOUTH_WEST) != 0) {
              painter.drawLine(xOffset+FACTOR1*deltaX, yOffset+deltaY,
                               xOffset, yOffset+FACTOR2*deltaY);
            }
            if ((connection & WEST) != 0) {
              painter.drawLine(xOffset, yOffset+FACTOR2*deltaY,
                               xOffset, yOffset+FACTOR1*deltaY);
            }
            if ((connection & NORTH_WEST) != 0) {
              painter.drawLine(xOffset, yOffset+FACTOR1*deltaY,
                               xOffset+FACTOR1*deltaX, yOffset);
            }
            if ((connection & NORTH) != 0) {
              painter.drawLine(xOffset+FACTOR1*deltaX, yOffset,
                               xOffset+FACTOR2*deltaX, yOffset);
            }
            if ((connection & NORTH_EAST) != 0) {
              painter.drawLine(xOffset+FACTOR2*deltaX, yOffset,
                               xOffset+deltaX, yOffset+FACTOR1*deltaY);
            }
            if ((connection & SOUTH_EAST) != 0) {
              painter.drawLine(xOffset+deltaX, yOffset+FACTOR2*deltaY,
                               xOffset+FACTOR2*deltaX, yOffset+deltaY);
            }
          }
        }
      } finally {
        painter.endPainting(MazePainter.PaintObjectType.CellBorders);
      }
    }

    try {
      painter.startPainting(MazePainter.PaintObjectType.InnerWalls);
      painter.setPaint(properties.getInnerWallsPaint());
      for (int y = 0;  y < nrVertical;  ++y) {
        final float yOffset = y*deltaY;
        for (int x = 0;  x < nrHorizontal;  ++x) {
          final float xOffset = x*deltaX;
          final OctogonalMazeCell cell = getOctogonalCell(x, y);
          final int connection = cell.getConnection();
          if ((connection & SOUTH_WEST) == 0) {
            painter.drawLine(xOffset + FACTOR1 * deltaX, yOffset + deltaY,
                             xOffset, yOffset + FACTOR2 * deltaY);
          }
          if ((connection & WEST) == 0) {
            painter.drawLine(xOffset, yOffset + FACTOR2 * deltaY,
                             xOffset, yOffset + FACTOR1 * deltaY);
          }
          if ((connection & NORTH_WEST) == 0) {
            painter.drawLine(xOffset, yOffset + FACTOR1 * deltaY,
                             xOffset + FACTOR1 * deltaX, yOffset);
          }
          if ((connection & NORTH) == 0) {
            painter.drawLine(xOffset + FACTOR1 * deltaX, yOffset,
                             xOffset + FACTOR2 * deltaX, yOffset);
          }
          if ((connection & NORTH_EAST) == 0) {
            painter.drawLine(xOffset + FACTOR2 * deltaX, yOffset,
                             xOffset + deltaX, yOffset + FACTOR1 * deltaY);
          }
          if ((connection & SOUTH_EAST) == 0) {
            painter.drawLine(xOffset + deltaX, yOffset + FACTOR2 * deltaY,
                             xOffset + FACTOR2 * deltaX, yOffset + deltaY);
          }
        }
      }
    } finally {
      painter.endPainting(MazePainter.PaintObjectType.InnerWalls);
    }

    try {
      painter.startPainting(MazePainter.PaintObjectType.OuterWalls);
      painter.setPaint(properties.getOuterWallPaint());
      painter.setStroke(new BasicStroke(deltaX / 6));
      painter.draw(outShape);
    } finally {
      painter.endPainting(MazePainter.PaintObjectType.OuterWalls);
    }
  }

  /**
   *  Get the preferred aspect ratio of this maze.
   *  @return aspect ratio (width/height)
   */
  public float getPreferredAspectRatio()
  {
    return (float)nrHorizontal/nrVertical;
  }

  /**
   * Get the necessary insets depending on the paint properties.
   * Usually the insets are necessary to allow for the thick border line to be drawn completely.
   *
   * @param properties paint properties
   * @param scaling    scaling used when painting
   * @return insets
   */
  public Insets getInsets(MazePaintPropertiesProvider properties, float scaling)
  {
    float width = scaling * BOX_SIZE / nrHorizontal;
    int margin = (int)Math.ceil(width/12);
    return new Insets(margin, margin, margin, margin);
  }

  /**
   * Create a octogonal shape.
   * @param startX start x
   * @param startY start y
   * @param deltaX cell width
   * @param deltaY cell height
   * @return octogonal shape
   */
  private static Shape getOctogonalShape(float startX, float startY, float deltaX, float deltaY)
  {
    GeneralPath shape = new GeneralPath();
    shape.moveTo(startX+FACTOR2*deltaX, startY);
    shape.lineTo(startX+FACTOR1*deltaX, startY);
    shape.lineTo(startX, startY+FACTOR1*deltaY);
    shape.lineTo(startX, startY+FACTOR2*deltaY);
    shape.lineTo(startX+FACTOR1*deltaX, startY+deltaY);
    shape.lineTo(startX+FACTOR2*deltaX, startY+deltaY);
    shape.lineTo(startX+deltaX, startY+FACTOR2*deltaY);
    shape.lineTo(startX+deltaX, startY+FACTOR1*deltaY);
    shape.closePath();
    return shape;
  }

  /**
   * Create a square shape.
   * @param startX start x
   * @param startY start y
   * @param deltaX cell width
   * @param deltaY cell height
   * @return square shape
   */
  private static Shape getSquareShape(float startX, float startY, float deltaX, float deltaY)
  {
    GeneralPath shape = new GeneralPath();
    shape.moveTo(startX-FACTOR1*deltaX, startY);
    shape.lineTo(startX, startY-FACTOR1*deltaY);
    shape.lineTo(startX+FACTOR1*deltaX, startY);
    shape.lineTo(startX, startY+FACTOR1*deltaY);
    shape.closePath();
    return shape;
  }

  /**
   * Create the outer bounds shape.
   * @param deltaX cell width
   * @param deltaY cell height
   * @return outer bounds shape
   */
  private Shape getOuterBounds(float deltaX, float deltaY)
  {
    GeneralPath shape = new GeneralPath();
    shape.moveTo(FACTOR1*deltaX, 0);
    for (int x = 0;  x < nrHorizontal;  ++x) {
      if (x > 0) {
        shape.lineTo((x+FACTOR1)*deltaX, 0);
      }
      shape.lineTo((x+FACTOR2)*deltaX, 0);
      shape.lineTo((x+1)*deltaX, FACTOR1*deltaY);
    }
    final float rightBorder = deltaX*nrHorizontal;
    for (int y = 0;  y < nrVertical;  ++y) {
      if (y > 0) {
        shape.lineTo(rightBorder, (y+FACTOR1)*deltaY);
      }
      shape.lineTo(rightBorder, (y+FACTOR2)*deltaY);
      shape.lineTo(rightBorder-FACTOR1*deltaX, (y+1)*deltaY);
    }
    final float lowerBorder = deltaY*nrVertical;
    for (int x = nrHorizontal-1;  x >= 0;  --x) {
      if (x < nrHorizontal-1) {
        shape.lineTo((x+FACTOR2)*deltaX, lowerBorder);
      }
      shape.lineTo((x+FACTOR1)*deltaX, lowerBorder);
      shape.lineTo(x*deltaX, lowerBorder-FACTOR1*deltaY);
    }
    for (int y = nrVertical-1;  y >= 0;  --y) {
      if (y < nrVertical-1) {
        shape.lineTo(0, (y+FACTOR2)*deltaY);
      }
      shape.lineTo(0, (y+FACTOR1)*deltaY);
      shape.lineTo(FACTOR1*deltaX, y*deltaY);
    }
    shape.closePath();
    return shape;
  }


  /**
   * Get a certain octogonal cell.
   * @param col   column
   * @param row   row
   * @return cell at given position
   */
  private OctogonalMazeCell getOctogonalCell(int col, int row)
  {
    return octogonalCells[row*nrHorizontal+col];
  }

  /**
   * Get a certain square cell.
   * @param col   column
   * @param row   row
   * @return cell at given position
   */
  private SquareMazeCell getSquareCell(int col, int row)
  {
    return squareCells[row*(nrHorizontal-1)+col];
  }

  /**
   *  Set some useful default way points.
   */
  public void setDefaultWayPoints()
  {
    setWayPoints(getOctogonalCell(0, 0), getOctogonalCell(nrHorizontal-1, nrVertical-1));
  }

  /**
   * Set the number of horizontal cells.
   * @param nrHorizontal number of horizontal cells
   */
  private void setNrHorizontal(int nrHorizontal)
  {
    if (this.nrHorizontal != nrHorizontal) {
      initialize(nrHorizontal, nrVertical);
      recreateMaze();
    }
  }

  /**
   * Set the number of vertical cells.
   * @param nrVertical number of vertical cells
   */
  private void setNrVertical(int nrVertical)
  {
    if (this.nrVertical != nrVertical) {
      initialize(nrHorizontal, nrVertical);
      recreateMaze();
    }
  }

  /**
   *  Property information handling the number of horizontal cells.
   */
  private IntegerDelayedPropertyInformation numberHorizontalPropertySetter = new IntegerDelayedPropertyInformation(PROPERTY_HORIZONTAL_CELLS, 10, 1000, 1) {
    protected int getMazeValue()
    {
      return nrHorizontal;
    }

    protected void setMazeValue(int value)
    {
      setNrHorizontal(value);
    }
  };

  /**
   *  Property information handling the number of vertical cells.
   */
  private IntegerDelayedPropertyInformation numberVerticalPropertySetter = new IntegerDelayedPropertyInformation(PROPERTY_VERTICAL_CELLS, 10, 1000, 1) {
    protected int getMazeValue()
    {
      return nrVertical;
    }

    protected void setMazeValue(int value)
    {
      setNrVertical(value);
    }
  };

  /**
   *  Get the property setters and displays for this maze.
   *
   *  Overriding classes should include this collection first.
   *  @return colletion of property setters
   */
  @Override
  public Collection<PropertyInformation> getPropertyInformations()
  {
    Collection<PropertyInformation> setters = super.getPropertyInformations();

    setters.add(numberHorizontalPropertySetter);
    setters.add(numberVerticalPropertySetter);

    return setters;
  }

  /**
   *  This is called during the call of {@link #setFromSetters()}
   *  and should be used to recreate the geometry of the maze from
   *  the setters of the geometric properties.
   */
  protected void recreateFromDelayedSetters()
  {
    initialize(numberHorizontalPropertySetter.getValue(),
               numberVerticalPropertySetter.getValue());
  }

  /**
   *  Get the borders of the maze as a shape.
   *  @return outer border
   */
  protected Shape getOuterBorder()
  {
    return outShape;
  }

  /**
   * Load extra data defining the maze from the system access.
   *
   * @param systemAccess system access
   */
  public void loadPersistentData(DataStorage systemAccess)
  {
    initialize(systemAccess.getInt(PROPERTY_HORIZONTAL_CELLS, nrHorizontal),
               systemAccess.getInt(PROPERTY_VERTICAL_CELLS, nrVertical));
    loadSeedWayAndVersion(systemAccess, MAZE_TYPE);
  }

  /**
   * Store extra data defining the maze to the system access.
   *
   * @param systemAccess system access
   */
  public void storePersistentData(DataStorage systemAccess)
  {
    systemAccess.setInt(PROPERTY_HORIZONTAL_CELLS, nrHorizontal);
    systemAccess.setInt(PROPERTY_VERTICAL_CELLS, nrVertical);
    storeSeedWayAndVersion(systemAccess, MAZE_TYPE);
  }

}
