// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: MultiMazeControlPanel.java,v $
// History:	       Revision 1.7  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.6  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.5  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.4  2005/04/19 20:00:45  rammi
// History:	       no message
// History:	
// History:	       Revision 1.3  2004/11/02 20:53:53  rammi
// History:	       Added keyboard shortcuts etc via actions
// History:	
// History:	       Revision 1.2  2004/10/31 15:06:07  rammi
// History:	       Added octogonal maze
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import de.caff.gimmix.I18n;
import de.caff.gimmix.ResourcedAction;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ItemEvent;
import java.awt.event.ItemListener;
import java.awt.image.BufferedImage;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.Map;

/**
 *  A control panel able to handle different mazes.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.7 $
 */
public class MultiMazeControlPanel
        extends JPanel
{
  /** I18n resource for create button. */
  public static final String RESOURCE_BUTTON_CREATE = "MULTI_MAZE_CONTROL_PANEL:buttonCreate";
  /** I18n resource for reset button. */
  public static final String RESOURCE_BUTTON_RESET = "MULTI_MAZE_CONTROL_PANEL:buttonReset";
  /** I18n resource for print error text. */
  public static final String RESOURCE_PRINT_ERROR  = "MULTI_MAZE_CONTROL_PANEL:printErrorText";

  /**
   *  A MazeCanvas and its assiociated control.
   */
  private class MazeControlPair
  {
    /** Name of this pair. */
    private String            name;
    /** The MazeCanvas. */
    private MazeCanvas        mazeCanvas;
    /** The Controller. */
    private JComponent        mazeControlPanel;

    /**
     * Constructor.
     * Creates a maze canvas and a control panel for the maze.
     * @param name          name of the pair (for card layouts)
     * @param maze          the maze
     * @param systemAccess  access to data storage to read settings
     */
    public MazeControlPair(String name, AbstractBasicMaze maze, DataStorage systemAccess)
    {
      this.name        = name;
      maze.loadPersistentData(systemAccess);
      mazeCanvas       = new MazeCanvas(maze, paintProperties, printProperties);
      mazeControlPanel = new PropertySetterPanel(maze.getPropertyInformations());
    }

    /**
     *  Get the maze canvas.
     *  @return maze canvas
     */
    public MazeCanvas getMazeCanvas()
    {
      return mazeCanvas;
    }

    /**
     * Get the control panel.
     * @return control panel
     */
    public JComponent getMazeControlPanel()
    {
      return mazeControlPanel;
    }

    /**
     * Get the name.
     * @return name
     */
    public String getName()
    {
      return name;
    }

    /**
     *  Get the maze.
     *  @return the maze
     */
    public Maze getMaze()
    {
      return mazeCanvas.getMaze();
    }

    /**
     * Get a description.
     * @return strung description
     */
    @Override
    public String toString()
    {
      return getName();
    }
  }

  /** The different types of mazes which are supported. */
  private ArrayList<MazeControlPair> mazeTypes;
  /** The mapping of maze type names to maze types. */
  private Map<String,MazeControlPair> mazeTypesMap;
  /** Selection of maze type. */
  private final JComboBox mazeSelect;
  /** The properties for painting. */
  private final MazePaintProperties paintProperties;
  /** The properties for printing. */
  private final MazePrintProperties printProperties;
  /** The storage key for saving a maze. */
  private static final String SAVED_MAZE_KEY = "STORED_MAZE_NAME";
  /** Card layout to switch between mazes. */
  private final CardLayout leftLayout;
  /** Card layout to switch between controllers. */
  private final CardLayout ctrlLayout;
  /** Left panel (maze display). */
  private final JPanel left;
  /** Right panel (controller display). */
  private final JPanel ctrl;

  /**
   *  Constructor.
   *  @param systemAccess access to the system to read settings
   */
  public MultiMazeControlPanel(final SystemAccess systemAccess)
  {
    super(new BorderLayout());

    paintProperties = MazePaintProperties.getDrawMazePaintProperties(systemAccess);
    printProperties = MazePrintProperties.getPrintMazePaintProperties(systemAccess);

    mazeTypes = new ArrayList<MazeControlPair>();
    mazeTypes.add(new MazeControlPair(I18n.getString(RectangularMaze.MAZE_TYPE),
                                      new RectangularMaze(60, 40), systemAccess));
    mazeTypes.add(new MazeControlPair(I18n.getString(TriangularMaze.MAZE_TYPE),
                                      new TriangularMaze(60), systemAccess));
    mazeTypes.add(new MazeControlPair(I18n.getString(HexagonalMaze.MAZE_TYPE),
                                      new HexagonalMaze(60, 40), systemAccess));
    mazeTypes.add(new MazeControlPair(I18n.getString(OctogonalMaze.MAZE_TYPE),
                                      new OctogonalMaze(40, 30), systemAccess));
    mazeTypes.add(new MazeControlPair(I18n.getString(DiamondMaze.MAZE_TYPE),
                                      new DiamondMaze(30, 20), systemAccess));
    mazeTypes.add(new MazeControlPair(I18n.getString(CircularMaze.MAZE_TYPE),
                                      new CircularMaze(2, 16, 30), systemAccess));

    mazeTypesMap = new HashMap<String,MazeControlPair>();

    leftLayout = new CardLayout();
    ctrlLayout = new CardLayout();
    left = new JPanel(leftLayout);
    ctrl = new JPanel(ctrlLayout);

    JPanel right = new JPanel(new BorderLayout());
    right.setBorder(BorderFactory.createCompoundBorder(BorderFactory.createEmptyBorder(5, 5, 5, 5),
                                                       BorderFactory.createEtchedBorder()));

    mazeSelect = new JComboBox();
    JPanel upperRight = new JPanel(new BorderLayout());
    /*
    Box mazeSelectBox = new Box(BoxLayout.X_AXIS);
    mazeSelectBox.add(new JLabel("Maze Type: "));
    mazeSelectBox.add(mazeSelect);
    upperRight.add(mazeSelectBox, BorderLayout.NORTH);
    */
    upperRight.add(mazeSelect, BorderLayout.NORTH);
    upperRight.add(new PropertySetterPanel(paintProperties.getPropertyInformations()), BorderLayout.CENTER);
    right.add(upperRight, BorderLayout.NORTH);
    right.add(ctrl, BorderLayout.CENTER);

    for (MazeControlPair pair: mazeTypes) {
      mazeTypesMap.put(pair.getName(), pair);
      mazeSelect.addItem(pair);
      left.add(pair.getMazeCanvas(), pair.getName());
      ctrl.add(pair.getMazeControlPanel(), pair.getName());
    }

    mazeSelect.addItemListener(new ItemListener() {
      public void itemStateChanged(ItemEvent e)
      {
        MazeControlPair pair = (MazeControlPair)mazeSelect.getSelectedItem();
        leftLayout.show(left, pair.getName());
        ctrlLayout.show(ctrl, pair.getName());
      }
    });

    final JButton resetButton = new JButton(new ResourcedAction(RESOURCE_BUTTON_RESET) {
      public void actionPerformed(ActionEvent e)
      {
        MazeControlPair pair = (MazeControlPair)mazeSelect.getSelectedItem();
        for (PropertyInformation information: pair.getMazeCanvas().getMaze().getPropertyInformations()) {
          information.forget();
        }
      }
      @Override
      protected Object clone() throws CloneNotSupportedException
      {
        throw new CloneNotSupportedException();
      }
    });
    final JButton createButton = new JButton(new ResourcedAction(RESOURCE_BUTTON_CREATE) {
      public void actionPerformed(ActionEvent e)
      {
        try {
          getTopLevelAncestor().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));

          MazeControlPair pair = (MazeControlPair)mazeSelect.getSelectedItem();
          pair.getMazeCanvas().getMaze().setFromSetters();
        } finally {
          getTopLevelAncestor().setCursor(Cursor.getDefaultCursor());
        }
      }
      @Override
      protected Object clone() throws CloneNotSupportedException
      {
        throw new CloneNotSupportedException();
      }
    });

    JPanel buttonPanel = new JPanel(new GridBagLayout());
    GridBagConstraints gbc = new GridBagConstraints();
    gbc.weightx   = 1.0;
    gbc.fill      = GridBagConstraints.HORIZONTAL;
    gbc.gridwidth = GridBagConstraints.RELATIVE;
    buttonPanel.add(resetButton, gbc);
    gbc.gridwidth = GridBagConstraints.REMAINDER;
    buttonPanel.add(createButton, gbc);


    right.add(buttonPanel, BorderLayout.SOUTH);

    add(left, BorderLayout.CENTER);
    add(right, BorderLayout.EAST);
  }

  /**
   *  Store all settings.
   *  @param dataStorage storage to store to
   */
  public void storePersistentData(DataStorage dataStorage)
  {
    paintProperties.storePersistentData(dataStorage);
    printProperties.storePersistentData(dataStorage);
    for (MazeControlPair pair: mazeTypes) {
      pair.getMaze().storePersistentData(dataStorage);
    }
  }

  /**
   * Get the maze paint properties.
   * @return paint properties
   */
  public MazePaintProperties getPaintProperties()
  {
    return paintProperties;
  }

  /**
   * Get the maze print properties.
   * @return print properties
   */
  public MazePrintProperties getPrintProperties()
  {
    return printProperties;
  }

  /**
   * Get the currently active maze.
   * @return the maze
   */
  public Maze getCurrentMaze()
  {
    MazeControlPair pair = (MazeControlPair)mazeSelect.getSelectedItem();
    return pair.getMaze();
  }

  /**
   *  Print the current maze.
   *  @param systemAccess access to system for printing
   */
  void printCurrentMaze(SystemAccess systemAccess)
  {
    MazeControlPair pair = (MazeControlPair)mazeSelect.getSelectedItem();

    systemAccess.print(pair.getMazeCanvas());

  }

  /**
   *  Save the current maze.
   *  @param dataStorage data storage for system access
   */
  void saveCurrentMaze(DataStorage dataStorage)
  {
    MazeControlPair pair = (MazeControlPair)mazeSelect.getSelectedItem();
    dataStorage.setString(SAVED_MAZE_KEY, pair.getName());
    pair.getMaze().storePersistentData(dataStorage);
  }

  /**
   *  Draw the current maze to an image.
   *  @param image      image to draw to
   *  @param properties properties to use for drawing
   */
  void drawCurrentMazeToImage(BufferedImage image, MazePaintPropertiesProvider properties)
  {
    MazeControlPair pair = (MazeControlPair)mazeSelect.getSelectedItem();
    pair.getMazeCanvas().paintToGraphics((Graphics2D)image.getGraphics(),
                                         new Dimension(image.getWidth(),
                                                       image.getHeight()),
                                         properties);
  }

  /**
   *  Create an image from the currently displayed maze.
   *  @param properties maze save image properties
   *  @return maze image
   */
  BufferedImage createImage(MazeSaveImageProperties properties)
  {
    MazeControlPair pair = (MazeControlPair)mazeSelect.getSelectedItem();
    float ratio = pair.getMaze().getPreferredAspectRatio();
    Insets insets = pair.getMaze().getInsets(properties, properties.getExtent()/AbstractBasicMaze.BOX_SIZE);
    int width = properties.getExtent();
    int height = properties.getExtent();
    int reducedWidth  = properties.getExtent() - insets.left - insets.right;
    int reducedHeight = properties.getExtent() - insets.top - insets.bottom;
    float reducedRatio = reducedWidth/(float)reducedHeight;
    if (reducedRatio > ratio) {
      height = (int)Math.ceil(reducedWidth/ratio) + insets.left + insets.right;
    }
    else {
      width = (int)Math.ceil(reducedHeight*ratio) + insets.top + insets.bottom;
    }
    BufferedImage image = new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB);
    drawCurrentMazeToImage(image, properties);
    return image;
  }

  /**
   *  Load a maze.
   *  @param dataStorage data storage for systen access
   *  @return <code>true</code> if a maze was loaded, otherwise <code>false</code>
   */
  boolean loadMaze(DataStorage dataStorage)
  {
    String name = dataStorage.getString(SAVED_MAZE_KEY, null);
    if (name != null) {
      for (MazeControlPair pair: mazeTypes) {
        if (name.equals(pair.getName())) {
          // found
          pair.getMaze().loadPersistentData(dataStorage);
          mazeSelect.setSelectedItem(pair);
          return true;
        }
      }
    }
    return false;
  }
}
