// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: MazeTool.java,v $
// History:	       Revision 1.7  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.6  2006/12/19 18:00:38  rammi
// History:	       Documentation improvements
// History:
// History:	       Revision 1.5  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.4  2005/04/19 22:15:14  rammi
// History:	       Fixed some Thread problems
// History:	
// History:	       Revision 1.3  2005/04/19 20:00:45  rammi
// History:	       no message
// History:	
// History:	       Revision 1.2  2004/10/31 12:45:14  rammi
// History:	       Added improved algorithm
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import de.caff.gimmix.I18n;

import java.util.*;

/**
 *  Create mazes and find ways.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.7 $
 */
public abstract class MazeTool
{
  /** I18n resource for creation message. */
  public static final String MESSAGE_CREATING = "MAZE_TOOL:messageCreating";
  /** Constant to be used for access to the latest maze tool version. */
  public static final int LATEST_VERSION = -1;


  /**
   *  This is the first shot at maze creation.
   *
   *  It's only used directly on old save files any longer.
   */
  private static class MazeTool0 extends MazeTool
  {
    /**
     *  Special version of a double-linked list.
     */
    protected static class DoubleLinkedList
    {
      class Item
      {
        ArrayList<MazeCell> set;
        private Item previous;
        private Item next;

        public Item(ArrayList<MazeCell> set)
        {
          this.set = set;
        }

        public void insertAfter(Item item)
        {
          item.previous = this;
          item.next     = this.next;
          this.next.previous = item;
          this.next     = item;
        }

        public void insertBefore(Item item)
        {
          item.next     = this;
          item.previous = this.previous;
          this.previous.next = item;
          this.previous = item;
        }

        public void remove()
        {
          if (this.previous != null) {
            this.previous.next = this.next;
          }
          if (this.next != null) {
            this.next.previous = this.previous;
          }
        }

        public DoubleLinkedList getList()
        {
          return DoubleLinkedList.this;
        }
      }
      private final Item START_ITEM = new Item(null);
      private final Item END_ITEM   = new Item(null);
      private int size;
      private Item listStart = START_ITEM;
      private Item listEnd   = END_ITEM;
      {
        listStart.next   = listEnd;
        listEnd.previous = listStart;
      }

      public Item add(ArrayList<MazeCell> set)
      {
        Item item = new Item(set);
        listEnd.insertBefore(item);
        ++size;
        return item;
      }

      public void remove(Item item)
      {
        item.remove();
        --size;
      }

      public int size()
      {
        return size;
      }

      public Item get(int pos)
      {
        if (pos < size/2) {
          // run forward
          Item item = listStart;
          while (item.next != listEnd) {
            item = item.next;
            if (pos-- == 0) {
              return item;
            }
          }
        }
        else if (pos < size) {
          // run backward
          pos = size-pos-1;
          Item item = listEnd;
          while (item.previous != listStart) {
            item = item.previous;
            if (pos-- == 0) {
              return item;
            }
          }
        }
        return null;
      }
    }

    /**
     *  Create a maze with the given random number seed.
     *  @param maze            maze geometry info
     *  @param progressShower  progress display (may be <code>null</code>)
     *  @param seed            random number seed
     *  @return <code>true</code>: maze was created, <code>false</code>: user canceled the creation
     */
    public boolean createMaze(Maze maze, ProgressShower progressShower, long seed)
    {
      maze.reset();
      final Random random = new Random(seed);

      final MazeCell[] cells = maze.getCells();

      DoubleLinkedList sets = new DoubleLinkedList();

      for (MazeCell cell: cells) {
        ArrayList<MazeCell> set = new ArrayList<MazeCell>(1);
        set.add(cell);
        cell.setSet(sets.add(set));
      }

      int size = sets.size();
      startProgress(progressShower, size);

      return joinSets(sets, size, progressShower, random);
    }

    /**
     *  Join the sets by picking a cell of a set and connecting it to another set.
     *  Randomly picking a set is slow for a large list of sets.
     *  @param sets   list of sets
     *  @param size   original size
     *  @param progressShower progress shower or <code>null</code>
     *  @param random random number generator
     *  @return <code>true</code>: maze was created, <code>false</code>: user canceled the creation
     */
    protected static boolean joinSets(DoubleLinkedList sets,
                                       int size,
                                       ProgressShower progressShower,
                                       final Random random)
    {
outer:
      while (sets.size() > 1) {
        final int setSize = sets.size();
        if (setProgress(progressShower, size-setSize)) {
          // user-requested abort
          return false;
        }
        final int pos = random.nextInt(setSize);
        //int pos = 0;
        // connect the set with another
        DoubleLinkedList.Item item = sets.get(pos);
        ArrayList<MazeCell> set = item.set;

        int cellIndex = random.nextInt(set.size());
        int cellTries = set.size();

        while (cellTries-- > 0) {
          MazeCell cell = set.get(cellIndex);
          MazeCell[] neighbours = cell.getNeighbours();

          int neighbourIndex = random.nextInt(neighbours.length);
          int neighbourTries = neighbours.length;
          while (neighbourTries-- > 0) {
            if (neighbours[neighbourIndex].getSet() != item) {
              join(sets, cell, neighbours[neighbourIndex]);
              continue outer;
            }
            neighbourIndex = (neighbourIndex+1) % neighbours.length;
          }
          cellIndex = (cellIndex+1)%set.size();
        }
        // should never get here otherwise the geometry defines disjunct parts
        throw new RuntimeException("Disjunct geometry!");
      }
      return true;
    }

    /**
     *  Join the sets of 2 cells.
     *  @param sets  all sets (will be reduced)
     *  @param cell1 first cell
     *  @param cell2 second cell
     */
    protected static void join(DoubleLinkedList sets, MazeCell cell1, MazeCell cell2)
    {
      DoubleLinkedList.Item item1 = (DoubleLinkedList.Item)cell1.getSet();
      DoubleLinkedList.Item item2 = (DoubleLinkedList.Item)cell2.getSet();
      ArrayList<MazeCell> set1 = item1.set;
      ArrayList<MazeCell> set2 = item2.set;
      sets.remove(item2);
      for (Iterator<MazeCell> it = set2.iterator();  it.hasNext();  ) {
        it.next().setSet(item1);
      }
      set1.addAll(set2);

      cell1.connectTo(cell2);
      cell2.connectTo(cell1);
    }
  }

  /**
   *  This is similar to the algorithm in MazeTool0 but speeds up the
   *  first part by accessing the maze cells directly. This avoids skipping
   *  randomly through the double linked-list as long as it is very long.
   *  Speed up is about 2 to 3 times for 40000 cells but of course the resulting maze
   *  is different for the same seed.
   */
  private static class MazeTool1 extends MazeTool0
  {
    /**
     * Create a maze with the given random number seed.
     *
     * @param maze           maze geometry info
     * @param progressShower progress display (may be <code>null</code>)
     * @param seed           random number seed
     * @return <code>true</code>: maze was created, <code>false</code>: user canceled the creation
     */
    public boolean createMaze(Maze maze, ProgressShower progressShower, long seed)
    {
      // System.out.println("Reset:                 "+System.currentTimeMillis());
      maze.reset();
      final Random random = new Random(seed);

      final MazeCell[] cells = maze.getCells();

      DoubleLinkedList sets = new DoubleLinkedList();

      for (MazeCell cell: cells) {
        ArrayList<MazeCell> set = new ArrayList<MazeCell>(1);
        set.add(cell);
        cell.setSet(sets.add(set));
      }

      int size = sets.size();
      // System.out.println("Ready:                 "+System.currentTimeMillis());
      startProgress(progressShower, size);

      final int border = (int)(cells.length/Math.exp(1));

      // 1st PART:
      // Pick any cell by random. then try the neighbors.
      // This is faster because it's quite probable that
      // a randomly picked cell is not already connected to it's neighbors.
      while (sets.size() > border) {
        if (setProgress(progressShower, size-sets.size())) {
          // user-requested abort
          return false;
        }
        MazeCell cell = cells[random.nextInt(cells.length)];
        // connect the set with another
        Object item = cell.getSet();
        MazeCell[] neighbours = cell.getNeighbours();

        int neighbourIndex = random.nextInt(neighbours.length);
        int neighbourTries = neighbours.length;
        while (neighbourTries-- > 0) {
          if (neighbours[neighbourIndex].getSet() != item) {
            join(sets, cell, neighbours[neighbourIndex]);
            break;
          }
          neighbourIndex = (neighbourIndex+1) % neighbours.length;
        }
      }
      // 2nd PART:
      // Pick any set, than use a cell of that set.
      // Randomly picking a set is slow, but there are a reduced number of sets
      // now. There must be at least 1 cell in the set which has a possible unconnected
      // neighbor or the geometry is invalid. This is the same code as in MazeTool0.
      return joinSets(sets, size, progressShower, random);
    }
  }

  /**
   *  The different maze creation algorithms.
   */
  private static final MazeTool[] mazeTools = {
    new MazeTool0(),
    new MazeTool1()
  };


  /**
   *  Get a special version of a maze tool.
   *  This is thought for backward compatibility of maze files.
   *  @param version  tool version
   *  @return matching tool or <code>null</code> if there is no such version
   */
  public static MazeTool getMazeTool(int version)
  {
    if (version == LATEST_VERSION) {
      return getMazeTool();
    }
    if (version >= 0  &&  version < mazeTools.length) {
      return mazeTools[version];
    }
    return null;
  }

  /**
   *  Get the current (latest) version of the maze tool.
   *  @return current maze tool
   */
  public static MazeTool getMazeTool()
  {
    return mazeTools[mazeTools.length-1];
  }

  /**
   *  Get the current version of the maze tool.
   *  @return current version id
   */
  public static int getCurrentMazeAlgorithmVersion()
  {
    return mazeTools.length-1;
  }

  /**
   *  Create a maze with the current time as random number seed.
   *  @param maze            maze geometry info
   *  @param progressShower  progress display (may be <code>null</code>)
   *  @return <code>true</code>: maze was created, <code>false</code>: user canceled the creation
   */
  public boolean createMaze(Maze maze, ProgressShower progressShower)
  {
    return createMaze(maze, progressShower, System.currentTimeMillis());
  }

  /**
   *  Create a maze with the given random number seed.
   *  @param maze            maze geometry info
   *  @param progressShower  progress display (may be <code>null</code>)
   *  @param seed            random number seed
   *  @return <code>true</code>: maze was created, <code>false</code>: user canceled the creation
   */
  public abstract boolean createMaze(Maze maze, ProgressShower progressShower, long seed);

  /**
   *  Find a way from one cell of a maze to another.
   *  @param from cell where the way starts
   *  @param to   cell where the way ends
   *  @return the resulting way (including from and to)
   *          or <code>null</code> if there is no way between the cells
   */
  public static Collection<MazeCell> solveMaze(MazeCell from, MazeCell to)
  {
    if (from != null  &&  to != null) {
      Maze maze = from.getMaze();
      if (maze != to.getMaze()) {
        return null;
      }
      return findWay(null, from, to);
    }
    else {
      return null;
    }
  }

  /**
   *  Find a way from cell <code>from</code> to cell <code>to</code> when coming
   *  from cell <code>before</code>.
   *  @param before  cell from which we came
   *  @param from    cell where we are
   *  @param to      destinatination cell
   *  @return  a list of cells from <code>from</code> to <code>to</code>
   */
  private static List<MazeCell> findWay(MazeCell before, MazeCell from, MazeCell to)
  {
    if (from == to) {
      LinkedList<MazeCell> list = new LinkedList<MazeCell>();
      list.add(from);
      return list;
    }
    final MazeCell[] neighbours = from.getConnectedNeighbours();
    for (int n = neighbours.length-1;  n >= 0;  --n) {
      final MazeCell neighbour = neighbours[n];
      if (neighbour != before) {
        final List<MazeCell> way = findWay(from, neighbour, to);
        if (way != null) {
          way.add(0, from);
          return way;
        }
      }
    }
    return null;
  }

  /**
   *  Start the progress display if a progress shower is given.
   *  @param progressShower  progress shower or <code>null</code>
   *  @param size            progress complete size
   */
  protected static void startProgress(ProgressShower progressShower, int size)
  {
    if (progressShower != null) {
      progressShower.start(I18n.getString(MESSAGE_CREATING), size);
    }
  }

  /**
   *  Set the progress display if a progress shower is given.
   *  @param progressShower   progress shower or <code>null</code>
   *  @param value            current progress value
   *  @return <code>true</code> if the user aborted the creation, otherwise <code>false</code>
   */
  protected static boolean setProgress(ProgressShower progressShower, int value)
  {
    if (progressShower != null) {
      if (progressShower.setProgress(value)) {
        // user-requested abort
        return true;
      }
    }
    return false;
  }
}
