// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: MazeSaveImageProperties.java,v $
// History:	       Revision 1.2  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.1  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.2  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.1.1.1  2004/10/25 14:47:55  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import java.awt.*;
import java.util.ArrayList;
import java.util.Collection;

/**
 *  Properties describing how a maze is painted.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @version $Revision: 1.2 $
 */
public class MazeSaveImageProperties
        extends MazePaintProperties
        implements MazeSaveImagePropertiesProvider
{
  /** Property key of extent property describing the extent (longer side of the image). */
  public static final String PROPERTY_EXTENT     = "MAZE_SAVE_IMG:extent";

  /** The default width and height. */
  public static final int DEFAULT_EXTENT = 1024;

  /**
   *  Get save image properties.
   *  @param systemAccess  system access used to overwrite default settings
   *  @return save image properties
   */
  public static MazeSaveImageProperties getMazeSaveImageProperties(DataStorage systemAccess)
  {
    MazeSaveImageProperties properties = new MazeSaveImageProperties("saveimg:");

    properties.setShowingSolution(false);
    properties.setShowingCellBorders(false);

    properties.setOuterBorderPaint(Color.black);
    properties.setInnerBorderPaint(Color.black);
    properties.setBackgroundPaint(null);
    properties.setWayStartPaint(Color.magenta);
    properties.setWayEndPaint(Color.green);
    properties.setBackgroundPaint(Color.white);
    properties.overwriteFromPersistentData(systemAccess);

    return properties;
  }

  /** The image extent. */
  private int extent = DEFAULT_EXTENT;

  /**
   *  Create a maze print properties element with the given key.
   *  @param myKey key of the properties
   */
  public MazeSaveImageProperties(String myKey)
  {
    super(myKey);
  }

  /**
   *  Get the image extent.
   *  The extent describes the longer side of the image.
   *  @return image extent
   */
  public int getExtent()
  {
    return extent;
  }

  /**
   *  Set the image extent.
   *  The extent describes the longer side of the image.
   *  @param extent image extent
   */
  public void setExtent(int extent)
  {
    if (extent > 0) {
      if (this.extent != extent) {
        int oldExtent = this.extent;
        this.extent = extent;
        firePropertyChange(PROPERTY_EXTENT, oldExtent, extent);
      }
    }
  }

  /**
   *  Get the property informations.
   *  @return collection of property informations
   */
  @Override public Collection<PropertyInformation> getPropertyInformations()
  {
    Collection<PropertyInformation> setters = new ArrayList<PropertyInformation>();

    setters.add(new IntegerPropertyInformation(PROPERTY_EXTENT, 16, Integer.MAX_VALUE) {
      /**
       * Set the value of the owner of the property.
       *
       * @param value new value
       */
      protected void setOwnerValue(int value)
      {
        setExtent(value);
      }

      /**
       * Get the value from the owner of the property.
       *
       * @return the current value
       */
      protected int getOwnerValue()
      {
        return getExtent();
      }
    });

    setters.addAll(super.getPropertyInformations());

    return setters;
  }

  /**
   *  Overwrite the properties from the given data storage.
   *  @param dataStorage data storage to use for access to stored settings
   */
  @Override void overwriteFromPersistentData(DataStorage dataStorage)
  {
    if (dataStorage == null) {
      return;
    }
    super.overwriteFromPersistentData(dataStorage);
    setExtent(dataStorage.getInt(myKey+PROPERTY_EXTENT, getExtent()));
  }

  /**
   *  Store the settings in a data storage.
   *  @param dataStorage storage to store to
   */
  @Override void storePersistentData(DataStorage dataStorage)
  {
    super.storePersistentData(dataStorage);
    dataStorage.setInt(myKey+PROPERTY_EXTENT, getExtent());
  }
}