// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: MazePropertyOwner.java,v $
// History:	       Revision 1.5  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.4  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.3  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.2  2005/04/19 22:15:14  rammi
// History:	       Fixed some Thread problems
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import de.caff.gimmix.Worker;

import javax.swing.*;
import javax.swing.event.ChangeEvent;
import javax.swing.event.ChangeListener;
import javax.swing.plaf.basic.BasicComboBoxRenderer;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.image.BufferedImage;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;

/**
 *  Something which knows about maze properties.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.5 $
 */
public abstract class MazePropertyOwner
{
  /** Collection of listeners which are interested that a property has changed. */
  private Collection<PropertyChangeListener> propertyChangeListeners = new ArrayList<PropertyChangeListener>();

  /**
   *  Inform property change listeners in another thread.
   */
  private static class FirePropertyRunnable
          implements Runnable
  {
    /** Listeners to inform. */
    private final Collection<PropertyChangeListener> listeners;
    /** Event to send. */
    private final PropertyChangeEvent event;

    /**
     * Constructor.
     * @param listeners   listeners to inform
     * @param event       event to send
     */
    public FirePropertyRunnable(Collection<PropertyChangeListener> listeners, PropertyChangeEvent event)
    {
      this.listeners = new ArrayList<PropertyChangeListener>(listeners);
      this.event = event;
    }

    /**
     *  Inform the listeners.
     */
    public void run()
    {
      for (PropertyChangeListener listener: listeners) {
        listener.propertyChange(event);
      }
    }
  }

  /**
   *  Add a property change listener.
   *  @param listener listener to add
   */
  public void addPropertyChangeListener(PropertyChangeListener listener)
  {
    synchronized(propertyChangeListeners) {
      propertyChangeListeners.add(listener);
    }
  }

  /**
   *  Remove a property change listener.
   *  @param listener listener to remove
   */
  public void removePropertyChangeListener(PropertyChangeListener listener)
  {
    synchronized(propertyChangeListeners) {
      propertyChangeListeners.remove(listener);
    }
  }

  /**
   *  Fire property changes in the event dispatch thread.
   *  @param propertyName name of change property
   *  @param oldValue     previous value
   *  @param newValue     new value
   */
  protected void firePropertyChange(String propertyName, Object oldValue, Object newValue)
  {
    FirePropertyRunnable runnable;
    synchronized(propertyChangeListeners) {
      runnable = new FirePropertyRunnable(propertyChangeListeners,
                                          new PropertyChangeEvent(this, propertyName, oldValue, newValue));
    }
    Worker.invokeInEventDispatchThread(runnable);
  }

  /**
   *  Get the property informations.
   *  @return collection of property informations
   */
  public abstract Collection<PropertyInformation> getPropertyInformations();

  /**
   *  A boolean property and its setters.
   */
  protected abstract static class BooleanPropertyInformation
        extends AbstractPropertyInformation
  {
    /** GUI controller. */
    private JRadioButton radioButton;

    /**
     *  Constructor.
     *  @param name property id
     */
    public BooleanPropertyInformation(String name)
    {
      super(name);
      radioButton = new JRadioButton();
      radioButton.addActionListener(new ActionListener()
      {
        public void actionPerformed(ActionEvent e)
        {
          setOwnerValue();
        }
      });
    }

    /**
     *  A component to set the property.
     *  @return setter component
     */
    public JComponent getSetterComponent()
    {
      radioButton.setSelected(getOwnerValue());
      return radioButton;
    }

    /**
     *  Set the value from the GUI component.
     */
    protected void setOwnerValue()
    {
      setOwnerValue(radioButton.isSelected());
    }

    /**
     *  Get the current value of property.
     *  @return property value
     */
    public Object getPropertyValue()
    {
      return Boolean.valueOf(getOwnerValue());
    }

    /**
     *  Set the value of the owner of the property.
     *  @param state new value
     */
    protected abstract void setOwnerValue(boolean state);

    /**
     *  Get the value from the owner of the property.
     *  @return value
     */
    protected abstract boolean getOwnerValue();
  }

  /**
   *  A color property and its setters.
   */
  protected abstract static class PaintPropertyInformation
        extends AbstractPropertyInformation
  {
    /** Standard colors. */
    private static Color[] STANDARD_COLORS = {
      Color.black,
      Color.white,
      Color.red,
      Color.green,
      Color.blue,
      Color.yellow,
      Color.magenta,
      Color.pink,
      Color.orange,
      Color.cyan,
      Color.lightGray,
      Color.gray,
      Color.darkGray
    };
    /** Color chooser. */
    private JComboBox colorCombo;

    /**
     *  Constructor.
     *  @param name      resource name
     *  @param allowNone if <code>true</code> it's allowed to give no color at all
     */
    public PaintPropertyInformation(String name, boolean allowNone)
    {
      super(name);
      final int width = 40;
      final int height = 10;

      colorCombo = new JComboBox();
      for (int c = 0;  c < STANDARD_COLORS.length;  ++c) {
        Image colorImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
        Graphics2D g = (Graphics2D)colorImage.getGraphics();
        g.setPaint(STANDARD_COLORS[c]);
        g.fillRect(0, 0, width, height);
        colorCombo.addItem(new ImageIcon(colorImage));
      }
      final BufferedImage selectImage;
      selectImage = new BufferedImage(width, height, BufferedImage.TYPE_INT_RGB);
      Graphics2D g = (Graphics2D)selectImage.getGraphics();
      g.setPaint(getOwnerValue());
      g.fillRect(0, 0, width, height);
      final ImageIcon selectIcon = new ImageIcon(selectImage);
      colorCombo.addItem(selectIcon);
      if (allowNone) {
        colorCombo.addItem(new ImageIcon(new BufferedImage(width, height, BufferedImage.TYPE_INT_ARGB)));
      }
      colorCombo.setRenderer(new BasicComboBoxRenderer() {
        @Override public Component getListCellRendererComponent(JList list, Object value, int index, boolean isSelected,
                                                      boolean cellHasFocus)
        {
          if (index < STANDARD_COLORS.length) {
            setText(null);
            return super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
          }
          else {
            if (index == STANDARD_COLORS.length) {
              setText("Select...");
            }
            else {
              setText("None");
              setIcon(null);
            }
            return super.getListCellRendererComponent(list, value, index, isSelected, cellHasFocus);
          }
        }

        @Override public Dimension getPreferredSize()
        {
          Dimension prefSize = super.getPreferredSize();
          prefSize.width += 50;  // todo: remove this, it is ugly!
          return prefSize;
        }
      });

      colorCombo.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e)
        {
          final int selectedIndex = colorCombo.getSelectedIndex();
          if (selectedIndex < STANDARD_COLORS.length) {
            // standard color
            setOwnerValue(STANDARD_COLORS[selectedIndex]);
          }
          else {
            if (selectedIndex == STANDARD_COLORS.length) {
              Color color = JColorChooser.showDialog(colorCombo.getRootPane(),
                                                     "Select Color",
                                                     (Color)getOwnerValue());
              if (color != null) {
                Graphics2D g = (Graphics2D)selectImage.getGraphics();
                g.setPaint(color);
                g.fillRect(0, 0, width, height);
                colorCombo.repaint();
                setOwnerValue(color);
              }
            }
            else {
              setOwnerValue(null);
            }
          }
        }
      });

      Color currentColor = (Color)getOwnerValue();
      for (int c = 0;  c < STANDARD_COLORS.length;  ++c) {
        if (STANDARD_COLORS[c].equals(currentColor)) {
          colorCombo.setSelectedIndex(c);
          return;
        }
      }
      if (currentColor == null) {
        colorCombo.setSelectedIndex(STANDARD_COLORS.length+1);
      }
      else {
        colorCombo.setSelectedIndex(STANDARD_COLORS.length);
      }
    }

    /**
     *  A component to set the property.
     *  @return setter component
     */
    public JComponent getSetterComponent()
    {
      return colorCombo;
    }

    /**
     *  Get the current value of property.
     *  @return property value
     */
    public Object getPropertyValue()
    {
      return getOwnerValue();
    }

    /**
     *  Set the value of the owner of the property.
     *  @param paint new value
     */
    protected abstract void setOwnerValue(Paint paint);

    /**
     *  Get the value from the owner of the property.
     *  @return value
     */
    protected abstract Paint getOwnerValue();
  }


  /**
   *  Display of a purely informational property.
   */
  protected abstract static class InfoPropertyDisplay
          extends AbstractPropertyInformation
          implements PropertyChangeListener
  {
    /** Property display. */
    private final JLabel displayLabel;
    /** Name of the property. */
    private final String propertyName;

    /**
     * Constructor.
     * @param name         id
     * @param propertyName name of property
     * @param owner        property owner
     */
    protected InfoPropertyDisplay(String name,
                                  String propertyName,
                                  MazePropertyOwner owner)
    {
      super(name);
      displayLabel = new JLabel(getValue().toString());
      this.propertyName = propertyName;
      owner.addPropertyChangeListener(this);
    }

    /**
     *  A component to set the property.
     *  @return setter component
     */
    public JComponent getSetterComponent()
    {
      return displayLabel;
    }

    /**
     *  Called when the property has changed.
     *  @param evt property change event
     */
    public void propertyChange(PropertyChangeEvent evt)
    {
      if (evt.getPropertyName().equals(propertyName)) {
        displayLabel.setText(getValue().toString());
      }
    }

    /**
     *  Get the property value from the owner.
     *  @return property value
     */
    protected abstract Object getValue();

    /**
     *  Get the current value of property.
     *  @return property value
     */
    public Object getPropertyValue()
    {
      return getValue();
    }

    /**
     *  Is the setter component just for display?
     *  @return always <code>true</code>
     */
    public boolean isInformational()
    {
      return true;
    }
  }

  /**
   *  An enumerated property and its setters.
   */
  protected abstract static class EnumPropertyInformation
          extends AbstractPropertyInformation
  {
    /** GUI component. */
    private final JComboBox box;

    /**
     *  Constructor.
     *  @param name   id
     *  @param values possible values
     */
    protected EnumPropertyInformation(String name, Enum[] values)
    {
      super(name);
      box = new JComboBox(values);
      box.setSelectedItem(getOwnerValue());
      box.addActionListener(new ActionListener() {
        public void actionPerformed(ActionEvent e)
        {
          setOwnerValue((Enum)box.getSelectedItem());
        }
      });
    }

    /**
     *  A component to set the property.
     *  @return setter component
     */
    public JComponent getSetterComponent()
    {
      return box;
    }

    /**
     *  Set the value of the owner of the property.
     *  @param value new value
     */
    protected abstract void setOwnerValue(Enum value);

    /**
     *  Get the value from the owner of the property.
     *  @return the current value
     */
    protected abstract Enum getOwnerValue();

    /**
     *  Get the current value of property.
     *  @return property value
     */
    public Object getPropertyValue()
    {
      return getOwnerValue();
    }
  }
  /**
   *  An integer property and its setters.
   */
  protected abstract static class IntegerPropertyInformation
          extends AbstractPropertyInformation
  {
    /** GUI component. */
    private final JSpinner spinner;

    /**
     *  Constructor.
     *  @param name   id
     */
    protected IntegerPropertyInformation(String name)
    {
      this(name, Integer.MIN_VALUE, Integer.MAX_VALUE);
    }
    /**
     *  Constructor.
     *  @param name   id
     *  @param minValue minimal value
     *  @param maxValue maximal value
     */
    protected IntegerPropertyInformation(String name, int minValue, int maxValue)
    {
      super(name);
      spinner = new JSpinner(new SpinnerNumberModel(getOwnerValue(),
                                                    minValue, maxValue, 1));
      spinner.addChangeListener(new ChangeListener()
      {
        public void stateChanged(ChangeEvent e)
        {
          setOwnerValue(((Number)spinner.getValue()).intValue());
        }
      });
    }

    /**
     *  A component to set the property.
     *  @return setter component
     */
    public JComponent getSetterComponent()
    {
      return spinner;
    }

    /**
     *  Set the value of the owner of the property.
     *  @param value new value
     */
    protected abstract void setOwnerValue(int value);

    /**
     *  Get the value from the owner of the property.
     *  @return the current value
     */
    protected abstract int getOwnerValue();

    /**
     *  Get the current value of property.
     *  @return property value
     */
    public Object getPropertyValue()
    {
      return getOwnerValue();
    }
  }
}
