// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: MazePrintProperties.java,v $
// History:	       Revision 1.3  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.2  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.1.1.1  2004/10/25 14:47:55  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import java.awt.*;
import java.util.Collection;

/**
 *  Properties describing how a maze is painted.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @version $Revision: 1.3 $
 */
public class MazePrintProperties
        extends MazePaintProperties
        implements MazePrintPropertiesProvider
{
  /** Property key of BlowupFactor property describing on how many pages a maze should be drawn. */
  public static final String PROPERTY_BLOW_UP_FACTOR = "MAZE_PRINT:blowUpFactor";
  /** Property key of boolean property describing whether information about the maze should be printed. */
  public static final String PROPERTY_PRINT_INFO     = "MAZE_PRINT:printInfo";

  /**
   *  Get print properties.
   *  @param systemAccess  system access used to overwrite default settings
   *  @return print properties
   */
  public static MazePrintProperties getPrintMazePaintProperties(DataStorage systemAccess)
  {
    MazePrintProperties printProperties = new MazePrintProperties("print:");

    printProperties.setShowingSolution(false);
    printProperties.setShowingCellBorders(false);

    printProperties.setOuterBorderPaint(Color.black);
    printProperties.setInnerBorderPaint(Color.black);
    printProperties.setBackgroundPaint(null);
    printProperties.setWayStartPaint(Color.lightGray);
    printProperties.setWayEndPaint(Color.lightGray);
    printProperties.overwriteFromPersistentData(systemAccess);

    return printProperties;
  }

  /**
   *  Map a number of pages to a blowup factor.
   *  @param factor number of pages
   *  @return matching BlowUpFactor or <code>null</code>
   */
  public static MazePrintPropertiesProvider.BlowUpFactor getBlowUpFactor(int factor)
  {
    for (MazePrintPropertiesProvider.BlowUpFactor f: MazePrintPropertiesProvider.BlowUpFactor.values()) {
      if (f.getPages() == factor) {
        return f;
      }
    }
    return null;
  }



  /** The blow up factor. It defines on how many pages the maze will be printed. */
  private BlowUpFactor blowUpFactor = BlowUpFactor.BLOW_UP_SINGLE;
  /** Whether information about the maze should be printed, too. */
  private boolean printInfo;

  /**
   *  Create a maze print properties element with the given key.
   *  @param myKey key of the properties
   */
  public MazePrintProperties(String myKey)
  {
    super(myKey);
  }

  /**
   *  Get the blow-up factor. It defines on how many pages the
   *  maze is printed.
   *  @return blow-up factor
   */
  public BlowUpFactor getBlowUpFactor()
  {
    return blowUpFactor;
  }

  /**
   *  Set the blow-up factor. It defines on how many pages the
   *  maze is printed.
   *  @param blowUpFactor  the blow-up factor
   */ 
  public void setBlowUpFactor(BlowUpFactor blowUpFactor)
  {
    if (blowUpFactor != null) {
      if (this.blowUpFactor != blowUpFactor) {
        BlowUpFactor oldFactor = this.blowUpFactor;
        this.blowUpFactor = blowUpFactor;
        firePropertyChange(PROPERTY_BLOW_UP_FACTOR, oldFactor, blowUpFactor);
      }
    }
  }

  /**
   *  Should the maze information be printed?
   *  @return the answer
   */
  public boolean isPrintInfo()
  {
    return printInfo;
  }

  /**
   *  Set whether the maze information should be printed.
   *  @param printInfo <code>true</code>: print info, <code>false</code>: don't print info
   */
  public void setPrintInfo(boolean printInfo)
  {
    if (this.printInfo != printInfo) {
      this.printInfo = printInfo;
      firePropertyChange(PROPERTY_PRINT_INFO, !printInfo, printInfo);
    }
  }

  /**
   *  Get the property informations.
   *  @return collection of property informations
   */
  @Override public Collection<PropertyInformation> getPropertyInformations()
  {
    Collection<PropertyInformation> setters = super.getPropertyInformations();

    setters.add(new BooleanPropertyInformation(PROPERTY_PRINT_INFO) {
      protected void setOwnerValue(boolean state)
      {
        setPrintInfo(state);
      }

      protected boolean getOwnerValue()
      {
        return isPrintInfo();
      }
    });

    setters.add(new EnumPropertyInformation(PROPERTY_BLOW_UP_FACTOR,
                                       BlowUpFactor.values()) {
      protected void setOwnerValue(Enum value)
      {
        setBlowUpFactor((BlowUpFactor)value);
      }

      protected Enum getOwnerValue()
      {
        return getBlowUpFactor();
      }
    });

    return setters;
  }

  /**
   *  Overwrite the properties from the given data storage.
   *  @param dataStorage data storage to use for access to stored settings
   */
  @Override void overwriteFromPersistentData(DataStorage dataStorage)
  {
    if (dataStorage == null) {
      return;
    }
    super.overwriteFromPersistentData(dataStorage);
    setPrintInfo(dataStorage.getBoolean(myKey+PROPERTY_PRINT_INFO, isPrintInfo()));
    setBlowUpFactor(getBlowUpFactor(dataStorage.getInt(myKey+PROPERTY_BLOW_UP_FACTOR,
                                                       getBlowUpFactor().getPages())));
  }

  /**
   *  Store the settings in a data storage.
   *  @param dataStorage storage to store to
   */
  @Override void storePersistentData(DataStorage dataStorage)
  {
    super.storePersistentData(dataStorage);
    dataStorage.setBoolean(myKey+PROPERTY_PRINT_INFO, isPrintInfo());
    dataStorage.setInt(myKey+PROPERTY_BLOW_UP_FACTOR, getBlowUpFactor().getPages());
  }
}
