// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: MazePaintProperties.java,v $
// History:	       Revision 1.4  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.3  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.2  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import java.awt.*;
import java.util.ArrayList;
import java.util.Collection;

/**
 *  Properties describing how a maze is painted.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.4 $
 */
public class MazePaintProperties
        extends MazePropertyOwner
        implements MazePaintPropertiesProvider
{
  /** Property key of boolean property describing whether the solution is shown. */
  public static final String PROPERTY_SHOWING_SOLUTION     = "MAZE:showingSolution";
  /** Property key of boolean property describing whether cell borders are shown. */
  public static final String PROPERTY_SHOWING_CELL_BORDERS = "MAZE:showingCellBorders";
  /** Property key of Paint property of outer border. */
  public static final String PROPERTY_OUTER_BORDER_PAINT   = "MAZE:outerBorderPaint";
  /** Property key of Paint property of inner borders. */
  public static final String PROPERTY_INNER_BORDER_PAINT   = "MAZE:innerBorderPaint";
  /** Property key of Paint property of cell borders. */
  public static final String PROPERTY_CELL_BORDER_PAINT    = "MAZE:cellBorderPaint";
  /** Property key of Paint property of solution cells. */
  public static final String PROPERTY_SOLUTION_PAINT       = "MAZE:solutionPaint";
  /** Property key of Paint property of the way start point. */
  public static final String PROPERTY_WAY_START_PAINT      = "MAZE:wayStartPaint";
  /** Property key of Paint property of the way end point. */
  public static final String PROPERTY_WAY_END_PAINT        = "MAZE:wayEndPaint";
  /** Property key of Paint property of the background. */
  public static final String PROPERTY_BACKGROUND_PAINT     = "MAZE:backgroundPaint";

  /** Key of this property. */
  protected final String  myKey;

  /** Display solution? */
  private boolean showingSolution;
  /** Display cell borders? */
  private boolean showingCellBorders;
  /** Outer border paint. */
  private Paint outerBorderPaint = Color.black;
  /** Inner border paint. */
  private Paint innerBorderPaint = Color.blue;
  /** Cell border paint (if showing cell borders). */
  private Paint cellBorderPaint  = new Color(1f, 1f, 1f, 0.5f);
  /** Solution paint (if showing solution). */
  private Paint solutionPaint    = Color.yellow;
  /** Background paint. */
  private Paint backgroundPaint  = null;
  /** Paint of start cell. */
  private Paint wayStartPaint    = Color.green;
  /** Paint of end cell. */
  private Paint wayEndPaint      = Color.orange;

  /**
   *  Get paint properties.
   *  @param systemAccess  system access used to overwrite default settings
   *  @return paint properties
   */
  public static MazePaintProperties getDrawMazePaintProperties(DataStorage systemAccess)
  {
    MazePaintProperties  paintProperties = new MazePaintProperties("paint:");
    paintProperties.overwriteFromPersistentData(systemAccess);
    return paintProperties;
  }

  /**
   *  Get paint properties for saving the maze as an image.
   *  @param systemAccess  system access used to overwrite default settings
   *  @return paint properties
   */
  public static MazePaintProperties getSaveMazePaintProperties(DataStorage systemAccess)
  {
    MazePaintProperties  paintProperties = new MazePaintProperties("image:");
    paintProperties.overwriteFromPersistentData(systemAccess);
    return paintProperties;
  }

  /**
   *  Constructor.
   *  @param myKey resource key
   */
  protected MazePaintProperties(String myKey)
  {
    this.myKey = myKey;
  }

  /**
   *  Set whether the solution is displayed.
   *  @param show show the solution?
   */
  public void setShowingSolution(boolean show)
  {
    if (show != showingSolution) {
      showingSolution = show;
      firePropertyChange(PROPERTY_SHOWING_SOLUTION, !show, show);
    }
  }

  /**
   * Is the solution displayed?
   * @return the answer
   */
  public boolean isShowingSolution()
  {
    return showingSolution;
  }

  /**
   *  Are cell borders displayed?
   *  Cell borders are drawn between connected cells.
   *  @return the answer
   */
  public boolean isShowingCellBorders()
  {
    return showingCellBorders;
  }

  /**
   * Set whether cell borders are displayed.
   * Cell borders are drawn between connected cells.
   * @param showingCellBorders display cell borders?
   */
  public void setShowingCellBorders(boolean showingCellBorders)
  {
    if (showingCellBorders != this.showingCellBorders) {
      this.showingCellBorders = showingCellBorders;
      firePropertyChange(PROPERTY_SHOWING_CELL_BORDERS,
                         !showingCellBorders,
                         showingCellBorders);
    }
  }

  /**
   *  Get the paint of the outer border.
   *  @return outer border paint
   */
  public Paint getOuterWallPaint()
  {
    return outerBorderPaint;
  }

  /**
   *  Set the paint of the outer border.
   *  @param outerBorderPaint new outer border paint
   */
  public void setOuterBorderPaint(Paint outerBorderPaint)
  {
    if (!outerBorderPaint.equals(this.outerBorderPaint)) {
      Paint oldPaint = this.outerBorderPaint;
      this.outerBorderPaint = outerBorderPaint;
      firePropertyChange(PROPERTY_OUTER_BORDER_PAINT, oldPaint, outerBorderPaint);
    }
  }

  /**
   *  Get the paint of the inner borders.
   *  Inner borders are drawn between unconnected cells.
   *  @return inner border paint
   */
  public Paint getInnerWallsPaint()
  {
    return innerBorderPaint;
  }

  /**
   *  Set the paint of the inner borders.
   *  Inner borders are drawn between unconnected cells.
   *  @param innerBorderPaint inner border paint
   */
  public void setInnerBorderPaint(Paint innerBorderPaint)
  {
    if (!innerBorderPaint.equals(this.innerBorderPaint)) {
      Paint oldPaint = this.innerBorderPaint;
      this.innerBorderPaint = innerBorderPaint;
      firePropertyChange(PROPERTY_INNER_BORDER_PAINT, oldPaint, innerBorderPaint);
    }
  }

  /**
   * Get the paint of cell borders.
   * Cell borders are drawn between connected cells if they are drawn at all.
   * @return cell border paint
   * @see #isShowingCellBorders()
   */
  public Paint getCellBorderPaint()
  {
    return cellBorderPaint;
  }

  /**
   * Set the paint of cell borders.
   * Cell borders are drawn between connected cells if they are drawn at all.
   * @param cellBorderPaint cell border paint
   * @see #isShowingCellBorders()
   */
  public void setCellBorderPaint(Paint cellBorderPaint)
  {
    if ((cellBorderPaint == null  &&  this.cellBorderPaint != null)  ||
            (cellBorderPaint != null  &&  !cellBorderPaint.equals(this.cellBorderPaint))) {
      Paint oldPaint = this.cellBorderPaint;
      this.cellBorderPaint = cellBorderPaint;
      firePropertyChange(PROPERTY_CELL_BORDER_PAINT, oldPaint, cellBorderPaint);
    }
  }

  /**
   * Get the paint of solution cells.
   * @return solution cell paint
   * @see #isShowingSolution()
   */
  public Paint getSolutionPaint()
  {
    return solutionPaint;
  }

  /**
   * Set the paint of solution cells.
   * @param solutionPaint solution cell paint
   * @see #isShowingSolution()
   */
  public void setSolutionPaint(Paint solutionPaint)
  {
    if (!solutionPaint.equals(this.solutionPaint)) {
      Paint oldPaint = this.solutionPaint;
      this.solutionPaint = solutionPaint;
      firePropertyChange(PROPERTY_SOLUTION_PAINT, oldPaint, solutionPaint);
    }
  }

  /**
   * Get the background paint.
   * @return background paint
   */
  public Paint getBackgroundPaint()
  {
    return backgroundPaint;
  }

  /**
   * Set the background paint.
   * @param backgroundPaint background paint
   */
  public void setBackgroundPaint(Paint backgroundPaint)
  {
    if ((backgroundPaint == null  &&  this.backgroundPaint != null)  ||
            (backgroundPaint != null  &&  !backgroundPaint.equals(this.backgroundPaint))) {
      Paint oldPaint = this.backgroundPaint;
      this.backgroundPaint = backgroundPaint;
      firePropertyChange(PROPERTY_BACKGROUND_PAINT, oldPaint, backgroundPaint);
    }
  }

  /**
   *  Get the paint for the start cell.
   *  @return start cell paint
   */
  public Paint getWayStartPaint()
  {
    return wayStartPaint;
  }

  /**
   *  Set the paint for the start cell.
   *  @param wayStartPaint start cell paint
   */
  public void setWayStartPaint(Paint wayStartPaint)
  {
    if (!wayStartPaint.equals(this.wayStartPaint)) {
      Paint oldPaint = this.wayStartPaint;
      this.wayStartPaint = wayStartPaint;
      firePropertyChange(PROPERTY_SOLUTION_PAINT, oldPaint, wayStartPaint);
    }
  }

  /**
   *  Get the paint for the end cell.
   *  @return end cell paint
   */
  public Paint getWayEndPaint()
  {
    return wayEndPaint;
  }

  /**
   *  Set the paint for the end cell.
   *  @param wayEndPaint end cell paint
   */
  public void setWayEndPaint(Paint wayEndPaint)
  {
    if (!wayEndPaint.equals(this.wayEndPaint)) {
      Paint oldPaint = this.wayEndPaint;
      this.wayEndPaint = wayEndPaint;
      firePropertyChange(PROPERTY_SOLUTION_PAINT, oldPaint, wayEndPaint);
    }
  }

  /**
   *  Get the property informations.
   *  @return collection of property informations
   */
  public Collection<PropertyInformation> getPropertyInformations()
  {
    Collection<PropertyInformation> setters = new ArrayList<PropertyInformation>();

    setters.add(new BooleanPropertyInformation(PROPERTY_SHOWING_SOLUTION) {
      protected void setOwnerValue(boolean state)
      {
        setShowingSolution(state);
      }

      protected boolean getOwnerValue()
      {
        return isShowingSolution();
      }

    });

    setters.add(new BooleanPropertyInformation(PROPERTY_SHOWING_CELL_BORDERS) {
      protected void setOwnerValue(boolean state)
      {
        setShowingCellBorders(state);
      }

      protected boolean getOwnerValue()
      {
        return isShowingCellBorders();
      }
    });

    setters.add(new PaintPropertyInformation(PROPERTY_OUTER_BORDER_PAINT, false) {
      protected void setOwnerValue(Paint paint)
      {
        setOuterBorderPaint(paint);
      }

      protected Paint getOwnerValue()
      {
        return getOuterWallPaint();
      }
    });

    setters.add(new PaintPropertyInformation(PROPERTY_INNER_BORDER_PAINT, false) {
      protected void setOwnerValue(Paint paint)
      {
        setInnerBorderPaint(paint);
      }

      protected Paint getOwnerValue()
      {
        return getInnerWallsPaint();
      }
    });

    setters.add(new PaintPropertyInformation(PROPERTY_SOLUTION_PAINT, false) {
      protected void setOwnerValue(Paint paint)
      {
        setSolutionPaint(paint);
      }

      protected Paint getOwnerValue()
      {
        return getSolutionPaint();
      }
    });

    setters.add(new PaintPropertyInformation(PROPERTY_WAY_START_PAINT, false) {
      protected void setOwnerValue(Paint paint)
      {
        setWayStartPaint(paint);
      }

      protected Paint getOwnerValue()
      {
        return getWayStartPaint();
      }
    });

    setters.add(new PaintPropertyInformation(PROPERTY_WAY_END_PAINT, false) {
      protected void setOwnerValue(Paint paint)
      {
        setWayEndPaint(paint);
      }

      protected Paint getOwnerValue()
      {
        return getWayEndPaint();
      }
    });

    setters.add(new PaintPropertyInformation(PROPERTY_BACKGROUND_PAINT, true) {
      protected void setOwnerValue(Paint paint)
      {
        setBackgroundPaint(paint);
      }

      protected Paint getOwnerValue()
      {
        return getBackgroundPaint();
      }
    });

    return setters;
  }

  /**
   *  Overwrite the properties from the given data storage.
   *  @param dataStorage data storage to use for access to stored settings
   */
  void overwriteFromPersistentData(DataStorage dataStorage)
  {
    if (dataStorage == null) {
      return;
    }
    setShowingSolution(dataStorage.getBoolean(myKey+PROPERTY_SHOWING_SOLUTION,
                                                         isShowingSolution()));
    setShowingCellBorders(dataStorage.getBoolean(myKey+PROPERTY_SHOWING_CELL_BORDERS,
                                                            isShowingCellBorders()));
    Color c = dataStorage.getColor(myKey+PROPERTY_OUTER_BORDER_PAINT, null);
    if (c != null) {
      setOuterBorderPaint(c);
    }

    c = dataStorage.getColor(myKey+PROPERTY_INNER_BORDER_PAINT, null);
    if (c != null) {
      setInnerBorderPaint(c);
    }

    c = dataStorage.getColor(myKey+PROPERTY_CELL_BORDER_PAINT, null);
    if (c != null) {
      setCellBorderPaint(c);
    }

    c = dataStorage.getColor(myKey+PROPERTY_SOLUTION_PAINT, null);
    if (c != null) {
      setSolutionPaint(c);
    }

    c = dataStorage.getColor(myKey+PROPERTY_BACKGROUND_PAINT, null);
    if (c != null) {
      setBackgroundPaint(c);
    }

    c = dataStorage.getColor(myKey+PROPERTY_WAY_START_PAINT, null);
    if (c != null) {
      setWayStartPaint(c);
    }

    c = dataStorage.getColor(myKey+PROPERTY_WAY_END_PAINT, null);
    if (c != null) {
      setWayEndPaint(c);
    }
  }

  /**
   *  Store a paint to a data storage.
   *
   *  Warning: this is currently only handling paint object which are colors, all others are silently discarded!
   *  @param dataStorage  data storage to store to
   *  @param key          property key
   *  @param paint        paint to store (must be a color to work)
   */
  private static void storePaint(DataStorage dataStorage, String key, Paint paint)
  {
    try {
      Color c = (Color)paint;
      dataStorage.setColor(key, c);
    } catch (ClassCastException x) {
    }
  }

  /**
   *  Store the settings in a data storage.
   *  @param dataStorage storage to store to
   */
  void storePersistentData(DataStorage dataStorage)
  {
    dataStorage.setBoolean(myKey+PROPERTY_SHOWING_SOLUTION, isShowingSolution());
    dataStorage.setBoolean(myKey+PROPERTY_SHOWING_CELL_BORDERS, isShowingCellBorders());

    storePaint(dataStorage, myKey+PROPERTY_OUTER_BORDER_PAINT, getOuterWallPaint());
    storePaint(dataStorage, myKey+PROPERTY_INNER_BORDER_PAINT, getInnerWallsPaint());
    storePaint(dataStorage, myKey+PROPERTY_CELL_BORDER_PAINT,  getCellBorderPaint());
    storePaint(dataStorage, myKey+PROPERTY_SOLUTION_PAINT,     getSolutionPaint());
    storePaint(dataStorage, myKey+PROPERTY_BACKGROUND_PAINT,   getBackgroundPaint());
    storePaint(dataStorage, myKey+PROPERTY_WAY_START_PAINT,    getWayStartPaint());
    storePaint(dataStorage, myKey+PROPERTY_WAY_END_PAINT,      getWayEndPaint());
  }
}
