// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 19:15:39 $
//
// History:	       $Log: MazeFrame.java,v $
// History:	       Revision 1.13  2012/06/07 19:15:39  rammi
// History:	       Removed properties dialog from SVG save
// History:
// History:	       Revision 1.12  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.11  2009/09/24 17:32:20  rammi
// History:	       Finetuning dialog placement
// History:
// History:	       Revision 1.10  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.9  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.8  2005/04/19 20:00:44  rammi
// History:	       no message
// History:	
// History:	       Revision 1.7  2004/11/02 20:54:26  rammi
// History:	       Advanced version
// History:	
// History:	       Revision 1.6  2004/11/02 20:53:53  rammi
// History:	       Added keyboard shortcuts etc via actions
// History:	
// History:	       Revision 1.5  2004/11/02 14:13:08  rammi
// History:	       Added keyboard shortcuts for the menus
// History:	
// History:	       Revision 1.4  2004/10/31 23:30:52  rammi
// History:	       Redraw speed-up
// History:	
// History:	       Revision 1.3  2004/10/31 22:06:59  rammi
// History:	       Updated version
// History:	
// History:	       Revision 1.2  2004/10/31 15:09:04  rammi
// History:	       Updated version
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import de.caff.gimmix.I18n;
import de.caff.gimmix.ResourcedAction;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.WindowAdapter;
import java.awt.event.WindowEvent;
import java.awt.image.BufferedImage;
import java.io.OutputStream;
import java.util.MissingResourceException;

/**
 *  The main window of a simple maze application.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.13 $
 */
public class MazeFrame
        extends JFrame
        implements Constants
{
  /** The i18n resource for the window title. */
  public static final String RESOURCE_TITLE = "MAZE_FRAME:title";
  /** The i18n resource for the file menu title. */
  public static final String RESOURCE_MENU_FILE = "MAZE_FRAME:menuFile";
  /** The i18n resource for the file open menu item. */
  public static final String RESOURCE_MENUITEM_OPEN  = "MAZE_FRAME:menuItemOpen";
  /** The i18n resource for the file save menu item. */
  public static final String RESOURCE_MENUITEM_SAVE  = "MAZE_FRAME:menuItemSave";
  /** The i18n resource for the file save image menu item. */
  public static final String RESOURCE_MENUITEM_SAVE_IMAGE = "MAZE_FRAME:menuItemSaveImage";
  /** The i18n resource for the DXF file save image menu item. */
  public static final String RESOURCE_MENUITEM_SAVE_DXF = "MAZE_FRAME:menuItemSaveDXF";
  /** The i18n resource for the SVG file save image menu item. */
  public static final String RESOURCE_MENUITEM_SAVE_SVG = "MAZE_FRAME:menuItemSaveSVG";
  /** The i18n resource for the print menu item. */
  public static final String RESOURCE_MENUITEM_PRINT = "MAZE_FRAME:menuItemPrint";
  /** The i18n resource for the print setup menu item. */
  public static final String RESOURCE_MENUITEM_PRINT_SETUP = "MAZE_FRAME:menuItemPrintSetup";
  /** The i18n resource for the exit menu item. */
  public static final String RESOURCE_MENUITEM_EXIT  = "MAZE_FRAME:menuItemExit";

  /** The i18n resource for file write error, arg 0 is the exception. */
  public static final String RESOURCE_ERROR_FILE_WRITE = "MAZE_FRAME:errorFileWrite";
  /** The i18n resource for file read error, arg 0 is the exception. */
  public static final String RESOURCE_ERROR_FILE_READ  = "MAZE_FRAME:errorFileRead";
  /** THe i18n resource for access to file system error. */
  public static final String RESOURCE_ERROR_FS_ACCESS  = "MAZE_FRAME:errorFsAccess";

  // make sure i18n resources are loaded
  static {
    I18n.addAppResourceBase("de.caff.maze.MazeResourceBundle");
  }

  /**
   *  The system access.
   *  It is set differently according to the environment (Java Web Start or direct).
   */
  private final SystemAccess systemAccess;
  /** The panel holding the maze and the maze controls. */
  private final MultiMazeControlPanel mmcp;

  /**
   *  Constructor.
   *
   *  @param systemAccess access to the system where the application was startet (direct or Java Web Start).
   */
  public MazeFrame(SystemAccess systemAccess)
  {
    super(I18n.getString(RESOURCE_TITLE)+" ["+VERSION+"]");

    this.systemAccess = systemAccess;

    addMenuBar();

    getContentPane().add(mmcp = new MultiMazeControlPanel(systemAccess));

    addWindowListener(new WindowAdapter() {
      public void windowClosing(WindowEvent e)
      {
        shutdown();
      }
    });
  }



  /**
   *  Add the menu bar.
   */
  private void addMenuBar()
  {
    JMenuBar menuBar = new JMenuBar();
    setJMenuBar(menuBar);
    JMenu fileMenu = new JMenu(I18n.getString(RESOURCE_MENU_FILE));
    // try setting mnemonic
    try {
      String mnemonic = I18n.getString(RESOURCE_MENU_FILE+ResourcedAction.MNEMO);
      if (mnemonic.length() > 0) {
        fileMenu.setMnemonic(mnemonic.charAt(0));
      }
    } catch (MissingResourceException e) {
    }
    fileMenu.add(new ResourcedAction(RESOURCE_MENUITEM_OPEN) {
      public void actionPerformed(ActionEvent e)
      {
        openMaze();
      }

      @Override
      protected Object clone() throws CloneNotSupportedException
      {
        throw new CloneNotSupportedException();
      }
    });
    fileMenu.add(new ResourcedAction(RESOURCE_MENUITEM_SAVE) {
      public void actionPerformed(ActionEvent e)
      {
        saveMaze();
      }

      @Override
      protected Object clone() throws CloneNotSupportedException
      {
        throw new CloneNotSupportedException();
      }
    });
    fileMenu.add(new ResourcedAction(RESOURCE_MENUITEM_SAVE_IMAGE) {
      public void actionPerformed(ActionEvent e)
      {
        saveImage();
      }

      @Override
      protected Object clone() throws CloneNotSupportedException
      {
        throw new CloneNotSupportedException();
      }
    });
    fileMenu.add(new ResourcedAction(RESOURCE_MENUITEM_SAVE_DXF) {
      public void actionPerformed(ActionEvent e)
      {
        saveAsDXF();
      }

      @Override
      protected Object clone() throws CloneNotSupportedException
      {
        throw new CloneNotSupportedException();
      }
    });
    fileMenu.add(new ResourcedAction(RESOURCE_MENUITEM_SAVE_SVG) {
      public void actionPerformed(ActionEvent e)
      {
        saveAsSVG();
      }

      @Override
      protected Object clone() throws CloneNotSupportedException
      {
        throw new CloneNotSupportedException();
      }
    });
    fileMenu.addSeparator();
    fileMenu.add(new ResourcedAction(RESOURCE_MENUITEM_PRINT_SETUP) {
      public void actionPerformed(ActionEvent e)
      {
        PrintPropertiesDialog dialog = new PrintPropertiesDialog(MazeFrame.this, mmcp.getPrintProperties());
        dialog.setVisible(true);
      }

      @Override
      protected Object clone() throws CloneNotSupportedException
      {
        throw new CloneNotSupportedException();
      }
    });
    fileMenu.add(new ResourcedAction(RESOURCE_MENUITEM_PRINT) {
      public void actionPerformed(ActionEvent e)
      {
        mmcp.printCurrentMaze(systemAccess);
      }

      @Override
      protected Object clone() throws CloneNotSupportedException
      {
        throw new CloneNotSupportedException();
      }
    });
    fileMenu.addSeparator();
    fileMenu.add(new ResourcedAction(RESOURCE_MENUITEM_EXIT) {
      public void actionPerformed(ActionEvent e)
      {
        shutdown();
      }

      @Override
      protected Object clone() throws CloneNotSupportedException
      {
        throw new CloneNotSupportedException();
      }
    });

    menuBar.add(fileMenu);
  }

  /**
   *  Show an error message.
   *  @param message error message
   */
  private void showError(String message)
  {
    JOptionPane.showMessageDialog(this, message);
  }

  /**
   *  Save the currently displayed maze.
   */
  private void saveMaze()
  {
    FileDataStorage dataStorage = new FileDataStorage(systemAccess);
    mmcp.saveCurrentMaze(dataStorage);
    try {
      if (!dataStorage.save()) {
        showError(I18n.getString(RESOURCE_ERROR_FS_ACCESS));
      }
    } catch (Exception e) {
      showError(I18n.format(RESOURCE_ERROR_FILE_WRITE, e.getLocalizedMessage()));
    }
  }

  /**
   *  Save the currently displayed maze as an image.
   */
  private void saveImage()
  {
    try {
      getContentPane().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
      SaveImagePropertiesDialog dialog = new SaveImagePropertiesDialog(this, systemAccess);
      dialog.setModal(true);
      dialog.setLocationRelativeTo(this);
      dialog.setVisible(true);
      MazeSaveImageProperties properties = dialog.getSaveImageProperties();
      if (properties != null) {
        properties.storePersistentData(systemAccess);
        BufferedImage image = mmcp.createImage(properties);
        FileDataStorage dataStorage = new FileDataStorage(systemAccess);
        try {
          if (!dataStorage.saveImage(image)) {
            showError(I18n.getString(RESOURCE_ERROR_FS_ACCESS));
          }
        } catch (Exception e) {
          showError(I18n.format(RESOURCE_ERROR_FILE_WRITE, e.getLocalizedMessage()));
        }
      }
    } finally {
      getContentPane().setCursor(Cursor.getDefaultCursor());
    }
  }

  /**
   * Save the currently displayed maze in SVG format.
   */
  private void saveAsSVG()
  {
    try {
      getContentPane().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
      MazePrintProperties properties = MazePrintProperties.getPrintMazePaintProperties(systemAccess);
      FileDataStorage dataStorage = new FileDataStorage(systemAccess);
      try {
        if (!dataStorage.saveToPainter(mmcp.getCurrentMaze(),
                                       properties,
                                       SvgMazePainter.FILE_TYPE,
                                       new OutputMazePainterCreator()
                                       {
                                         public MazePainter createPainter(OutputStream out)
                                         {
                                           return new SvgMazePainter(out);
                                         }
                                       })) {
          showError(I18n.getString(RESOURCE_ERROR_FS_ACCESS));
        }
      } catch (Exception e) {
        showError(I18n.format(RESOURCE_ERROR_FILE_WRITE, e.getLocalizedMessage()));
      }
    } finally {
      getContentPane().setCursor(Cursor.getDefaultCursor());
    }
  }

  /**
   * Save the currently displayed maze in DXF format.
   */
  private void saveAsDXF()
  {
    try {
      getContentPane().setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
      FileDataStorage dataStorage = new FileDataStorage(systemAccess);
      try {
        if (!dataStorage.saveToPainter(mmcp.getCurrentMaze(),
                                       new MazePaintPropertiesProvider()
                                       {
                                         public boolean isShowingSolution()
                                         {
                                           return false; // DXF painter does not paint solution
                                         }

                                         public boolean isShowingCellBorders()
                                         {
                                           return false; // DXF painter does not paint solution
                                         }

                                         public Paint getOuterWallPaint()
                                         {
                                           return null; // paint is ignored by DXF painter
                                         }

                                         public Paint getInnerWallsPaint()
                                         {
                                           return null;  // paint is ignored by DXF painter
                                         }

                                         public Paint getCellBorderPaint()
                                         {
                                           return null;  // paint is ignored by DXF painter
                                         }

                                         public Paint getSolutionPaint()
                                         {
                                           return null;  // paint is ignored by DXF painter
                                         }

                                         public Paint getBackgroundPaint()
                                         {
                                           return null;  // paint is ignored by DXF painter
                                         }

                                         public Paint getWayStartPaint()
                                         {
                                           return null;  // paint is ignored by DXF painter
                                         }

                                         public Paint getWayEndPaint()
                                         {
                                           return null;  // paint is ignored by DXF painter
                                         }
                                       },
                                       DxfMazePainter.FILE_TYPE,
                                       new OutputMazePainterCreator()
                                       {
                                         public MazePainter createPainter(OutputStream out)
                                         {
                                           return new DxfMazePainter(out);
                                         }
                                       })) {
          showError(I18n.getString(RESOURCE_ERROR_FS_ACCESS));
        }
      } catch (Exception e) {
        showError(I18n.format(RESOURCE_ERROR_FILE_WRITE, e.getLocalizedMessage()));
      }
    } finally {
      getContentPane().setCursor(Cursor.getDefaultCursor());
    }
  }

  /**
   *  Load a maze.
   */
  private void openMaze()
  {
    FileDataStorage dataStorage = new FileDataStorage(systemAccess);
    try {
      if (dataStorage.load()) {
        if (!mmcp.loadMaze(dataStorage)) {
          showError(I18n.getString(RESOURCE_ERROR_FS_ACCESS));
        }
      }
    } catch (Exception e) {
      showError(I18n.format(RESOURCE_ERROR_FILE_READ, e.getLocalizedMessage()));
    }
  }

  /**
   *  End this application.
   *  Do necessary cleanup first.
   */
  private void shutdown()
  {
    mmcp.storePersistentData(systemAccess);
    systemAccess.storePersistentData();
    System.exit(0);
  }

  /**
   *  Standard entry point.
   *  @param args unused
   */
  public static void main(String[] args)
  {
    SystemAccess systemAccess;
    // look for JNLP
    SecurityManager securityManager = System.getSecurityManager();
    if (securityManager != null) {
      // has security manager
      try {
        Class.forName("javax.jnlp.ServiceManager");
        systemAccess = new JnlpSystemAccess();
      } catch (Throwable x) {
        systemAccess = new NullSystemAccess();
      }
    }
    else {
      systemAccess = new ApplicationSystemAccess();
    }
    MazeFrame frame = new MazeFrame(systemAccess);
    systemAccess.setMainFrame(frame);
    final Dimension screenSize = frame.getToolkit().getScreenSize();
    final Insets screenInsets = frame.getToolkit().getScreenInsets(frame.getGraphicsConfiguration());
    frame.setBounds(new Rectangle(screenInsets.left, screenInsets.top,
                                  screenSize.width-screenInsets.left-screenInsets.right,
                                  screenSize.height-screenInsets.top-screenInsets.bottom));
    frame.setVisible(true);
  }
}
