// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: MazeCanvas.java,v $
// History:	       Revision 1.8  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.7  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.6  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.5  2005/04/19 22:15:14  rammi
// History:	       Fixed some Thread problems
// History:	
// History:	       Revision 1.4  2005/04/19 20:00:44  rammi
// History:	       no message
// History:	
// History:	       Revision 1.3  2004/10/31 23:30:52  rammi
// History:	       Redraw speed-up
// History:	
// History:	       Revision 1.2  2004/10/31 15:07:05  rammi
// History:	       Changed drawing to be always in BOX_SIZE
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import de.caff.gimmix.I18n;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.GeneralPath;
import java.awt.geom.Point2D;
import java.awt.image.BufferedImage;
import java.awt.print.PageFormat;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.Arrays;
import java.util.Comparator;

/**
 *  A canvas for maze display (including progress display if creation takes too long).
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.8 $
 */
public class MazeCanvas
        extends JPanel
        implements ProgressShower, Printable, MazeFinishedListener
{
  /** Basic resource key for abort button. */
  public static final String RESOURCE_BUTTON_ABORT = "MAZE_CANVAS:buttonAbort";
  /** Basic resource key for message when displaying dialog with exception during creation. */
  public static final String MESSAGE_CREATION_EXCEPTION = "MAZE_CANVAS:messageCreationException";
  /** Basic resource key for title when displaying dialog with exception during creation. */
  public static final String TITLE_CREATION_EXCEPTION = "MAZE_CANVAS:titleCreationException";

  /** Card layout id for standard view. */
  private static final String STANDARD_VIEW = "Standard";
  /** Card layout id for progress view. */
  private static final String PROGRESS_VIEW = "Progress";
  /** The maze currently displayed. */
  private final AbstractBasicMaze maze;
  /** Double-buffered image. */
  private Image buffer;
  /** Repaint double buffer? */
  private boolean expliciteRepaint;
  /** Maze cell hit by mouse. */
  private MazeCell mouseCell = null;
  /** Additional translation when painting. */
  private float[] translation = { 0, 0 };
  /** Card layout to switch between standard and progress view. */
  private final CardLayout cardLayout = new CardLayout();
  /** Label where notes are displayed. */
  private final JLabel  noteLabel;
  /** Progress bar during creation. */
  private final JProgressBar progressBar;
  /** Has the user pressed abort? */
  private boolean abortAction = false;
  /** Thread for delayed display of progress panel. */
  private Thread delayThread;
  /** The component where the maze is displayed. */
  private final MazeComponent mazeComponent;
  /** Maze properties (colors etc) when painting. */
  private final MazePaintPropertiesProvider drawProperties;
  /** Maze properties (colors etc) when printing. */
  private final MazePrintPropertiesProvider printProperties;
  /** Button to abort ongoing action. */
  private final JButton abortButton;

  /**
   *  Similar to java.awt.Dimension, but using floats.
   */
  private static class FloatDimension
  {
    /** Width. */
    private final float width;
    /** Height. */
    private final float height;

    /**
     * Constructor.
     * @param width  the width
     * @param height the height
     */
    public FloatDimension(float width, float height)
    {
      this.width = width;
      this.height = height;
    }

    /**
     * Get the width.
     * @return the width
     */
    public float getWidth()
    {
      return width;
    }

    /**
     * Get the height.
     * @return the height
     */
    public float getHeight()
    {
      return height;
    }
  }

  /**
   *  How to fit a drawing onto several pages.
   */
  private static class Fit
  {
    /** The factor used for the height. */
    private final int heightFactor;
    /** The factor used for the width. */
    private final int widthFactor;

    /**
     *  Create fit option.
     *  @param heightFactor factor for height
     *  @param widthFactor  factor for width
     */
    public Fit(int heightFactor, int widthFactor)
    {
      this.heightFactor      = heightFactor;
      this.widthFactor       = widthFactor;
    }

    /**
     *  Get the height factor.
     *  @return height factor
     */
    public int getHeightFactor()
    {
      return heightFactor;
    }

    /**
     *  Get the width factor.
     *  @return width factor
     */
    public int getWidthFactor()
    {
      return widthFactor;
    }

    /**
     *  Get the proportion used when fitting.
     *  @param width        width of image
     *  @param height       heigth of image
     *  @param widthDiff    width difference
     *  @param heightDiff   height difference
     *  @return scaling factor
     */
    public double getProportion(double width, double height, double widthDiff, double heightDiff)
    {
      return proportion(width*widthFactor-widthDiff, height*heightFactor-heightDiff);
    }
  }

  /**
   *  Component where a maze is drawn (or printed).
   */
  private class MazeComponent
          extends JComponent
          implements Printable
  {
    /** Heuristic factor how to reduce the size so everything fits even with thicker borders. */
    private static final double SIZE_REDUCE_FACTOR = 0.98;
    /** Offset for text when printing. */
    private static final double TEXT_OFFSET = 10;
    /** Size of information when printing. */
    private static final int INFO_FONT_SIZE = 5;

    /**
     *  Constructor.
     */
    public MazeComponent()
    {
      addMouseMotionListener(new MouseMotionAdapter() {
        @Override public void mouseMoved(MouseEvent e)
        {
          if (false) {
            Point2D position = correctMousePosition(e.getX(), e.getY());

            MazeCell oldCell = mouseCell;
            mouseCell = MazeCanvas.this.maze.getCellAt(position);

            if (oldCell != mouseCell) {
              repaint();
            }
          }
        }
      });

      addMouseListener(new MouseAdapter() {
        @Override public void mouseExited(MouseEvent e)
        {
          mouseCell = null;
          repaint();
        }

        @Override public void mouseClicked(MouseEvent e)
        {
          Point2D position = correctMousePosition(e.getX(), e.getY());

          MazeCell cell = MazeCanvas.this.maze.getCellAt(position);
          if (e.getButton() == MouseEvent.BUTTON1) {
            if ((e.getModifiersEx() & MouseEvent.SHIFT_DOWN_MASK) != 0) {
              MazeCanvas.this.maze.setWayEnd(cell);
            }
            else {
              MazeCanvas.this.maze.setWayStart(cell);
            }
          }
          else if (e.getButton() == MouseEvent.BUTTON3) {
            MazeCanvas.this.maze.setWayEnd(cell);
          }
        }
      });
      //setCursor(Cursor.getPredefinedCursor(Cursor.CROSSHAIR_CURSOR));
    }

    /**
     *  Map a mouse position into the coordinate system of the displayed maze.
     *  @param x  mouse x
     *  @param y  mouse y
     *  @return corresponding position in maze coordinates
     */
    private Point2D correctMousePosition(int x, int y)
    {
      final Dimension size = getSize();
      float width = size.width-10;
      float height = size.height-10;
      if (width/height > maze.getPreferredAspectRatio()) {
        width = maze.getPreferredAspectRatio()*height;
      }
      else {
        height = (int)(width/maze.getPreferredAspectRatio());
      }
      return new Point2D.Float((x - translation[0])*AbstractBasicMaze.BOX_SIZE/width,
                               (y - translation[1])*AbstractBasicMaze.BOX_SIZE/height);
    }

    /**
     *  Paint this component. This paints the maze.
     *  @param g graphics context
     */
    @Override public void paint(Graphics g)
    {
      final AbstractBasicMaze maze = MazeCanvas.this.maze;
      if (maze.isDuringRecreation()) {
        return;
      }

      super.paint(g);

      final Graphics2D g2 = (Graphics2D)g;
      final Dimension size = getSize();
      float width = size.width-10;
      float height = size.height-10;
      if (width/height > maze.getPreferredAspectRatio()) {
        width = maze.getPreferredAspectRatio()*height;
      }
      else {
        height = (int)(width/maze.getPreferredAspectRatio());
      }
      final boolean needNewBuffer = buffer == null  ||  buffer.getWidth(this) != size.width  ||  buffer.getHeight(this) != size.height;
      if (expliciteRepaint || needNewBuffer) {
        if (needNewBuffer) {
          buffer = new BufferedImage(size.width, size.height, BufferedImage.TYPE_INT_ARGB);
        }
        expliciteRepaint = false;
        final Graphics2D bg2 = (Graphics2D)buffer.getGraphics();
        bg2.setBackground(new Color(0, 0, 0, 0));
        bg2.clearRect(0, 0, size.width, size.height);
        bg2.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS,
                             RenderingHints.VALUE_FRACTIONALMETRICS_ON);
        bg2.setRenderingHint(RenderingHints.KEY_RENDERING,
                             RenderingHints.VALUE_RENDER_QUALITY);
        bg2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                             RenderingHints.VALUE_ANTIALIAS_ON);
        bg2.translate(translation[0] = (size.width-width)/2,
                      translation[1] = (size.height-height)/2);
        bg2.scale(width/AbstractBasicMaze.BOX_SIZE,
                  height/AbstractBasicMaze.BOX_SIZE);
        maze.draw(new Graphics2DMazePainter(bg2), drawProperties);
      }

      g2.drawImage(buffer, 0, 0, this);

      if (mouseCell != null) {
        g2.translate(translation[0], translation[1]);
        g2.scale(width/AbstractBasicMaze.BOX_SIZE,
                 height/AbstractBasicMaze.BOX_SIZE);
        g2.setPaint(Color.magenta);
        g2.fill(mouseCell.getShape());
      }
    }

    /**
     *  Prepare the printing of the maze.
     *  This method evaluates the print properties and sets the necessary
     *  transformations. It also prints info and fitters, if necessary.
     *  @param g2          graphics print context
     *  @param pageFormat  page format description
     *  @param pageIndex   page index
     *  @param info        additional info to be printed (may be <code>null</code>)
     *  @return the dimension on which to print
     */
    private FloatDimension preparePrintGraphics(Graphics2D g2,
                                                PageFormat pageFormat, 
                                                int pageIndex,
                                                String info)
    {
      if (printProperties.getBlowUpFactor() == MazePrintPropertiesProvider.BlowUpFactor.BLOW_UP_SINGLE) {
        // simple one page drawing
        if (pageIndex == 0) {
          final double pageX      = pageFormat.getImageableX();
          double pageY      = pageFormat.getImageableY();
          double pageWidth  = pageFormat.getImageableWidth();
          double pageHeight = pageFormat.getImageableHeight();
          double textOffset;
          if (info != null) {
            g2.setFont(new Font("SansSerif", Font.PLAIN, INFO_FONT_SIZE));
            g2.setColor(Color.black);
            g2.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
            g2.drawString(info, (float)pageX, (float)pageY+INFO_FONT_SIZE);
            textOffset = TEXT_OFFSET;
          }
          else {
            textOffset = 0;
          }
          if ((maze.getPreferredAspectRatio() > 1.0  &&  pageWidth < pageHeight) ||
              maze.getPreferredAspectRatio() < 1.0 &&  pageHeight < pageWidth) {
            // rotate
            double minSize = Math.min(pageFormat.getWidth(), pageFormat.getHeight());
            g2.rotate(Math.PI/2, minSize/2, minSize/2);
            double tmp = pageWidth;
            pageWidth  = pageHeight;
            pageHeight = tmp;
          }
          pageHeight -= textOffset;
          double width  = pageWidth*SIZE_REDUCE_FACTOR;
          double height = pageHeight*SIZE_REDUCE_FACTOR;
          if (width/height > maze.getPreferredAspectRatio()) {
            width = maze.getPreferredAspectRatio()*height;
          }
          else {
            height = (int)(width/maze.getPreferredAspectRatio());
          }
          g2.translate((pageWidth-width)/2+pageX, (pageHeight-height)/2+pageY+textOffset);
          return new FloatDimension((float)width, (float)height);
        }
      }
      else {
        // complex more page drawing
        final double pageX      = pageFormat.getImageableX();
        final double pageY      = pageFormat.getImageableY();
        final double pageWidth  = pageFormat.getImageableWidth();
        final double pageHeight = pageFormat.getImageableHeight();
        final double mazeProportion = proportion(maze.getPreferredAspectRatio(), 1);
        Fit[] fits = null;
        switch (printProperties.getBlowUpFactor()) {
        case BLOW_UP_DOUBLE:
          fits = new Fit[] {
            new Fit(2, 1),
            new Fit(1, 2)
          };
          break;

        case BLOW_UP_QUAD:
          fits = new Fit[] {
            new Fit(4, 1),
            new Fit(2, 2),
            new Fit(1, 4)
          };
          break;

        case BLOW_UP_EIGHT:
          fits = new Fit[] {
            new Fit(8, 1),
            new Fit(4, 2),
            new Fit(2, 4),
            new Fit(1, 8)
          };
          break;

        case BLOW_UP_SIXTEEN:
          fits = new Fit[] {
            new Fit(16, 1),
            new Fit(8, 2),
            new Fit(4, 4),
            new Fit(2, 8),
            new Fit(1, 16)
          };
          break;
        }
        Arrays.sort(fits, new Comparator<Fit>() {
          public int compare(Fit fit1, Fit fit2)
          {
            double prop1 = proportion(fit1.getProportion(pageWidth, pageHeight, 2*TEXT_OFFSET, 2*TEXT_OFFSET), mazeProportion);
            double prop2 = proportion(fit2.getProportion(pageWidth, pageHeight, 2*TEXT_OFFSET, 2*TEXT_OFFSET), mazeProportion);
            if (prop1 < prop2) {
              return -1;
            }
            if (prop1 > prop2) {
              return 1;
            }
            return 0;
          }
        });
        final Fit bestFit = fits[0];

        if (pageIndex < bestFit.getWidthFactor()*bestFit.getHeightFactor()) {
          int widthFactor  = bestFit.getWidthFactor();
          int heightFactor = bestFit.getHeightFactor();
          double completeWidth  = widthFactor*pageWidth   - 2*TEXT_OFFSET;
          double completeHeight = heightFactor*pageHeight - 2*TEXT_OFFSET;
          double offsetX;
          double offsetY;

          if (info != null  &&  pageIndex == 0) {
            g2.setFont(new Font("SansSerif", Font.PLAIN, INFO_FONT_SIZE));
            g2.setColor(Color.black);
            g2.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS, RenderingHints.VALUE_FRACTIONALMETRICS_ON);
            g2.drawString(info, (float)pageX, (float)pageY+INFO_FONT_SIZE);
          }

          // print marker
          Graphics2D markerG = (Graphics2D)g2.create();
          markerG.translate(pageX-(pageIndex%widthFactor)*pageWidth, pageY-(pageIndex/widthFactor)*pageHeight);
          markerG.setFont(new Font("SansSerif", Font.PLAIN, 10));
          markerG.setColor(Color.black);
          char mark = 'A';
          for (int i = bestFit.getWidthFactor()*bestFit.getHeightFactor()-1;  i >= 0;  --i) {
            final int row    = i/widthFactor;
            final int column = i%widthFactor;
            if (row == 0) {
              if (column > 0) {
                setHorizontalFitter(markerG, (float)(column*pageWidth), FITTER_DELTA, mark++);
              }
            }
            if (row == heightFactor-1) {
              if (column > 0) {
                setHorizontalFitter(markerG, (float)(column*pageWidth), (float)((row+1)*pageHeight-FITTER_DELTA), mark++);
              }
            }
            if (column == 0) {
              if (row > 0) {
                setVerticalFitter(markerG, FITTER_DELTA, (float)(row*pageHeight), mark++);
              }
            }
            if (column == widthFactor-1) {
              if (row > 0) {
                setVerticalFitter(markerG, (float)((column+1)*pageWidth-FITTER_DELTA),
                                  (float)(row*pageHeight), mark++);
              }
            }
          }

          if ((maze.getPreferredAspectRatio() > 1.0  &&  completeWidth < completeHeight) ||
              maze.getPreferredAspectRatio() < 1.0 &&  completeHeight < completeWidth) {
            // rotate
            double minSize = Math.min(pageFormat.getWidth(), pageFormat.getHeight());
            g2.rotate(Math.PI/2, minSize/2, minSize/2);
            double temp = completeWidth;
            completeWidth  = completeHeight;
            completeHeight = temp;
            offsetX = pageHeight*(pageIndex/bestFit.getWidthFactor());
            offsetY = pageWidth *(pageIndex%bestFit.getWidthFactor());
          }
          else {
            offsetX = pageWidth *(pageIndex%bestFit.getWidthFactor());
            offsetY = pageHeight*(pageIndex/bestFit.getWidthFactor());
          }

          double width  = completeWidth*SIZE_REDUCE_FACTOR;
          double height = completeHeight*SIZE_REDUCE_FACTOR;
          if (width/height > maze.getPreferredAspectRatio()) {
            width = maze.getPreferredAspectRatio()*height;
          }
          else {
            height = width/maze.getPreferredAspectRatio();
          }
          g2.translate((completeWidth-width)/2   + pageX - offsetX + TEXT_OFFSET,
                       (completeHeight-height)/2 + pageY - offsetY + TEXT_OFFSET);
          return new FloatDimension((float)width, (float)height);
        }
      }
      return null;
    }

    private static final float FITTER_DELTA = 4;

    /**
     *  Set a horizontal fit marker to the given positon.
     *  @param g2  graphics context
     *  @param x   x coord of target position
     *  @param y   y coord of target position
     *  @param mark marker id
     */
    private void setHorizontalFitter(Graphics2D g2, float x, float y, char mark)
    {
      GeneralPath tie = new GeneralPath(GeneralPath.WIND_EVEN_ODD);
      tie.moveTo(x-FITTER_DELTA, y-FITTER_DELTA);
      tie.lineTo(x+FITTER_DELTA, y+FITTER_DELTA);
      tie.lineTo(x+FITTER_DELTA, y-FITTER_DELTA);
      tie.lineTo(x-FITTER_DELTA, y+FITTER_DELTA);
      tie.closePath();
      g2.fill(tie);
      g2.drawString(Character.toString(mark), x-FITTER_DELTA-8, y+FITTER_DELTA);
      g2.drawString(Character.toString(mark), x+FITTER_DELTA+1, y+FITTER_DELTA);
    }

    /**
     *  Set a vertical fit marker to the given positon.
     *  @param g2  graphics context
     *  @param x   x coord of target position
     *  @param y   y coord of target position
     *  @param mark marker id
     */
    private void setVerticalFitter(Graphics2D g2, float x, float y, char mark)
    {
      GeneralPath tie = new GeneralPath(GeneralPath.WIND_EVEN_ODD);
      tie.moveTo(x-FITTER_DELTA, y-FITTER_DELTA);
      tie.lineTo(x+FITTER_DELTA, y+FITTER_DELTA);
      tie.lineTo(x-FITTER_DELTA, y+FITTER_DELTA);
      tie.lineTo(x+FITTER_DELTA, y-FITTER_DELTA);
      tie.closePath();
      g2.fill(tie);
      g2.drawString(Character.toString(mark), x-FITTER_DELTA, y-FITTER_DELTA-3);
      g2.drawString(Character.toString(mark), x-FITTER_DELTA, y+FITTER_DELTA+9);
    }

    /**
     *  Print the maze to the given context.
     *  @param graphics   graphics print context
     *  @param pageFormat page format description
     *  @param pageIndex  page index
     *  @return either {@link Printable#PAGE_EXISTS} if the page was successfully rendered
     *          or  {@link Printable#NO_SUCH_PAGE} if the page index is invalid
     *  @throws PrinterException on printer errors
     */
    public int print(Graphics graphics, PageFormat pageFormat, int pageIndex) throws PrinterException
    {
      final Graphics2D g2 = (Graphics2D)graphics;
      final String info = printProperties.isPrintInfo() ?  maze.getInfo(true)  :  null;
      final FloatDimension dim = preparePrintGraphics(g2, pageFormat, pageIndex, info);

      if (dim != null) {
        g2.scale(dim.width/AbstractBasicMaze.BOX_SIZE,
                 dim.height/AbstractBasicMaze.BOX_SIZE);
        maze.draw(new Graphics2DMazePainter(g2), printProperties);

        return Printable.PAGE_EXISTS;
      }
      else {
        return Printable.NO_SUCH_PAGE;
      }
    }
  }

  /**
   *  Calculate a proportion between two values.
   *  A proportion is always greater or equal one.
   *  @param value1 positive value
   *  @param value2 positive value
   *  @return the proportion of the two values
   */
  private static double proportion(double value1, double value2)
  {
    assert(value1 > 0);
    assert(value2 > 0);
    return value1 > value2  ?  value1/value2  :  value2/value1;
  }

  /**
   *  Create a maze canvas.
   *  @param maze  the maze to display
   *  @param drawProperties maze paint properties for drawing
   *  @param printProperties maze paint properties for printing
   */
  public MazeCanvas(AbstractBasicMaze maze, MazePaintProperties drawProperties, MazePrintProperties printProperties)
  {
    this.maze = maze;
    this.drawProperties  = drawProperties;
    this.printProperties = printProperties;
    setMinimumSize(new Dimension(100, 100));
    setDoubleBuffered(false);

    final PropertyChangeListener propertyChangeListener = new PropertyChangeListener() {
      public void propertyChange(PropertyChangeEvent evt)
      {
        redraw();
      }
    };
    drawProperties.addPropertyChangeListener(propertyChangeListener);
    maze.addPropertyChangeListener(propertyChangeListener);

    setLayout(cardLayout);

    mazeComponent = new MazeComponent();
    add(mazeComponent, STANDARD_VIEW);

    Box progressBox = Box.createHorizontalBox();
    add(progressBox, PROGRESS_VIEW);

    progressBox.add(Box.createHorizontalGlue());
    Box innerColumn = Box.createVerticalBox();
    progressBox.add(innerColumn);
    progressBox.add(Box.createHorizontalGlue());

    innerColumn.add(Box.createVerticalGlue());
    noteLabel = new JLabel("");
    innerColumn.add(noteLabel);
    progressBar = new JProgressBar();
    progressBar.setStringPainted(true);
    innerColumn.add(progressBar);
    abortButton = new JButton(I18n.getString(RESOURCE_BUTTON_ABORT));
    innerColumn.add(abortButton);
    abortButton.addActionListener(new ActionListener()
    {
      public void actionPerformed(ActionEvent e)
      {
        synchronized(MazeCanvas.this) {
          abortAction = true;
          abortButton.setEnabled(false);
        }
      }
    });

    innerColumn.add(Box.createVerticalGlue());

    maze.setProgressShower(this);
    maze.addMazeFinishedListener(this);
  }

  /**
   *  Get the maze which is displayed.
   *  @return the maze
   */
  public AbstractBasicMaze getMaze()
  {
    return maze;
  }

  /**
   *  Called during maze creation when the creation starts.
   *  This method waits a given amount of time and than displays
   *  a progress indicator instead of the maze.
   *  @param note     note to display with the progress indicator
   *  @param maxValue maximum value for progress
   */
  public synchronized void start(final String note, final int maxValue)
  {

    abortAction = false;
    SwingUtilities.invokeLater(new Runnable() {
      public void run()
      {
        setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
        abortButton.setEnabled(true);
        abortButton.setCursor(Cursor.getDefaultCursor());
        noteLabel.setText(note);
        progressBar.setMaximum(maxValue);
        progressBar.setValue(0);
      }
    });
    if (delayThread == null) {
      delayThread = new Thread(new Runnable() {
        public void run()
        {
          try {
            Thread.sleep(500);
          } catch (InterruptedException e) {
          }
          //System.out.println("Inserting show of Progress View: "+System.currentTimeMillis());
          SwingUtilities.invokeLater(new Runnable() {
            public void run()
            {
              //System.out.println("Before sync:                     "+System.currentTimeMillis());
              synchronized(MazeCanvas.this) {
                //System.out.println("Thread is null? "+(delayThread==null?"yes":"no ")+"              "+System.currentTimeMillis());
                if (delayThread != null) {
                  cardLayout.show(MazeCanvas.this, PROGRESS_VIEW);
                  delayThread = null;
                  //System.out.println("Progress View:                   "+System.currentTimeMillis());
                }
              }
            }
          });
        }
      });
      delayThread.start();
    }
  }

  /**
   *  Display a progress value.
   *  @param value  progress value
   *  @return <code>true</code> if the user requested an abort,
   *          otherwise <code>false</code>
   */
  public synchronized boolean setProgress(final int value)
  {
    SwingUtilities.invokeLater(new Runnable() {
      public void run()
      {
        progressBar.setValue(value);
      }
    });
    return abortAction;
  }

  /**
   *  End the maze creation.
   *  This hides the progress indicator if necessary.
   */
  public synchronized void end()
  {
    abortAction = false;
    delayThread = null;
    SwingUtilities.invokeLater(new Runnable() {
      public void run()
      {
        cardLayout.show(MazeCanvas.this, STANDARD_VIEW);
        setCursor(Cursor.getDefaultCursor());
        //System.out.println("Standard View:                   "+System.currentTimeMillis());
      }
    });
  }

  /**
   *  Print the maze to the given print graphics.
   *  @param graphics    print graphics
   *  @param pageFormat  page format description
   *  @param pageIndex   page index
   *  @return either {@link Printable#PAGE_EXISTS} if the page was successfully rendered
   *          or  {@link Printable#NO_SUCH_PAGE} if the page index is invalid
   *  @throws PrinterException on printer errors
   */
  public synchronized int print(Graphics graphics, PageFormat pageFormat, int pageIndex) throws PrinterException
  {
    return mazeComponent.print(graphics, pageFormat, pageIndex);
  }

  /**
   *  Repaint the background buffer.
   */
  public void redraw()
  {
    expliciteRepaint = true;
    repaint(50);
  }

  /**
   * Called if the maze creation is finished.
   *
   * @param maze      maze which creation is finished
   * @param exception if not <code>null</code> this exception occured during maze creation
   */
  public void finished(Maze maze, Throwable exception)
  {
    if (exception != null) {
      JOptionPane.showMessageDialog(this,
                                    I18n.format(MESSAGE_CREATION_EXCEPTION,
                                                exception.getClass(),
                                                exception.getMessage(),
                                                exception.toString()),
                                    I18n.getString(TITLE_CREATION_EXCEPTION),
                                    JOptionPane.ERROR_MESSAGE);
    }
    repaint();
  }

  /**
   *  Paint to a graphics context.
   *  @param g2    graphics context
   *  @param size  assume size of target
   *  @param propertiesProvider paint properties
   */
  public void paintToGraphics(Graphics2D g2,
                              Dimension size,
                              MazePaintPropertiesProvider propertiesProvider)
  {
    Insets insets = maze.getInsets(propertiesProvider, size.width/AbstractBasicMaze.BOX_SIZE);
    float width  = size.width - insets.left - insets.right;
    float height = size.height - insets.top - insets.bottom;
    if (width/height > maze.getPreferredAspectRatio()) {
      width = maze.getPreferredAspectRatio()*height;
    }
    else {
      height = (int)(width/maze.getPreferredAspectRatio());
    }
    g2.setBackground(new Color(0, 0, 0, 0));
    g2.clearRect(0, 0, size.width, size.height);
    g2.setRenderingHint(RenderingHints.KEY_FRACTIONALMETRICS,
                         RenderingHints.VALUE_FRACTIONALMETRICS_ON);
    g2.setRenderingHint(RenderingHints.KEY_RENDERING,
                         RenderingHints.VALUE_RENDER_QUALITY);
    g2.setRenderingHint(RenderingHints.KEY_ANTIALIASING,
                         RenderingHints.VALUE_ANTIALIAS_ON);
    g2.translate((size.width-width)/2 + insets.left - insets.right,
                  (size.height-height)/2 + insets.top - insets.bottom);
    g2.scale(width/AbstractBasicMaze.BOX_SIZE,
              height/AbstractBasicMaze.BOX_SIZE);
    maze.draw(new Graphics2DMazePainter(g2), propertiesProvider);
  }


}
