// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: Maze.java,v $
// History:	       Revision 1.6  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.5  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.4  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.3  2005/04/19 22:15:14  rammi
// History:	       Fixed some Thread problems
// History:	
// History:	       Revision 1.2  2004/10/31 15:07:05  rammi
// History:	       Changed drawing to be always in BOX_SIZE
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import java.awt.*;
import java.awt.geom.Point2D;

/**
 *  Basic interface of a Maze.
 *
 *  A maze is defined by the following items:
 *  <ul>
 *   <li>its cells: see {@link de.caff.maze.Maze#getCells()}</it>
 *   <li>its clear state (all cells are unconnected): see {@link de.caff.maze.Maze#reset()}</it>
 *   <li>how to draw itself: see {@link Maze#draw(MazePainter, MazePaintPropertiesProvider)}</li>
 *   <li>a start and an end cell: see {@link Maze#setWayStart(MazeCell)} and {@link Maze#setWayEnd(MazeCell)}</li>
 *   <li>a way to create a purely random maze (see {@link de.caff.maze.Maze#createMaze()}) or a special maze
 *       defined by a seed value (see {@link Maze#createMaze(long)}</li>
 *   <li>a preferred aspect ratio: see {@link de.caff.maze.Maze#getPreferredAspectRatio()}</li>
 *   <li>how to map a screen position to a cell: see {@link Maze#getCellAt(java.awt.geom.Point2D)}</li>
 *   <li>how to map an integer id to a cell: see {@link Maze#getCellByID(int)}</li>
 *   <li>how to store and load itself: see {@link Maze#storePersistentData(DataStorage)} and
 *                                         {@link Maze#loadPersistentData(DataStorage)}</li>
 *   <li>how to manage listeners which are interested to know when the creation of a maze has finished:
 *       {@link Maze#addMazeFinishedListener(MazeFinishedListener)} and
 *       {@link Maze#removeMazeFinishedListener(MazeFinishedListener)}</li>
 *  </ul>
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.6 $
 */
public interface Maze
{
  /**
   *  Get all cells of this maze.
   *  @return maze cells
   */
  public MazeCell[] getCells();

  /**
   *   Reset all borders and ways.
   *   After this all cells are unconnected.
   */
  void reset();

  /**
   *  Draw this maze.
   *  @param painter  painter to draw to
   *  @param properties access to properties for drawing (colors etc)
   */
  void draw(MazePainter painter, MazePaintPropertiesProvider properties);

  /**
   *  Set the start cell of the way.
   *  @param start start cell
   */
  void setWayStart(MazeCell start);

  /**
   *  Set the end cell of the way.
   *  @param end end cell
   */
  void setWayEnd(MazeCell end);

  /**
   *  Create a purely random maze.
   */
  void createMaze();

  /**
   *  Create a maze from a given seed.
   *  @param seed random seed
   */
  void createMaze(long seed);

  /**
   *  Get the preferred aspect ratio of this maze.
   *  @return aspect ratio (width/height)
   */
  float getPreferredAspectRatio();

  /**
   *  Get the necessary insets depending on the paint properties.
   *  Usually the insets are necessary to allow for the thick border line to be drawn completely.
   *  @param properties paint properties
   *  @param scaling scaling used when painting    
   *  @return insets
   */
  Insets getInsets(MazePaintPropertiesProvider properties, float scaling);

  /**
   * Get the cell at a given position.
   * @param position cell position
   * @return cell at the given position or <code>null</code> if there is no cell at that position
   */
  public MazeCell getCellAt(Point2D position);

  /**
   *  Get the cell with the given id.
   *  @param id cell id
   *  @return the cell with the given id or <code>null</code> if there is no such cell
   *  @see MazeCell#getID()
   */
  public MazeCell getCellByID(int id);

  /**
   *  Load extra data defining the maze from the system access.
   *  @param systemAccess system access
   */
  public void loadPersistentData(DataStorage systemAccess);

  /**
   *  Store extra data defining the maze to the system access.
   *  @param systemAccess system access
   */
  public void storePersistentData(DataStorage systemAccess);

  /**
   *  Add a maze finished listener which is called when the maze (re)creation is finished.
   *  @param listener listener to add
   */
  public void addMazeFinishedListener(MazeFinishedListener listener);

  /**
   *  Remove a maze finished listener which was called when the maze (re)creation is finished.
   *  @param listener listener to remove
   */
  public void removeMazeFinishedListener(MazeFinishedListener listener);


}
