// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: JnlpSystemAccess.java,v $
// History:	       Revision 1.8  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.7  2009/09/24 17:32:20  rammi
// History:	       Finetuning dialog placement
// History:
// History:	       Revision 1.6  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.5  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.4  2004/11/25 11:57:31  rammi
// History:	       Fixed bug in openeing FileSaveService
// History:	
// History:	       Revision 1.3  2004/11/02 20:53:53  rammi
// History:	       Added keyboard shortcuts etc via actions
// History:	
// History:	       Revision 1.2  2004/10/31 22:07:43  rammi
// History:	       Fixed problem with default ways on first execution
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import de.caff.gimmix.I18n;

import javax.jnlp.*;
import javax.swing.*;
import java.awt.*;
import java.awt.print.Printable;
import java.io.*;
import java.net.URL;
import java.util.Properties;

/**
 *  System access when running under Java WebStart.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.8 $
 */
class JnlpSystemAccess
        implements SystemAccess
{

  /**
   *  Default disk space in JNLP.
   *  If you try to require more the user will be asked.
   *  For this simple application it's more than enough.
   */
  public static long MAX_SPACE = 256 * 1024;

  /** The properties holding the persistent values. */
  private Properties properties = new Properties();

  /**
   *  Constructor.
   */
  public JnlpSystemAccess()
  {
    try {
      PersistenceService ps = (PersistenceService)ServiceManager.lookup("javax.jnlp.PersistenceService");
      FileContents fileContents = ps.get(getResourceUrl());
      properties.load(fileContents.getInputStream());
    } catch (Exception e) {
    }
  }

  /**
   * Print the given printable.
   *
   * @param printable printable to be printed
   * @return <code>true</code>: printing was successful,
   *         <code>false</code>: printing failed or was canceled
   */
  public boolean print(Printable printable)
  {
    try {
      PrintService printService = (PrintService)ServiceManager.lookup("javax.jnlp.PrintService");
      return printService.print(printable);
    } catch (UnavailableServiceException x) {
      JOptionPane.showConfirmDialog(null,
                                    I18n.format(MultiMazeControlPanel.RESOURCE_PRINT_ERROR, x.getMessage()),
                                    MultiMazeControlPanel.RESOURCE_PRINT_ERROR+MazeResourceBundle.TITLE,
                                    JOptionPane.ERROR_MESSAGE);
    }
    return false;
  }

  /**
   *  Get the URL of the persistent resource.
   *  @return  URL
   *  @exception Exception on errors
   */
  private static URL getResourceUrl() throws Exception
  {
    BasicService basic = (BasicService)ServiceManager.lookup("javax.jnlp.BasicService");
    // using codebase to allow relocation of the application
    return basic.getCodeBase();

    /*
     * The following would be better because different users could
     * use different configs but we are not allowed to read the
     * user.name property. Wow, that's what I call secure!
     * This is not a problem on systems which really know about users
     * (like unix where every user gets its own ~/.javaws directory)
     * but may give problems on other platforms (like some Windows
     * game systems).
     */
    // return new URL(basic.getCodeBase(), System.getProperty("user.name"));
  }

  /**
   * Get a persistent string value.
   * A persistent string is a string which should survive the
   * end of the program.
   *
   * @param key          key for the string
   * @param defaultValue default value which is returned if the string is not defined
   * @return the string value defined by the key or the default value
   */
  public String getString(String key, String defaultValue)
  {
    return properties.getProperty(key, defaultValue);
  }

  /**
   * Set a persistent string.
   * The string should be stored so it may
   * be read again in this or a later session.
   *
   * @param key   key for the string
   * @param value the string value to store
   */
  public void setString(String key, String value)
  {
    properties.setProperty(key, value);
  }

  /**
   * Get a persistent boolean value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the boolean value defined by the key of the default value
   */
  public boolean getBoolean(String key, boolean defaultValue)
  {
    return Stringizer.booleanFromString(getString(key, null), defaultValue);
  }

  /**
   * Set a persistent boolean.
   *
   * @param key   key for the boolean
   * @param value value of the boolean
   */
  public void setBoolean(String key, boolean value)
  {
    setString(key, Stringizer.booleanToString(value));
  }

  /**
   * Get a persistent color value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the color value deg´fined by the key of the default value
   */
  public Color getColor(String key, Color defaultValue)
  {
    return Stringizer.colorFromString(getString(key, null), defaultValue);
  }

  /**
   * Set a persistent color.
   *
   * @param key   key for the color
   * @param value value of the color
   */
  public void setColor(String key, Color value)
  {
    setString(key, Stringizer.colorToString(value));
  }

  /**
   * Get a persistent integer value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the integer value defined by the key of the default value
   */
  public int getInt(String key, int defaultValue)
  {
    return Stringizer.intFromString(getString(key, null), defaultValue);
  }

  /**
   * Set a persistent integer value.
   *
   * @param key   key for the int
   * @param value value of the int
   */
  public void setInt(String key, int value)
  {
    setString(key, Stringizer.intToString(value));
  }

  /**
   * Get a persistent long integer value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the long integer value defined by the key of the default value
   */
  public long getLong(String key, long defaultValue)
  {
    return Stringizer.longFromString(getString(key, null), defaultValue);
  }

  /**
   * Set a persistent long integer value.
   *
   * @param key   key for the long int
   * @param value value of the long int
   */
  public void setLong(String key, long value)
  {
    setString(key, Stringizer.longToString(value));
  }

  /**
   * Get a persistent double value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the double value defined by the key of the default value
   */
  public double getDouble(String key, double defaultValue)
  {
    return Stringizer.doubleFromString(getString(key, null), defaultValue);
  }

  /**
   * Set a persistent double value.
   *
   * @param key   key for the int
   * @param value value of the int
   */
  public void setDouble(String key, double value)
  {
    setString(key, Stringizer.doubleToString(value));
  }

  /**
   * Is his key defined?
   *
   * @param key key to look for
   * @return <code>true</code> if the key is defined, <code>false</code> otherwise
   */
  public boolean hasKey(String key)
  {
    return properties.getProperty(key) != null;
  }

  /**
   * Store all persistent data.
   * This is called when the process is about to shutdown.
   */
  public void storePersistentData()
  {
    try {
      PersistenceService ps = (PersistenceService)ServiceManager.lookup("javax.jnlp.PersistenceService");
      // using codebase to allow relocation of the application
      URL url = getResourceUrl();

      // try to create
      try {
        ps.create(url, MAX_SPACE);
        ps.setTag(url, PersistenceService.DIRTY);
      } catch (Exception x) {
        // will throw an exception the second time it's called
      }

      FileContents fileContents = ps.get(url);
      properties.store(fileContents.getOutputStream(true),
                       "Persistent Maze data");
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

  /**
   * Get the contents of a user selected file.
   *
   * @param filetype type of file to be used
   * @return byte array or <code>null</code> if the user canceled the operation
   * @throws java.io.IOException on errors during file access
   */
  public byte[] getUserFileContent(FileType filetype) throws IOException
  {
    try {
      FileOpenService fos = (FileOpenService)ServiceManager.lookup("javax.jnlp.FileOpenService");
      final FileContents fileContents = fos.openFileDialog(".", filetype.getExtensions());
      if (fileContents != null) {
        InputStream is = fileContents.getInputStream();
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        int c;
        while ((c = is.read()) != -1) {
          bos.write(c);
        }
        is.close();
        bos.close();
        return bos.toByteArray();
      }
    } catch (UnavailableServiceException e) {
      e.printStackTrace();
    }
    return null;
  }

  /**
   * Save a byte array into a user selected file.
   *
   * @param content  the content of the file
   * @param filetype type of file to be used
   * @return <code>true</code> if the user saved the file, <code>false</code> otherwise
   * @throws java.io.IOException on errors during file access
   */
  public boolean saveUserFileContent(byte[] content, FileType filetype) throws IOException
  {
    try {
      FileSaveService fos = (FileSaveService)ServiceManager.lookup("javax.jnlp.FileSaveService");
      ByteArrayInputStream bis = new ByteArrayInputStream(content);
      final FileContents fileContents = fos.saveFileDialog(".", filetype.getExtensions(),
                                                           bis, "");
      bis.close();
      if (fileContents != null) {
        return true;
      }
    } catch (UnavailableServiceException e) {
      e.printStackTrace();
    }
    return false;
  }

  /**
   * Sets the main frame of the application.
   * This is useful for centering dialogs.
   *
   * @param frame application's main frame
   */
  public void setMainFrame(JFrame frame)
  {
    // deliberately ignored
  }
}
