// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 19:01:35 $
//
// History:	       $Log: HexagonalMaze.java,v $
// History:	       Revision 1.9  2012/06/07 19:01:35  rammi
// History:	       Fixed jdoc ref
// History:
// History:	       Revision 1.8  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.7  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.6  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.5  2004/11/02 20:53:53  rammi
// History:	       Added keyboard shortcuts etc via actions
// History:	
// History:	       Revision 1.4  2004/10/31 23:30:52  rammi
// History:	       Redraw speed-up
// History:	
// History:	       Revision 1.3  2004/10/31 15:07:05  rammi
// History:	       Changed drawing to be always in BOX_SIZE
// History:	
// History:	       Revision 1.2  2004/10/25 14:54:57  rammi
// History:	       Javadoc changes
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.GeneralPath;
import java.util.ArrayList;
import java.util.Collection;
import java.util.List;

/**
 *  Maze with hexagonal cells, while the complete maze tries to be rectangular which better fits
 *  on screen and paper.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.9 $
 */
public class HexagonalMaze
        extends AbstractBasicMaze
{
  /** The type of this maze (for storage). */
  public static final String MAZE_TYPE = "Hexagonal Maze";

  /** Property key for the settable number of horizontal cells property. */
  public static final String PROPERTY_HORIZONTAL_CELLS = "HEXAGON_MAZE:nrHorizontalCells";
  /** Property key for the settable number of vertical cells property. */
  public static final String PROPERTY_VERTICAL_CELLS   = "HEXAGON_MAZE:nrVerticalCells";

  /** Internally used bit value for direction W. */
  private static final int WEST = 1;
  /** Internally used bit value for direction E. */
  private static final int EAST = 2;
  /** Internally used bit value for direction NW. */
  private static final int NORTH_WEST  = 4;
  /** Internally used bit value for direction NE. */
  private static final int NORTH_EAST  = 8;
  /** Internally used bit value for direction SW. */
  private static final int SOUTH_WEST  = 16;
  /** Internally used bit value for direction SE. */
  private static final int SOUTH_EAST  = 32;

  /**
   *  A hexagonal cell.
   */
  private class HexagonalMazeCell extends MazeCell
  {
    /** The x coordinate of this cell. */
    private int x;
    /** The y coordinate of this cell. */
    private int y;
    /** The connections of this cell as a bitmask. */
    private int connection;
    /** The shape of this cell. */
    private Shape shape;

    /**
     *  Constructor.
     *  @param x  x coordinate
     *  @param y  y coordinate
     */
    HexagonalMazeCell(int x, int y)
    {
      this.x = x;
      this.y = y;
    }

    /**
     *  Forget all connections.
     */
    public void reset()
    {
      connection = 0;
    }

    /**
     * Get the neighbour cells of this one.
     *
     * @return neigbour cells
     */
    public MazeCell[] getNeighbours()
    {
      List<MazeCell> neighbours = new ArrayList<MazeCell>(6);
      if (x > 0) {
        neighbours.add(getCell(x-1, y));
      }
      if (x < nrHorizontal-1) {
        neighbours.add(getCell(x+1, y));
      }
      if (y > 0) {
        neighbours.add(getCell(x, y-1));
        if (y % 2 == 0) {
          if (x > 0) {
            neighbours.add(getCell(x-1, y-1));
          }
        }
        else {
          if (x < nrHorizontal-1) {
            neighbours.add(getCell(x+1, y-1));
          }
        }
      }
      if (y < nrVertical - 1) {
        neighbours.add(getCell(x, y+1));
        if (y % 2 == 0) {
          if (x > 0) {
            neighbours.add(getCell(x-1, y+1));
          }
        }
        else {
          if (x < nrHorizontal-1) {
            neighbours.add(getCell(x+1, y+1));
          }
        }
      }
      return neighbours.toArray(new MazeCell[neighbours.size()]);
    }

    /**
     *  Get the maze to which this cell belongs.
     *  @return the maze of this cell
     */
    public Maze getMaze()
    {
      return HexagonalMaze.this;
    }

    /**
     * Get the id of this cell.
     * The id has to be unique for a given geometry of a maze.
     *
     * @return unique id
     * @see Maze#getCellByID(int)
     */
    public int getID()
    {
      return y*nrHorizontal + x;
    }

    /**
     * Connect this cell to the given one.
     * The algorithm will also call the inverse method for cell.
     *
     * @param cell cell to connect to
     */
    public void connectTo(MazeCell cell)
    {
      HexagonalMazeCell rCell = (HexagonalMazeCell)cell;
      if (rCell.y == y) {
        if (rCell.x < x) {
          connection |= WEST;
        }
        else {
          connection |= EAST;
        }
      }
      else if (rCell.y < y) {
        if (y % 2 == 0) {
          if (rCell.x == x) {
            connection |= NORTH_EAST;
          }
          else {
            connection |= NORTH_WEST;
          }
        }
        else {
          if (rCell.x == x) {
            connection |= NORTH_WEST;
          }
          else {
            connection |= NORTH_EAST;
          }
        }
      }
      else {
        if (y % 2 == 0) {
          if (rCell.x == x) {
            connection |= SOUTH_EAST;
          }
          else {
            connection |= SOUTH_WEST;
          }
        }
        else {
          if (rCell.x == x) {
            connection |= SOUTH_WEST;
          }
          else {
            connection |= SOUTH_EAST;
          }
        }
      }


    }

    /**
     * Get the connected neighbour cells of this one.
     *
     * @return neigbour cells
     */
    public MazeCell[] getConnectedNeighbours()
    {
      List<MazeCell> neighbours = new ArrayList<MazeCell>(6);
      boolean even = y%2 == 0;
      if ((connection & WEST) != 0) {
        neighbours.add(getCell(x-1, y));
      }
      if ((connection & EAST) != 0) {
        neighbours.add(getCell(x+1, y));
      }
      if ((connection & NORTH_WEST) != 0) {
        neighbours.add(getCell(x-(even?1:0), y-1));
      }
      if ((connection & NORTH_EAST) != 0) {
        neighbours.add(getCell(x+(even?0:1), y-1));
      }
      if ((connection & SOUTH_WEST) != 0) {
        neighbours.add(getCell(x-(even?1:0), y+1));
      }
      if ((connection & SOUTH_EAST) != 0) {
        neighbours.add(getCell(x+(even?0:1), y+1));
      }
      return neighbours.toArray(new MazeCell[neighbours.size()]);
    }

    /**
     *  Get the connections of this cell as a bitmask of direction bits.
     *  @return bitmask of direction bits
     */
    public int getConnection()
    {
      return connection;
    }

    /**
     *  Get the shape of this cell in the current display.
     *  @return maze cell shape
     */
    public Shape getShape()
    {
      return shape;
    }

    /**
     *  Set the shape of this cell in the current display.
     *  @param shape maze cell shape
     */
    public void setShape(Shape shape)
    {
      this.shape = shape;
    }
  }
  /** Number of horizontal cells (maze width). */
  private int nrHorizontal;
  /** Number of vertical cells (maze height). */
  private int nrVertical;
  /** The cells of this maze. */
  private HexagonalMazeCell[] cells;

  /**
   *  Constructor.
   *  @param horizontalCells  number of horizontal cells (maze width)
   *  @param verticalCells    number of vertical cells (maze height)
   */
  public HexagonalMaze(int horizontalCells, int verticalCells)
  {
    initialize(horizontalCells, verticalCells);
  }

  /**
   *  Initialize this maze.
   *  @param horizontalCells  number of horizontal cells (maze width)
   *  @param verticalCells    number of vertical cells (maze height)
   */
  private void initialize(int horizontalCells, int verticalCells)
  {
    boolean differs = horizontalCells != nrHorizontal  ||  verticalCells != nrVertical;

    if (differs) {
      if (this.nrHorizontal != horizontalCells) {
        int oldValue = this.nrHorizontal;
        this.nrHorizontal = horizontalCells;
        firePropertyChange(PROPERTY_HORIZONTAL_CELLS,
                           new Integer(oldValue),
                           new Integer(horizontalCells));
      }
      if (this.nrVertical != verticalCells) {
        int oldValue = this.nrVertical;
        this.nrVertical = verticalCells;
        firePropertyChange(PROPERTY_VERTICAL_CELLS,
                                new Integer(oldValue),
                                new Integer(verticalCells));
      }
      cells = new HexagonalMazeCell[nrHorizontal*nrVertical];
      for (int x = 0;  x < nrHorizontal;  ++x) {
        for (int y = 0;  y < nrVertical;  ++y) {
          cells[x*nrVertical+y] = new HexagonalMazeCell(x, y);
        }
      }
      createShapes();
      setDefaultWayPoints();
    }
  }

  /**
   * Get a deep copy of this maze geometry.
   *
   * @return deep copy
   */
  protected AbstractBasicMaze getGeometryClone()
  {
    return new HexagonalMaze(nrHorizontal, nrVertical);  
  }

  /**
   *  Get a internally used string describing the maze type.
   *  @return maze type
   */
  public String getMazeType()
  {
    return MAZE_TYPE;
  }

  /**
   *  Resets the internal data.
   *  Should be overwritten, and overwritung methods should call this.
   */
  @Override public void reset()
  {
    for (int c = 0;  c < cells.length;  ++c) {
      cells[c].reset();
    }
    super.reset();
  }

  /**
   * Get the cells of this maze.
   * @return the cells of this maze
   */
  public MazeCell[] getCells()
  {
    MazeCell[] mCells = new MazeCell[cells.length];
    System.arraycopy(cells, 0, mCells, 0, cells.length);
    return mCells;
  }

  /**
   *  Get the cell in column x and row y.
   *  @param x column id
   *  @param y row id
   *  @return cell at columns x and row y
   */
  private HexagonalMazeCell getCell(int x, int y)
  {
    return cells[x*nrVertical+y];
  }

  /** Border shape. */
  private Shape outShape;

  /**
   *  Create the cell shapes.
   */
  private void createShapes()
  {
    final float width  = BOX_SIZE;
    final float height = BOX_SIZE;

    final float deltaX = width/(nrHorizontal+(nrVertical>0 ? 0.5f : 0));
    final float deltaY = 4*height/(3*nrVertical+1);

    outShape = getOuterBounds(deltaX, deltaY);

    for (int y = 0;  y < nrVertical;  ++y) {
      final float xOffset = (y%2==1) ? deltaX/2 : 0;
      final float yOffset = 3*y*deltaY/4;
      for (int x = 0;  x < nrHorizontal;  ++x) {
        final HexagonalMazeCell cell = getCell(x, y);
        cell.setShape(getHexagonalShape(x*deltaX+xOffset, yOffset, deltaX, deltaY));
      }
    }
  }

  /**
   * Get the cell with the given id.
   *
   * @param id cell id
   * @return the cell with the given id or <code>null</code> if there is no such cell
   * @see MazeCell#getID()
   */
  public MazeCell getCellByID(int id)
  {
    final int x = id%nrHorizontal;
    final int y = id/nrHorizontal;
    return (y >= 0  &&   y < nrVertical  &&  x >= 0   &&   x < nrHorizontal)  ?
            getCell(x, y)  :
            null;
  }

  /**
   * Do the actual drawing.
   * The call to this method is embedded in the the calls to
   * {@link de.caff.maze.MazePainter#startPaintingMaze(Maze)} and
   * {@link de.caff.maze.MazePainter#endPaintingMaze()}.
   *
   * @param painter    painter to draw to
   * @param properties access to properties for drawing (colors etc)
   */
  @Override
  protected void doDraw(MazePainter painter, MazePaintPropertiesProvider properties)
  {
    final float width  = BOX_SIZE;
    final float height = BOX_SIZE;

    final float deltaX = width/(nrHorizontal+(nrVertical>0 ? 0.5f : 0));
    final float deltaY = 4*height/(3*nrVertical+1);

    drawBackgroundAndWay(painter, properties);

    painter.setStroke(new BasicStroke(deltaX / 12, BasicStroke.CAP_ROUND, BasicStroke.JOIN_MITER));
    if (properties.isShowingCellBorders()) {
      try {
        painter.startPainting(MazePainter.PaintObjectType.CellBorders);
        painter.setPaint(properties.getCellBorderPaint());
        for (int y = 0;  y < nrVertical;  ++y) {
          final float xOffset = (y%2==1) ? deltaX/2 : 0;
          final float yOffset = 3*y*deltaY/4;
          for (int x = 0;  x < nrHorizontal;  ++x) {
            final HexagonalMazeCell cell = getCell(x, y);
            final int connection = cell.getConnection();
            if ((connection & WEST) != 0) {
              painter.drawLine(x * deltaX + xOffset, 3 * deltaY / 4 + yOffset,
                               x * deltaX + xOffset, deltaY / 4 + yOffset);
            }
            if ((connection & NORTH_WEST) != 0) {
              painter.drawLine(x * deltaX + xOffset, deltaY / 4 + yOffset,
                               x * deltaX + deltaX / 2 + xOffset, yOffset);
            }
            if ((connection & NORTH_EAST) != 0) {
              painter.drawLine(x * deltaX + deltaX / 2 + xOffset, yOffset,
                               x * deltaX + deltaX + xOffset, deltaY / 4 + yOffset);
            }
          }
        }
      } finally {
        painter.endPainting(MazePainter.PaintObjectType.CellBorders);
      }
    }

    try {
      painter.startPainting(MazePainter.PaintObjectType.InnerWalls);
      painter.setPaint(properties.getInnerWallsPaint());
      for (int y = 0;  y < nrVertical;  ++y) {
        final float xOffset = (y%2==1) ? deltaX/2 : 0;
        final float yOffset = 3*y*deltaY/4;
        for (int x = 0;  x < nrHorizontal;  ++x) {
          final HexagonalMazeCell cell = getCell(x, y);
          final int connection = cell.getConnection();
          if ((connection & WEST) == 0) {
            painter.drawLine(x * deltaX + xOffset, 3 * deltaY / 4 + yOffset,
                             x * deltaX + xOffset, deltaY / 4 + yOffset);
          }
          if ((connection & NORTH_WEST) == 0) {
            painter.drawLine(x * deltaX + xOffset, deltaY / 4 + yOffset,
                             x * deltaX + deltaX / 2 + xOffset, yOffset);
          }
          if ((connection & NORTH_EAST) == 0) {
            painter.drawLine(x * deltaX + deltaX / 2 + xOffset, yOffset,
                             x * deltaX + deltaX + xOffset, deltaY / 4 + yOffset);
          }
        }
      }
    } finally {
      painter.endPainting(MazePainter.PaintObjectType.InnerWalls);
    }

    try {
      painter.startPainting(MazePainter.PaintObjectType.OuterWalls);
      painter.setPaint(properties.getOuterWallPaint());
      painter.setStroke(new BasicStroke(deltaX / 6));
      painter.draw(outShape);
    } finally {
      painter.endPainting(MazePainter.PaintObjectType.OuterWalls);
    }
  }

  /**
   *  Get the preferred aspect ratio of this maze.
   *  @return aspect ratio (width/height)
   */
  public float getPreferredAspectRatio()
  {
    return (float)(2*Math.sqrt(3)*nrHorizontal/(3*nrVertical+1));
  }

  /**
   * Get the necessary insets depending on the paint properties.
   * Usually the insets are necessary to allow for the thick border line to be drawn completely.
   *
   * @param properties paint properties
   * @param scaling    scaling used when painting
   * @return insets
   */
  public Insets getInsets(MazePaintPropertiesProvider properties, float scaling)
  {
    final float delta = scaling * BOX_SIZE/(nrHorizontal+(nrVertical>0 ? 0.5f : 0));
    int tb = (int)Math.ceil(delta/12);
    int lr = (int)Math.ceil(delta/12/Math.cos(Math.toRadians(30)));
    return new Insets(tb, lr, tb, lr);
  }

  /**
   *  Create a hexagonal shape.
   *  @param startX  start position (x)
   *  @param startY  start position (y)
   *  @param deltaX  cell width
   *  @param deltaY  cell height
   *  @return shape
   */
  private static Shape getHexagonalShape(float startX, float startY, float deltaX, float deltaY)
  {
    GeneralPath shape = new GeneralPath();
    shape.moveTo(startX+deltaX/2, startY);
    shape.lineTo(startX, startY+deltaY/4);
    shape.lineTo(startX, startY+3*deltaY/4);
    shape.lineTo(startX+deltaX/2, startY+deltaY);
    shape.lineTo(startX+deltaX, startY+3*deltaY/4);
    shape.lineTo(startX+deltaX, startY+deltaY/4);
    shape.closePath();
    return shape;
  }

  /**
   * Create the border around the maze.
   * @param deltaX cell width
   * @param deltaY cell height
   * @return border shape
   */
  private Shape getOuterBounds(float deltaX, float deltaY)
  {
    final boolean even = nrVertical%2 == 0;
    GeneralPath shape = new GeneralPath();
    shape.moveTo(0, deltaY/4);
    for (int y = 0;  y < nrVertical-1;  ++y) {
      if (y%2 == 0) {
        shape.lineTo(0, (3*y+3)*deltaY/4);
        shape.lineTo(deltaX/2, (3*y+4)*deltaY/4);
      }
      else {
        shape.lineTo(deltaX/2, (3*y+3)*deltaY/4);
        shape.lineTo(0, (3*y+4)*deltaY/4);
      }
    }
    if (even) {
      shape.lineTo(deltaX/2, (3*nrVertical)*deltaY/4);
    }
    else {
      shape.lineTo(0, (3*nrVertical)*deltaY/4);
    }
    final float xOffset = even ? deltaX/2 : 0;
    for (int x = 0;  x < nrHorizontal;  ++x) {
      shape.lineTo((x+0.5f)*deltaX+xOffset, (3*nrVertical+1)*deltaY/4);
      shape.lineTo((x+1.0f)*deltaX+xOffset, 3*nrVertical*deltaY/4);
    }
    for (int y = nrVertical-1;  y > 0;  --y) {
      if (y % 2 == 0) {
        shape.lineTo(nrHorizontal*deltaX, (3*y+1)*deltaY/4);
        shape.lineTo((nrHorizontal+0.5f)*deltaX, (3*y)*deltaY/4);
      }
      else {
        shape.lineTo((nrHorizontal+0.5f)*deltaX, (3*y+1)*deltaY/4);
        shape.lineTo(nrHorizontal*deltaX, (3*y)*deltaY/4);
      }
    }
    shape.lineTo(nrHorizontal*deltaX, deltaY/4);
    for (int x = nrHorizontal-1;  x > 0;  --x) {
      shape.lineTo((x+0.5f)*deltaX, 0);
      shape.lineTo(x*deltaX, deltaY/4);
    }
    shape.lineTo(deltaX/2, 0);
    shape.closePath();
    return shape;
  }

  /**
   *  Set some useful default way points.
   */
  public void setDefaultWayPoints()
  {
    setWayPoints(getCell(0, 0), getCell(nrHorizontal-1, nrVertical-1));
  }

  /**
   *  Set the number of horizontal cells.
   *  @param nrHorizontal number of horizontal cells
   */
  private void setNrHorizontal(int nrHorizontal)
  {
    if (this.nrHorizontal != nrHorizontal) {
      initialize(nrHorizontal, nrVertical);
      recreateMaze();
    }
  }

  /**
   *  Set ehe number of vertical cells.
   *  @param nrVertical number of vertical cells
   */
  private void setNrVertical(int nrVertical)
  {
    if (this.nrVertical != nrVertical) {
      initialize(nrHorizontal, nrVertical);
      recreateMaze();
    }
  }

  /**
   *  Property information handling the number of horizontal cells.
   */
  private IntegerDelayedPropertyInformation numberHorizontalPropertySetter = new IntegerDelayedPropertyInformation(PROPERTY_HORIZONTAL_CELLS, 10, 1000, 1) {
    protected int getMazeValue()
    {
      return nrHorizontal;
    }

    protected void setMazeValue(int value)
    {
      setNrHorizontal(value);
    }
  };

  /**
   *  Property information handling the number of vertical cells.
   */
  private IntegerDelayedPropertyInformation numberVerticalPropertySetter = new IntegerDelayedPropertyInformation(PROPERTY_VERTICAL_CELLS, 10, 1000, 1) {
    protected int getMazeValue()
    {
      return nrVertical;
    }

    protected void setMazeValue(int value)
    {
      setNrVertical(value);
    }
  };

  /**
   *  Get the property setters and displays for this maze.
   *
   *  Overriding classes should include this collection first.
   *  @return colletion of property setters
   */
  @Override
  public Collection<PropertyInformation> getPropertyInformations()
  {
    Collection<PropertyInformation> setters = super.getPropertyInformations();

    setters.add(numberHorizontalPropertySetter);
    setters.add(numberVerticalPropertySetter);

    return setters;
  }

  /**
   *  This is called during the call of {@link #setFromSetters()}
   *  and should be used to recreate the geometry of the maze from
   *  the setters of the geometric properties.
   */
  protected void recreateFromDelayedSetters()
  {
    initialize(numberHorizontalPropertySetter.getValue(),
               numberVerticalPropertySetter.getValue());
  }

  /**
   *  Get the borders of the maze as a shape.
   *  @return outer border
   */
  protected Shape getOuterBorder()
  {
    return outShape;
  }

  /**
   * Load extra data defining the maze from the system access.
   *
   * @param systemAccess system access
   */
  public void loadPersistentData(DataStorage systemAccess)
  {
    initialize(systemAccess.getInt(PROPERTY_HORIZONTAL_CELLS, nrHorizontal),
               systemAccess.getInt(PROPERTY_VERTICAL_CELLS, nrVertical));
    loadSeedWayAndVersion(systemAccess, MAZE_TYPE);
  }

  /**
   * Store extra data defining the maze to the system access.
   *
   * @param systemAccess system access
   */
  public void storePersistentData(DataStorage systemAccess)
  {
    systemAccess.setInt(PROPERTY_HORIZONTAL_CELLS, nrHorizontal);
    systemAccess.setInt(PROPERTY_VERTICAL_CELLS, nrVertical);
    storeSeedWayAndVersion(systemAccess, MAZE_TYPE);
  }

  /**
   *  Test code.
   *  @param args unused
   */
  public static void main(String[] args)
  {
    JFrame frame = new JFrame("TEST");
    final int width  = arg2int(args, 0, 20);
    final int height = arg2int(args, 1, 15);
    final HexagonalMaze maze = new HexagonalMaze(width, height);
    final MazePaintProperties properties = MazePaintProperties.getDrawMazePaintProperties(null);
    final MazeCanvas drawarea = new MazeCanvas(maze, properties,
                                              MazePrintProperties.getPrintMazePaintProperties(null));
    maze.createMaze();
    maze.setDefaultWayPoints();
    properties.setShowingSolution(true);
    properties.setShowingCellBorders(false);
    drawarea.addMouseListener(new MouseAdapter() {
      @Override public void mouseClicked(MouseEvent e)
      {
        try {
          drawarea.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
          maze.createMaze();
        } finally {
          drawarea.redraw();
          drawarea.setCursor(Cursor.getDefaultCursor());
        }
      }
    });
    frame.getContentPane().add(drawarea);
    frame.setSize(800, 600);

    frame.setVisible(true);

    frame.addWindowListener(new WindowAdapter() {
      @Override public void windowClosing(WindowEvent e)
      {
        System.exit(0);
      }
    });

  }
}
