// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: FileDataStorage.java,v $
// History:	       Revision 1.4  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.3  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.2  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.1.1.1  2004/10/25 14:47:55  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import javax.imageio.ImageIO;
import java.awt.image.RenderedImage;
import java.io.*;
import java.util.Map;

/**
 *  Data storage using a system access.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.4 $
 */
public class FileDataStorage
        extends TemporaryDataStorage
{
  /** System access used for access of the underlying file system. */
  private final SystemAccess systemAccess;

  /**
   *  Constructor.
   *  @param systemAccess system access used for access of the underlying file system
   */
  public FileDataStorage(SystemAccess systemAccess)
  {
    this.systemAccess = systemAccess;
  }

  /**
   *  Load data.
   *  @return <code>true</code> if data could be loaded successfully, otherwise <code>false</code>
   *  @throws Exception   on access and i/o errors
   */
  @SuppressWarnings("unchecked")
  boolean load() throws Exception
  {
    byte[] content = systemAccess.getUserFileContent(SystemAccess.MAZE_FILE_TYPE);
    if (content != null) {
      ByteArrayInputStream bis = new ByteArrayInputStream(content);
      ObjectInputStream ois = new ObjectInputStream(bis);
      try {
        persistentData = (Map<String,Object>)ois.readObject();
        return true;
      } finally {
        ois.close();
      }
    }
    return false;
  }

  /**
   *  Save data.
   *  @return <code>true</code> if data could be saved successfully, otherwise <code>false</code>
   *  @throws Exception  on access and i/o errors
   */
  boolean save() throws Exception
  {
    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    ObjectOutputStream oos = new ObjectOutputStream(bos);
    oos.writeObject(persistentData);
    oos.close();
    return systemAccess.saveUserFileContent(bos.toByteArray(), SystemAccess.MAZE_FILE_TYPE);
  }

  /**
   *  Save an image.
   *  @param image image to save
   *  @return <code>true</code> if the image could be saved successfully, otherwise <code>false</code>
   *  @throws Exception  on access and i/o errors
   */
  boolean saveImage(RenderedImage image) throws Exception
  {
    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    ImageIO.write(image, SystemAccess.IMAGE_FILE_TYPE.getExtensions()[0], bos);
    bos.close();
    return systemAccess.saveUserFileContent(bos.toByteArray(), SystemAccess.IMAGE_FILE_TYPE);
  }

  /**
   * Save using a generic output painter.
   * @param maze            maze to save
   * @param properties      paint properties to use
   * @param fileType        file type to save to
   * @param painterCreator  creator for the output maze painter
   * @return <code>true</code> if the created file could be saved successfully, otherwise <code>false</code>
   * @throws Exception on access and i/o errors
   */
  boolean saveToPainter(Maze maze,
                        MazePaintPropertiesProvider properties,
                        SystemAccess.FileType fileType,
                        OutputMazePainterCreator painterCreator) throws Exception
  {
    ByteArrayOutputStream bos = new ByteArrayOutputStream();
    MazePainter painter = painterCreator.createPainter(bos);
    maze.draw(painter, properties);
    bos.close();
    return systemAccess.saveUserFileContent(bos.toByteArray(), fileType);
  }

  /**
   * Store all persistent data.
   * This is called when the process is about to shutdown.
   */
  @Override 
  public void storePersistentData()
  {
    try {
      save();
    } catch (Exception e) {
      e.printStackTrace();
    }
  }

}
