// ============================================================================
// File:               $File$
//
// Project:            Maze creator.
//
// Purpose:
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2012  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: DxfMazePainter.java,v $
// History:	       Revision 1.1  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
//=============================================================================
package de.caff.maze;

import java.awt.*;
import java.awt.geom.*;
import java.io.IOException;
import java.io.OutputStream;

/**
 * Painter which outputs the maze into a DXF file.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @version $Revision: 1.1 $
 */
public class DxfMazePainter
        implements MazePainter,
                   Constants
{
  /** File type for DXF files. */
  public static final SystemAccess.FileType FILE_TYPE = new SystemAccess.FileType("dxf", "FILE_TYPE:dxf");

  private static final String ENCODING = "ISO-8859-1";
  private static final byte[] LINEFEED = new byte[] { '\n' };
  private Throwable error;
  private final OutputStream out;
  private String layer;
  private float aspectRatio;
  private AffineTransform transform;


  /**
   * Constructor.
   * This will not automatically close the stream.
   * @param out  output stream
   */
  public DxfMazePainter(OutputStream out)
  {
    this.out = out;
  }

  /**
   * Get any error which happened during usage of the painter.
   * @return error or <code>null</code> if the painter worked nicely
   */
  public Throwable getError()
  {
    return error;
  }

  /**
   * Write a single line to the file.
   * Only intended to be used by {@link #writeGroup(int, String)}
   * @param line line to write
   * @throws IOException when writing fails
   */
  private void writeLine(String line) throws IOException
  {
    out.write(line.getBytes(ENCODING));
    out.write(LINEFEED);
  }

  /**
   * Write the beginning of an entity.
   * This writes entity code and layer.
   * It is silently assumed that layer is not <code>null</code>.
   */
  private void writeEntityStart(String entity)
  {
    writeGroup(0, entity);
    writeGroup(8, layer);
  }

  /**
   * Write a typical DXF group.
   * @param groupCode group code
   * @param content   content of group
   */
  private void writeGroup(int groupCode, String content)
  {
    if (error != null) {
      return;
    }
    try {
      writeLine(Integer.toString(groupCode));
      writeLine(content);
    } catch (IOException e) {
      error = e;
    }
  }

  /**
   * Write a DXF group with float value.
   * @param groupCode group code
   * @param value     float value
   */
  private void writeFloatGroup(int groupCode, float value)
  {
    writeGroup(groupCode, Float.toString(value));
  }

  /**
   * Write a DXF group with an int value.
   * @param groupCode group code
   * @param value     int value
   */
  private void writeShortGroup(int groupCode, int value)
  {
    writeGroup(groupCode, Integer.toString(value));
  }

  /**
   * Start painting the maze.
   *
   * @param maze painted maze
   */
  public void startPaintingMaze(Maze maze)
  {
    aspectRatio = 1.0f / maze.getPreferredAspectRatio();
    transform = AffineTransform.getScaleInstance(1, -aspectRatio);
    layer = null;
    writeGroup(0, "SECTION");
    writeGroup(2, "ENTITIES");
  }

  /**
   * End painting the maze.
   */
  public void endPaintingMaze()
  {
    writeGroup(0, "ENDSEC");
    writeGroup(0, "EOF");
    writeGroup(999, String.format("Created by irrGardener version %s (see %s)", VERSION, WEB_PAGE));
  }

  /**
   * Start painting the given type of paint objects.
   *
   * @param type object type which painting starts
   */
  public void startPainting(PaintObjectType type)
  {
    switch (type) {
    case Background:
    case WayStart:
    case WayEnd:
    case Solution:
    case CellBorders:
      layer = null; // we don't care for these
      break;
    case InnerWalls:
      layer = "INNER_WALLS";
      break;
    case OuterWalls:
      layer = "OUTER_WALLS";
      break;
    }
  }

  /**
   * End painting the given type of paint objects.
   *
   * @param type object type which painting has ended
   */
  public void endPainting(PaintObjectType type)
  {
    layer = null;
  }

  /**
   * Set a stroke.
   *
   * @param stroke new stroke to use in upcoming drawing commands
   */
  public void setStroke(Stroke stroke)
  {
    // don't care for stroke
  }

  /**
   * Set a paint.
   *
   * @param paint paint to use in upcoming drawing commands.
   */
  public void setPaint(Paint paint)
  {
    // don't care for paint
  }

  /**
   * Draw a line with current paint and stroke..
   *
   * @param startX starting point X coordinate
   * @param startY starting point Y coordinate
   * @param endX   ending point X coordinate
   * @param endY   ending point Y coordinate
   */
  public void drawLine(float startX, float startY, float endX, float endY)
  {
    if (layer == null) {
      return;
    }
    writeEntityStart("LINE");
    writeFloatGroup(10, startX);
    writeFloatGroup(20, -startY * aspectRatio);
    writeFloatGroup(11, endX);
    writeFloatGroup(21, -endY * aspectRatio);
  }

  /**
   * Draw an open arc with current paint and stroke.
   *
   * @param x      x of rectangle enclosing ellipse
   * @param y      y of rectangle enclosing ellipse
   * @param w      width of rectangle enclosing ellipse
   * @param h      height of rectangle enclosing ellipse
   * @param start  start of angle in degrees, from x axis,
   * @param extent extent of angle in degrees
   */
  public void drawArc(float x, float y, float w, float h, float start, float extent)
  {
    if (layer == null) {
      return;
    }
    if (Math.abs(w - h * aspectRatio) > 1e4f) { // todo: care for abs values?
      draw(new Arc2D.Float(x, y, w, h, start, extent, Arc2D.OPEN));
    }
    else {
      float r = 0.5f * w;                      // radius
      float cx = x + 0.5f * w;                 // center X
      float cy = -y - 0.5f * h * aspectRatio;  // center Y
      float end;
      if (extent < 0) {
        end = start;
        start += extent;
      } else {
        end = start + extent;
      }
      writeEntityStart("ARC");
      writeFloatGroup(10, cx);
      writeFloatGroup(20, cy * aspectRatio);
      writeFloatGroup(40, r);
      writeFloatGroup(50, start);
      writeFloatGroup(51, end);
    }
  }

  /**
   * Check whether a shape contains bezier segments.
   * @param shape shape to check
   * @param coords helper array for segment return
   * @return <code>true</code> if the shape contains bezier segment<br/>
   *         <code>false</code> if it only contains straight segments
   */
  private static boolean hasBezierSegments(Shape shape, float[] coords)
  {
    for (PathIterator pit = shape.getPathIterator(null);  !pit.isDone();  pit.next()) {
      switch (pit.currentSegment(coords)) {
      case PathIterator.SEG_QUADTO:
      case PathIterator.SEG_CUBICTO:
        return true;
      }
    }
    return false;
  }

  /**
   * Draw a shape with current paint and stroke.
   *
   * @param shape shape to draw
   */
  public void draw(Shape shape)
  {
    if (layer == null) {
      return;
    }
    // output an approximation
    float[] coords = new float[6];
    boolean openLine = false;
    Point2D lastMove = null;
    for (PathIterator pit = shape.getPathIterator(transform, 1e-8);  !pit.isDone();  pit.next()) {
      Point2D p = null;
      switch (pit.currentSegment(coords)) {
      case PathIterator.SEG_MOVETO:
        if (openLine) {
          writeEntityStart("SEQEND");
        }
        writeEntityStart("POLYLINE");
        writeShortGroup(66, 1);
        writeFloatGroup(10, coords[0]);
        writeFloatGroup(20, coords[1]);
        openLine = true;
        p = lastMove = new Point2D.Float(coords[0], coords[1]);
        break;

      case PathIterator.SEG_LINETO:
        p = new Point2D.Float(coords[0], coords[1]);
        break;

      case PathIterator.SEG_CLOSE:
        p = lastMove;
        break;
      }
      writeEntityStart("VERTEX");
      writeFloatGroup(10, (float)p.getX());
      writeFloatGroup(20, (float)p.getY());
    }
    if (openLine) {
      writeEntityStart("SEQEND");
    }
  }

  /**
   * Fill a shape with the current paint.
   *
   * @param shape shape to fill
   */
  public void fill(Shape shape)
  {
    // as the DXF version created by this is too old to support HATCH which
    // could be easily used for filled shapes we just draw the shape as well
    draw(shape);
  }
}
