// ============================================================================
// File:               $File$
//
// Project:            
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2005-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 19:01:35 $
//
// History:	       $Log: DiamondMaze.java,v $
// History:	       Revision 1.5  2012/06/07 19:01:35  rammi
// History:	       Fixed jdoc ref
// History:
// History:	       Revision 1.4  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.3  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.2  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.1  2005/04/19 20:00:44  rammi
// History:	       no message
// History:	
//=============================================================================
package de.caff.maze;

import de.caff.gimmix.I18n;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.GeneralPath;
import java.util.ArrayList;
import java.util.Collection;

/**
 *  A rectangular maze with diamond shaped cells.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.5 $
 */
public class DiamondMaze
        extends AbstractBasicMaze
{
  /** The type of this maze (for storage). */
  public static final String MAZE_TYPE = "Diamond-Cell Maze";

  /** Property key for the settable number of horizontal cells property. */
  public static final String PROPERTY_HORIZONTAL_CELLS  = "DIAMOND_MAZE:nrHorizontalCells";
  /** Property key for the settable number of vertical cells property. */
  public static final String PROPERTY_VERTICAL_CELLS = "DIAMOND_MAZE:nrVerticalCells";
  /** Internally used bit value for direction NE. */
  private static final int NORTH_EAST = 1;
  /** Internally used bit value for direction SE. */
  private static final int SOUTH_EAST = 2;
  /** Internally used bit value for direction SW. */
  private static final int SOUTH_WEST = 4;
  /** Internally used bit value for direction NW. */
  private static final int NORTH_WEST = 8;

  /**
   *  A diamond shape maze cell.
   */
  private class DiamondMazeCell extends MazeCell
  {
    /** Number (id) of this cell. */
    private int number;
    /** Collection of bit values describing the connections of this cell. */
    private int connection;
    /** The shape of this cell. */
    private Shape shape;

    /**
     *  Constructor.
     *  @param num  number (id) if this cell
     */
    DiamondMazeCell(int num)
    {
      this.number = num;
    }

    /**
     *  Forget all connections.
     */
    public void reset()
    {
      connection = 0;
    }

    /**
     * Get the neighbour cells of this one.
     *
     * @return neigbour cells
     */
    public MazeCell[] getNeighbours()
    {
      java.util.List<MazeCell> neighbours = new ArrayList<MazeCell>(4);
      int row = number/(2*nrHorizontal+1);
      int rowPos = number%(2*nrHorizontal+1);
      final boolean longRow = rowPos >= nrHorizontal;
      if (longRow) {
        rowPos -= nrHorizontal;
        if (rowPos > 0) {
          neighbours.add(getCell(number-nrHorizontal-1)); // NORTH_WEST
          neighbours.add(getCell(number+nrHorizontal));   // SOUTH_WEST
        }
        if (rowPos < nrHorizontal) {
          neighbours.add(getCell(number-nrHorizontal));   // NORTH_EAST
          neighbours.add(getCell(number+nrHorizontal+1)); // SOUTH_EAST
        }
      }
      else {
        if (row > 0) {
          neighbours.add(getCell(number-nrHorizontal-1)); // NORTH_WEST
          neighbours.add(getCell(number-nrHorizontal));   // NORTH_EAST
        }
        if (row < nrVertical) {
          neighbours.add(getCell(number+nrHorizontal));   // SOUTH_WEST
          neighbours.add(getCell(number+nrHorizontal+1)); // SOUTH_EAST
        }
      }
      return neighbours.toArray(new MazeCell[neighbours.size()]);
    }

    /**
     * Connect this cell to the given one.
     * The algorithm will also call the inverse method for cell.
     *
     * @param cell cell to connect to
     */
    public void connectTo(MazeCell cell)
    {
      final int diff = number - ((DiamondMazeCell)cell).number;
      if (diff < 0) {
        // SOUTH
        if (-diff == nrHorizontal) {
          connection |= SOUTH_WEST;
        }
        else if (-diff == nrHorizontal+1) {
          connection |= SOUTH_EAST;
        }
        else {
          throw new InternalError(String.format("Unexpected connection request between cell number %d and cell number %d",
                                                number, ((DiamondMazeCell)cell).number));
        }
      }
      else {
        // NORTH
        if (diff == nrHorizontal) {
          connection |= NORTH_EAST;
        }
        else if (diff == nrHorizontal+1) {
          connection |= NORTH_WEST;
        }
        else {
          throw new InternalError(String.format("Unexpected connection request between cell number %d and cell number %d",
                                                number, ((DiamondMazeCell)cell).number));
        }
      }
    }

    /**
     * Get the connected neighbour cells of this one.
     *
     * @return neigbour cells
     */
    public MazeCell[] getConnectedNeighbours()
    {
      java.util.List<MazeCell> neighbours = new ArrayList<MazeCell>(4);
      if ((connection & NORTH_WEST) != 0) {
        neighbours.add(getCell(number-nrHorizontal-1));
      }
      if ((connection & NORTH_EAST) != 0) {
        neighbours.add(getCell(number-nrHorizontal));
      }
      if ((connection & SOUTH_WEST) != 0) {
        neighbours.add(getCell(number+nrHorizontal));
      }
      if ((connection & SOUTH_EAST) != 0) {
        neighbours.add(getCell(number+nrHorizontal+1));
      }
      return neighbours.toArray(new MazeCell[neighbours.size()]);
    }

    /**
     *  Get the connections of this cell as an or-ed collection of direction flags.
     *  @return direction flags: combination of {@link de.caff.maze.DiamondMaze#NORTH_EAST},
     *                           {@link de.caff.maze.DiamondMaze#NORTH_WEST},
     *                           {@link de.caff.maze.DiamondMaze#SOUTH_EAST} and
     *                           {@link de.caff.maze.DiamondMaze#SOUTH_WEST}
     */
    public int getConnection()
    {
      return connection;
    }

    /**
     *  Get the maze to which this cell belongs.
     *  @return the maze of this cell
     */
    public Maze getMaze()
    {
      return DiamondMaze.this;
    }

    /**
     *  Get the shape of this cell in the current display.
     *  @return maze cell shape
     */
    public Shape getShape()
    {
      return shape;
    }

    /**
     *  Set the shape of this cell in the current display.
     *  @param shape maze cell shape
     */
    public void setShape(Shape shape)
    {
      this.shape = shape;
    }

    /**
     * Get the id of this cell.
     * The id has to be unique for a given geometry of a maze.
     *
     * @return unique id
     * @see Maze#getCellByID(int)
     */
    public int getID()
    {
      return number;
    }
  }


  /** Number of horizontal cells. */
  private int nrHorizontal;
  /** Number of vertical cells. */
  private int nrVertical;
  /** The cells of this maze. */
  private DiamondMazeCell[] cells;

  /**
   *  Consructor.
   *  @param width  number of horizontal cells
   *  @param height number of vertical cells
   */
  public DiamondMaze(int width, int height)
  {
    initialize(width, height);
  }

  /**
   *  Initialize this maze.
   *  @param width  number of horizontal cells
   *  @param height number of vertical cells
   */
  private void initialize(int width, int height)
  {
    if (width != this.nrHorizontal  ||  height != this.nrVertical) {
      this.nrVertical = height;
      if (width != this.nrHorizontal) {
        int oldWidth  = this.nrHorizontal;
        this.nrHorizontal = width;
        firePropertyChange(PROPERTY_HORIZONTAL_CELLS, Integer.valueOf(oldWidth), Integer.valueOf(width));
      }
      if (height != this.nrVertical) {
        int oldHeight  = this.nrVertical;
        this.nrVertical = height;
        firePropertyChange(PROPERTY_VERTICAL_CELLS, Integer.valueOf(oldHeight), Integer.valueOf(height));
      }
      cells = new DiamondMazeCell[2*width*height+width+height];
      for (int i = 0;  i < cells.length;  ++i) {
        cells[i] = new DiamondMazeCell(i);
      }
      createShapes();
      setDefaultWayPoints();
    }
  }

  /**
   * Get a deep copy of this maze geometry.
   *
   * @return deep copy
   */
  protected AbstractBasicMaze getGeometryClone()
  {
    return new DiamondMaze(nrHorizontal, nrVertical);
  }

  /**
   *  Get a internally used string describing the maze.
   *  @return maze type
   */
  public String getMazeType()
  {
    return MAZE_TYPE;
  }

  /**
   *  Resets the internal data.
   */
  @Override public void reset()
  {
    for (int c = 0;  c < cells.length;  ++c) {
      cells[c].reset();
    }
    super.reset();
  }

  /**
   *  Get the cells of this maze.
   *  @return the cells of this maze
   */
  public MazeCell[] getCells()
  {
    MazeCell[] mCells = new MazeCell[cells.length];
    System.arraycopy(cells, 0, mCells, 0, cells.length);
    return mCells;
  }

  private DiamondMazeCell getCell(int nr)
  {
    return cells[nr];
  }

  /**
   * Get the cell with the given id.
   *
   * @param id cell id
   * @return the cell with the given id or <code>null</code> if there is no such cell
   * @see MazeCell#getID()
   */
  public MazeCell getCellByID(int id)
  {
    return id >= 0  &&  id < cells.length  ?
            getCell(id)  :
            null;
  }

  /** Shape of the border around this maze. */
  private Shape outShape;

  /**
   *  Create the shapes of the cells.
   */
  private void createShapes()
  {
    final float boxWidth  = BOX_SIZE;
    final float boxHeight = BOX_SIZE;

    final float deltaX = boxWidth/(nrHorizontal+1);
    final float deltaY = boxHeight/(nrVertical+1);

    outShape = createOutShape(deltaX, deltaY);

    for (int row = 0;  row < 2*nrVertical+1;  ++row) {
      final boolean longRow = row%2 != 0;
      float centerX = longRow ? deltaX/2 : deltaX;
      final float centerY = (row+1)*deltaY/2;
      final int rowStart = row/2*(2*nrHorizontal+1)+(longRow?nrHorizontal:0);
      final int rowLength = longRow ? nrHorizontal+1 : nrHorizontal;
      for (int c = 0;  c < rowLength;  ++c) {
        final DiamondMazeCell cell = getCell(rowStart+c);
        cell.setShape(getDiamondShape(centerX, centerY, deltaX, deltaY));

        centerX += deltaX;
      }
    }
  }

  /**
   *  Create the shape of the border.
   *  @param deltaX  cell width
   *  @param deltaY  cell height
   *  @return border shape
   */
  private Shape createOutShape(final float deltaX, final float deltaY)
  {
    final float maxX = deltaX*(nrHorizontal+1);
    final float maxY = deltaY*(nrVertical+1);

    GeneralPath path = new GeneralPath();
    path.moveTo(deltaX/2, deltaY/2);
    for (int i = 1;  i <= nrHorizontal;  ++i) {
      final float x = i*deltaX;
      path.lineTo(x, 0);
      path.lineTo(x+deltaX/2, deltaY/2);
    }
    for (int i = 1;  i <= nrVertical;  ++i) {
      final float y = i*deltaY;
      path.lineTo(maxX, y);
      path.lineTo(maxX-deltaX/2, y+deltaY/2);
    }
    for (int i = nrHorizontal;  i > 0;  --i) {
      final float x = i*deltaX;
      path.lineTo(x, maxY);
      path.lineTo(x-deltaX/2, maxY-deltaY/2);
    }
    for (int i = nrVertical;  i > 0;  --i) {
      final float y = i*deltaY;
      path.lineTo(0, y);
      path.lineTo(deltaX/2, y-deltaY/2);
    }
    path.closePath();
    return path;
  }

  /**
   * Do the actual drawing.
   * The call to this method is embedded in the the calls to
   * {@link de.caff.maze.MazePainter#startPaintingMaze(Maze)} and
   * {@link de.caff.maze.MazePainter#endPaintingMaze()}.
   *
   * @param painter    painter to draw to
   * @param properties access to properties for drawing (colors etc)
   */
  @Override
  protected void doDraw(MazePainter painter, MazePaintPropertiesProvider properties)
  {
    final float boxWidth  = BOX_SIZE;
    final float boxHeight = BOX_SIZE;

    final float deltaX = boxWidth/(nrHorizontal+1);
    final float deltaY = boxHeight/(nrVertical+1);

    drawBackgroundAndWay(painter, properties);

    painter.setStroke(new BasicStroke(deltaX / 10, BasicStroke.CAP_ROUND, BasicStroke.JOIN_BEVEL));
    if (properties.isShowingCellBorders()) {
      try {
        painter.startPainting(MazePainter.PaintObjectType.CellBorders);
        painter.setPaint(properties.getCellBorderPaint());
        for (int drow = 0;  drow < nrVertical;  ++drow) {
          float centerX = deltaX/2;
          final float centerY = (drow+1)*deltaY;
          final int rowStart = drow*(2*nrHorizontal+1)+nrHorizontal;
          final int rowEnd   = rowStart+nrHorizontal;
          for (int c = rowStart;  c <= rowEnd;  ++c) {
            int connection = getCell(c).getConnection();
            if ((connection & NORTH_WEST) != 0) {
              painter.drawLine(centerX - deltaX / 2, centerY, centerX, centerY - deltaY / 2);
            }
            if ((connection & NORTH_EAST) != 0) {

              painter.drawLine(centerX, centerY - deltaY / 2, centerX + deltaX / 2, centerY);
            }
            if ((connection & SOUTH_EAST) != 0) {
              painter.drawLine(centerX + deltaX / 2, centerY, centerX, centerY + deltaY / 2);
            }
            if ((connection & SOUTH_WEST) != 0) {
              painter.drawLine(centerX, centerY + deltaY / 2, centerX - deltaX / 2, centerY);
            }
            centerX += deltaX;
          }
        }
      } finally {
        painter.endPainting(MazePainter.PaintObjectType.CellBorders);
      }
    }

    try {
      painter.startPainting(MazePainter.PaintObjectType.InnerWalls);
      painter.setPaint(properties.getInnerWallsPaint());
      for (int drow = 0;  drow < nrVertical;  ++drow) {
        float centerX = deltaX/2;
        final float centerY = (drow+1)*deltaY;
        final int rowStart = drow*(2*nrHorizontal+1)+nrHorizontal;
        final int rowEnd   = rowStart+nrHorizontal;
        for (int c = rowStart;  c <= rowEnd;  ++c) {
          int connection = getCell(c).getConnection();
          if ((connection & NORTH_WEST) == 0) {
            painter.drawLine(centerX - deltaX / 2, centerY, centerX, centerY - deltaY / 2);
          }
          if ((connection & NORTH_EAST) == 0) {
            painter.drawLine(centerX, centerY - deltaY / 2, centerX + deltaX / 2, centerY);
          }
          if ((connection & SOUTH_EAST) == 0) {
            painter.drawLine(centerX + deltaX / 2, centerY, centerX, centerY + deltaY / 2);
          }
          if ((connection & SOUTH_WEST) == 0) {
            painter.drawLine(centerX, centerY + deltaY / 2, centerX - deltaX / 2, centerY);
          }
          centerX += deltaX;
        }
      }
    } finally {
      painter.endPainting(MazePainter.PaintObjectType.InnerWalls);
    }

    try {
      painter.startPainting(MazePainter.PaintObjectType.OuterWalls);
      painter.setPaint(properties.getOuterWallPaint());
      painter.setStroke(new BasicStroke(deltaX / 5));
      painter.draw(outShape);
    } finally {
      painter.endPainting(MazePainter.PaintObjectType.OuterWalls);
    }

  }

  /**
   *  Get the preferred aspect ratio of this maze.
   *  @return aspect ratio (width/height)
   */
  public float getPreferredAspectRatio()
  {
    return (float)nrHorizontal/(float)nrVertical;
  }

  /**
   * Get the necessary insets depending on the paint properties.
   * Usually the insets are necessary to allow for the thick border line to be drawn completely.
   *
   * @param properties paint properties
   * @param scaling    scaling used when painting
   * @return insets
   */
  public Insets getInsets(MazePaintPropertiesProvider properties, float scaling)
  {
    final float deltaX = BOX_SIZE/(nrHorizontal+1);
    int margin = (int)Math.ceil(scaling * deltaX/10 * Math.sqrt(2));
    return new Insets(margin, margin, margin, margin);
  }

  /**
   *  Create a diamond shape.
   *  @param centerX  center of diamond, x coord
   *  @param centerY  center of diamond, y coord
   *  @param deltaX   diamond width
   *  @param deltaY   diamond height
   *  @return diamond shape
   */
  private static Shape getDiamondShape(float centerX, float centerY, float deltaX, float deltaY)
  {
    GeneralPath shape = new GeneralPath();
    shape.moveTo(centerX, centerY-deltaY/2);
    shape.lineTo(centerX-deltaX/2, centerY);
    shape.lineTo(centerX, centerY+deltaY/2);
    shape.lineTo(centerX+deltaX/2, centerY);
    shape.closePath();
    return shape;
  }

  /**
   *  Set some useful default way points.
   */
  public void setDefaultWayPoints()
  {
    setWayPoints(getCell(0), getCell(cells.length-1));
  }

  /**
   *  Set the number of horizontal cells.
   *  @param nrHorizontal number of horizontal cells
   */
  private void setNrHorizontal(int nrHorizontal)
  {
    if (this.nrHorizontal != nrHorizontal) {
      initialize(nrHorizontal, nrVertical);
      recreateMaze();
    }
  }

  /**
   *  Set the number of vertical cells.
   *  @param nrVertical number of vertical cells
   */
  private void setNrVertical(int nrVertical)
  {
    if (this.nrVertical != nrVertical) {
      initialize(nrHorizontal, nrVertical);
      recreateMaze();
    }
  }

  /**
   *  Property information handling the number of horizontal cells.
   */
  private IntegerDelayedPropertyInformation numberHorizontalPropertySetter = new IntegerDelayedPropertyInformation(PROPERTY_HORIZONTAL_CELLS, 10, 1000, 1) {
    protected int getMazeValue()
    {
      return nrHorizontal;
    }

    protected void setMazeValue(int value)
    {
      setNrHorizontal(value);
    }
  };

  /**
   *  Property information handling the number of vertical cells.
   */
  private IntegerDelayedPropertyInformation numberVerticalPropertySetter = new IntegerDelayedPropertyInformation(PROPERTY_VERTICAL_CELLS, 10, 1000, 1) {
    protected int getMazeValue()
    {
      return nrVertical;
    }

    protected void setMazeValue(int value)
    {
      setNrVertical(value);
    }
  };

  /**
   *  Get the property setters and displays for this maze.
   *  @return colletion of property setters
   */
  @Override public Collection<PropertyInformation> getPropertyInformations()
  {
    Collection<PropertyInformation> setters  = super.getPropertyInformations();

    setters.add(numberHorizontalPropertySetter);
    setters.add(numberVerticalPropertySetter);

    return setters;
  }

  /**
   *  This is called during the call of {@link #setFromSetters()}
   *  and should be used to recreate the geometry of the maze from
   *  the setters of the geometric properties.
   */
  protected void recreateFromDelayedSetters()
  {
    initialize(numberHorizontalPropertySetter.getValue(),
               numberVerticalPropertySetter.getValue());
  }

  /**
   *  Get the borders of the maze as a shape.
   *  @return outer border
   */
  protected Shape getOuterBorder()
  {
    return outShape;
  }

  /**
   * Load extra data defining the maze from the system access.
   *
   * @param systemAccess system access
   */
  public void loadPersistentData(DataStorage systemAccess)
  {
    initialize(systemAccess.getInt(PROPERTY_HORIZONTAL_CELLS, nrHorizontal),
               systemAccess.getInt(PROPERTY_VERTICAL_CELLS, nrVertical));
    loadSeedWayAndVersion(systemAccess, MAZE_TYPE);
  }

  /**
   * Store extra data defining the maze to the system access.
   *
   * @param systemAccess system access
   */
  public void storePersistentData(DataStorage systemAccess)
  {
    systemAccess.setInt(PROPERTY_HORIZONTAL_CELLS, nrHorizontal);
    systemAccess.setInt(PROPERTY_VERTICAL_CELLS, nrVertical);
    storeSeedWayAndVersion(systemAccess, MAZE_TYPE);
  }

  /**
   *  Test code.
   *  @param args unused
   */
  public static void main(String[] args)
  {
    I18n.addAppResourceBase("de.caff.maze.MazeResourceBundle");
    JFrame frame = new JFrame("TEST");
    final int width = 12;
    final int height = 9;
    final DiamondMaze maze = new DiamondMaze(width, height);
    final MazePaintProperties properties = MazePaintProperties.getDrawMazePaintProperties(null);
    final MazeCanvas drawarea = new MazeCanvas(maze, properties,
                                              MazePrintProperties.getPrintMazePaintProperties(null));
    maze.createMaze();
    maze.setDefaultWayPoints();
    properties.setShowingSolution(true);
    properties.setShowingCellBorders(false);
    drawarea.addMouseListener(new MouseAdapter() {
      @Override public void mouseClicked(MouseEvent e)
      {
        try {
          drawarea.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
          maze.createMaze();
        } finally {
          drawarea.redraw();
          drawarea.setCursor(Cursor.getDefaultCursor());
        }
      }
    });
    frame.getContentPane().add(drawarea);
    frame.setSize(800, 600);

    frame.setVisible(true);

    frame.addWindowListener(new WindowAdapter() {
      @Override public void windowClosing(WindowEvent e)
      {
        System.exit(0);
      }
    });

  }
}
