// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 19:01:35 $
//
// History:	       $Log: CircularMaze.java,v $
// History:	       Revision 1.10  2012/06/07 19:01:35  rammi
// History:	       Fixed jdoc ref
// History:
// History:	       Revision 1.9  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.8  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.7  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.6  2005/04/19 20:00:44  rammi
// History:	       no message
// History:	
// History:	       Revision 1.5  2004/11/02 20:53:53  rammi
// History:	       Added keyboard shortcuts etc via actions
// History:	
// History:	       Revision 1.4  2004/10/31 23:30:52  rammi
// History:	       Redraw speed-up
// History:	
// History:	       Revision 1.3  2004/10/31 15:07:05  rammi
// History:	       Changed drawing to be always in BOX_SIZE
// History:	
// History:	       Revision 1.2  2004/10/25 14:54:57  rammi
// History:	       Javadoc changes
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import javax.swing.*;
import java.awt.*;
import java.awt.event.*;
import java.awt.geom.Arc2D;
import java.awt.geom.Ellipse2D;
import java.awt.geom.GeneralPath;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Iterator;

/**
 *  A circular maze consisting of cells in concentric rings.
 *  It is possible to create it with a center cell, no center cell or
 *  a hole in the center (a &quote;ring&quote;).
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.10 $
 */
public class CircularMaze
        extends AbstractBasicMaze
{
  /** The type of this maze (for storage). */
  public static final String MAZE_TYPE = "Circular Maze";

  /** Property key for the settable inner radius property. */
  public static final String PROPERTY_INNER_RADIUS      = "CIRCULAR_MAZE:innerRadius";
  /** Property key for the settable number of innermost rings property. */
  public static final String PROPERTY_NR_INNERMOST_RING = "CIRCULAR_MAZE:nrInnermostRing";
  /** Property key for the settable number of rings property. */
  public static final String PROPERTY_NR_RINGS          = "CIRCULAR_MAZE:nrRings";
  /** Special id for center cell. */
  public static final int CENTER_CELL_ID = -1;

  /** The factor we use for doubling the cells in the next outermost ring. */
  private static final float SIZE_SWITCH = 1.5f;

  /**
   *  Basic implementation of a shaped maze cell which is already able to store the cells connected to it.
   */
  private abstract class ConnectedMazeCell
          extends MazeCell
  {
    /** The cells connected to this one. */
    private ArrayList<MazeCell> connected = new ArrayList<MazeCell>(3);
    /** The shape of this cell. */
    private Shape shape;
    /** The id of this cell. */
    private final int id;

    /**
     *  Constructor.
     *  @param id  cell id
     */
    protected ConnectedMazeCell(int id)
    {
      this.id = id;
    }

    /**
     *  Connect this cell to the given one.
     *  @param cell  cell to connect to
     */
    public void connectTo(MazeCell cell)
    {
      connected.add(cell);
    }

    /**
     *  Get the cells connected to this one.
     *  @return  the connected cells
     */
    public MazeCell[] getConnectedNeighbours()
    {
      return connected.toArray(new MazeCell[connected.size()]);
    }

    /**
     *  Reset the connections.
     */
    public void reset()
    {
      connected.clear();
    }

    /**
     *  Is this cell connected to the given one.
     *  @param cell  cell to check for connection
     *  @return <code>true</code> if the cells are conneccted,
     *          <code>false</code> otherwise
     */
    public boolean isConnected(MazeCell cell)
    {
      return connected.contains(cell);
    }

    /**
     *  Get the maze to which this cell belongs.
     *  @return the cell's maze
     */
    public Maze getMaze()
    {
      return CircularMaze.this;
    }

    /**
     *  Get the shape of this cell.
     *  @return cell shape
     */
    public Shape getShape()
    {
      return shape;
    }

    /**
     *  Set the shape of this cell.
     *  @param shape cell shape
     */
    public void setShape(Shape shape)
    {
      this.shape = shape;
    }

    /**
     * Get the id of this cell.
     * The id has to be unique for a given geometry of a maze.
     *
     * @return unique id
     * @see Maze#getCellByID(int)
     */
    public int getID()
    {
      return id;
    }
  }

  /**
   *  A cell in one of the concentric rings.
   */
  private class RingMazeCell
    extends ConnectedMazeCell
  {
    /** The neighbour of the inner side (may be <code>null</code>. */
    private MazeCell inner;
    /** The clockwise neighbour in the ring. */
    private MazeCell clockwise;
    /** The counterclockwise neighbour in the ring. */
    private MazeCell counterClockwise;
    /** The neighbours of the outside (0, 1, or 2). */
    private MazeCell[] outer;

    /**
     *  Constructor.
     *  @param id   cell id
     */
    public RingMazeCell(int id)
    {
      super(id);
    }

    /**
     *  Return the neighbours of this cell.
     *  @return the neighbours
     */
    public MazeCell[] getNeighbours()
    {
      ArrayList<MazeCell> ret = new ArrayList<MazeCell>(5);
      if (inner != null) {
        ret.add(inner);
      }
      if (clockwise != null) {
        ret.add(clockwise);
      }
      if (counterClockwise != null) {
        ret.add(counterClockwise);
      }
      if (outer != null) {
        for (int o = 0;  o < outer.length;  ++o) {
          ret.add(outer[o]);
        }
      }
      return ret.toArray(new MazeCell[ret.size()]);
    }

    /**
     *  Get the neighbour cell of the inner side.
     *  @return inner side neighbour
     */
    public MazeCell getInner()
    {
      return inner;
    }

    /**
     *  Set the neighbour on the inner side.
     *  @param inner inner side neighbour
     */
    public void setInner(MazeCell inner)
    {
      this.inner = inner;
    }

    /**
     *  Get the clockwise neighbour cell.
     *  @return clockwise neighbour
     */
    public MazeCell getClockwise()
    {
      return clockwise;
    }

    /**
     *  Set the clockwise neighbour cell.
     *  @param clockwise clockwise neighbour
     */
    public void setClockwise(MazeCell clockwise)
    {
      this.clockwise = clockwise;
    }

    /**
     *  Get the counterclockwise neighbour cell.
     *  @return counterclockwise neighbour
     */
    public MazeCell getCounterClockwise()
    {
      return counterClockwise;
    }

    /**
     *  Set the counterclockwise neighbour cell.
     *  @param counterClockwise counterclockwise neighbour
     */
    public void setCounterClockwise(MazeCell counterClockwise)
    {
      this.counterClockwise = counterClockwise;
    }

    /**
     *  Get the outer neighbour cells.
     *  @return outer neighbour cells (<code>null</code> or an array with 1 or 2 members)
     */
    public MazeCell[] getOuter()
    {
      return outer;
    }

    /**
     *  Add an outer neighbour.
     *  @param outer outer neighbour
     */
    public void addOuter(MazeCell outer)
    {
      if (this.outer == null) {
        this.outer = new MazeCell[] { outer };
      }
      else {
        MazeCell[] temp = new MazeCell[this.outer.length+1];
        System.arraycopy(this.outer, 0, temp, 0, this.outer.length);
        temp[this.outer.length] = outer;
        this.outer = temp;
      }
    }
  }

  /**
   *  The cell in the center of the maze.
   */
  private class CenterMazeCell
    extends ConnectedMazeCell
  {
    /** The neighbours of this cell (i.e. the cells in the first ring. */
    private ArrayList<MazeCell> neighbours;

    /**
     *  Constructor.
     */
    public CenterMazeCell()
    {
      super(CENTER_CELL_ID);
    }

    /**
     *  Create a center cell with the given number of neighbours.
     *  @param numNeighbours number of neighbours
     */
    public CenterMazeCell(int numNeighbours)
    {
      this();
      neighbours = new ArrayList<MazeCell>(numNeighbours);
    }

    /**
     *  Get the neighbours of this cell.
     *  @return the neighbour cells
     */
    public MazeCell[] getNeighbours()
    {
      return neighbours.toArray(new MazeCell[neighbours.size()]);
    }

    /**
     *  Add a neighbour cell.
     *  @param cell neighbour cell to add
     */
    public void addNeighbour(MazeCell cell)
    {
      neighbours.add(cell);
    }
  }

  /**
   *  The radius of the center in units of ring thickness.
   *  There are 3 different cases:
   *  <ul>
   *    <li>The value is positive. This defines a maze with a circular center cell.</li>
   *    <li>The value is zero. This defines a maze with no center cell,
   *        i.e. a maze where the inner ring cells are formed like pie slices.</li>
   *    <li>The value is less than zero. This defines a maze with an empty center, i.e. a maze formed like a ring.
   *  </ul>
   */
  private float centerRadius;
  /** The center cell, if any. */
  private CenterMazeCell centerCell;
  /** The cells in the rings. */
  private ArrayList<RingMazeCell> cells = new ArrayList<RingMazeCell>();
  /** A integer array defining how much cells are in each ring (starting from the center). */
  private int[] numInRing;
  /** The outline shape of this maze. */
  private GeneralPath outShape;

  /**
   *  Property setter for the number of cells in the innermost ring.
   */
  private IntegerDelayedPropertyInformation innerRingPropertySetter = new IntegerDelayedPropertyInformation(PROPERTY_NR_INNERMOST_RING, 2, 1000, 1) {
    protected int getMazeValue()
    {
      return numInRing[0];
    }

    protected void setMazeValue(int value)
    {
      // todo: handle this
    }
  };

  /**
   *  Property setter for the number of rings.
   */
  private IntegerDelayedPropertyInformation numberRingsPropertySetter = new IntegerDelayedPropertyInformation(PROPERTY_NR_RINGS, 5, 1000, 1) {
    protected int getMazeValue()
    {
      return numInRing.length;
    }

    protected void setMazeValue(int value)
    {
      // todo: handle this
    }
  };

  /**
   *  Property setter for the inner cell size.
   */
  private DoubleDelayedPropertyInformation innerCellSizePropertySetter = new DoubleDelayedPropertyInformation(PROPERTY_INNER_RADIUS, -50, 50, 0.5) {
    protected double getMazeValue()
    {
      return centerRadius;
    }

    protected void setMazeValue(double value)
    {
      // todo: handle this
    }
  };

  /**
   *  Create a circular maze.
   *  @param centerRadius  the radius of the center in units of ring thickness.
   *         There are 3 different cases:
   *         <ul>
   *          <li>The value is positive. This defines a maze with a circular center cell.</li>
   *          <li>The value is zero. This defines a maze with no center cell,
   *              i.e. a maze where the inner ring cells are formed like pie slices.</li>
   *          <li>The value is less than zero. This defines a maze with an empty center, i.e. a maze formed like a ring.
   *         </ul>
   *  @param numFirstRing number of cells in innermost ring
   *  @param numRings number of rings
   */
  public CircularMaze(float centerRadius, int numFirstRing, int numRings)
  {
    initialize(centerRadius, numRings, numFirstRing);
  }

  /**
   *  Initialize the maze geometry.
   *  @param centerRadius  the radius of the center in units of ring thickness.
   *         There are 3 different cases:
   *         <ul>
   *          <li>The value is positive. This defines a maze with a circular center cell.</li>
   *          <li>The value is zero. This defines a maze with no center cell,
   *              i.e. a maze where the inner ring cells are formed like pie slices.</li>
   *          <li>The value is less than zero. This defines a maze with an empty center, i.e. a maze formed like a ring.
   *         </ul>
   *  @param numFirstRing number of cells in innermost ring
   *  @param numRings number of rings
   */
  private void initialize(float centerRadius, int numRings, int numFirstRing)
  {
    boolean differs = centerRadius != this.centerRadius  ||  numInRing == null  ||
            numInRing.length != numRings  ||  numInRing[0] != numFirstRing;

    if (differs) {
      cells.clear();
      if (this.centerRadius != centerRadius) {
        float oldValue = this.centerRadius;
        this.centerRadius = centerRadius;
        firePropertyChange(PROPERTY_INNER_RADIUS, Float.valueOf(oldValue), Float.valueOf(centerRadius));
      }
      centerRadius = Math.abs(centerRadius);
      if (numInRing != null) {
        if (numInRing.length != numRings) {
          firePropertyChange(PROPERTY_NR_RINGS,
                             Integer.valueOf(numInRing.length),
                             Integer.valueOf(numRings));
        }
        if (numInRing[0] != numFirstRing) {
          firePropertyChange(PROPERTY_NR_INNERMOST_RING,
                             Integer.valueOf(numInRing[0]),
                             Integer.valueOf(numFirstRing));
        }
      }
      numInRing = new int[numRings];
      int numInLastRing = numFirstRing;
      float measure = (centerRadius > 0) ? centerRadius/numFirstRing : 1.0f/numFirstRing;
      for (int r = 0;  r < numRings;  ++r) {
        int nrCellsInRing = numInLastRing;
        if ((centerRadius+r)/nrCellsInRing >= SIZE_SWITCH*measure) {
          nrCellsInRing *= 2;
        }
        numInRing[r] = nrCellsInRing;
        int mult = nrCellsInRing/numInLastRing;
        int start = cells.size();
        for (int c = 0;  c < nrCellsInRing;  ++c) {
          RingMazeCell cell = new RingMazeCell(cells.size());
          if (c > 0) {
            RingMazeCell lastCell = cells.get(cells.size()-1);
            lastCell.setCounterClockwise(cell);
            cell.setClockwise(lastCell);
          }
          if (r > 0) {
            RingMazeCell inner = cells.get(start+c/mult-numInLastRing);
            inner.addOuter(cell);
            cell.setInner(inner);
          }
          cells.add(cell);
        }
        RingMazeCell startCell = cells.get(start);
        RingMazeCell lastCell  = cells.get(cells.size()-1);
        startCell.setClockwise(lastCell);
        lastCell.setCounterClockwise(startCell);

        numInLastRing = nrCellsInRing;
      }

      if (this.centerRadius > 0) {
        centerCell = new CenterMazeCell(numFirstRing);
        for (int c = 0; c < numFirstRing;  ++c) {
          RingMazeCell cell = cells.get(c);
          cell.setInner(centerCell);
          centerCell.addNeighbour(cell);
        }
      }
      else {
        centerCell = null;
      }

      createShapes();

      setDefaultWayPoints();
    }
  }

  /**
   * Get a deep copy of this maze geometry.
   *
   * @return deep copy
   */
  protected AbstractBasicMaze getGeometryClone()
  {
    return new CircularMaze(centerRadius, numInRing.length, numInRing[0]);
  }

  /**
   *  Get a internally used string describing the maze.
   *  @return maze type
   */
  public String getMazeType()
  {
    return MAZE_TYPE;
  }

  /**
   *  Get all cells of this maze.
   *  @return the maze cells
   */
  public MazeCell[] getCells()
  {
    MazeCell[] ret = new MazeCell[centerCell != null  ?  cells.size()+1  :  cells.size()];
    cells.toArray(ret);
    if (centerCell != null) {
      ret[ret.length-1] = centerCell;
    }
    return ret;
  }

  /**
   *  Forget all connections.
   */
  @Override public void reset()
  {
    for (Iterator<RingMazeCell> it = cells.iterator();  it.hasNext();  ) {
      it.next().reset();
    }
    if (centerCell != null) {
      centerCell.reset();
    }
    super.reset();
  }

  /**
   *  Create the shapes of the cells if necessary.
   */
  private void createShapes()
  {
    final float width  = BOX_SIZE;
    final float height = BOX_SIZE;

    final float deltaR = 0.5f*width/(numInRing.length+Math.abs(centerRadius));
    final float aspect = height/width;
    final float startR = Math.abs(centerRadius)*deltaR;
    outShape = new GeneralPath(new Ellipse2D.Double(0, 0, width, height));
    if (centerRadius < 0) {
      // empty center
      outShape.append(new Ellipse2D.Double(0.5*width-startR, 0.5*height-startR*aspect,
                                           2*startR, 2*startR*aspect),
                      false);
      outShape.setWindingRule(GeneralPath.WIND_EVEN_ODD);
    }

    int pos = 0;
    for (int r = 0;  r < numInRing.length;  ++r) {
      float deltaA = 360.0f/numInRing[r];
      float radius = r*deltaR+startR;
      for (int c = 0;  c < numInRing[r];  ++c, ++pos) {
        RingMazeCell cell = cells.get(pos);
        GeneralPath path = new GeneralPath(new Arc2D.Float(0.5f*width-radius, 0.5f*height-radius*aspect,
                                                           2*radius, 2*radius*aspect,
                                                           c*deltaA, deltaA,
                                                           Arc2D.OPEN));
        path.append(new Arc2D.Float(0.5f*width-(radius+deltaR), 0.5f*height-(radius+deltaR)*aspect,
                                    2*(radius+deltaR), 2*(radius+deltaR)*aspect,
                                    c*deltaA+deltaA, -deltaA,
                                    Arc2D.OPEN),
                    true);
        path.closePath();
        cell.setShape(path);
      }
    }
    if (centerCell != null) {
      centerCell.setShape(new Ellipse2D.Double(0.5*width-startR, 0.5*height-startR*aspect,
                                               2*startR, 2*startR*aspect));
    }
  }


  /**
   * Do the actual drawing.
   * The call to this method is embedded in the the calls to
   * {@link de.caff.maze.MazePainter#startPaintingMaze(Maze)} and
   * {@link de.caff.maze.MazePainter#endPaintingMaze()}.
   *
   * @param painter    painter to draw to
   * @param properties access to properties for drawing (colors etc)
   */
  @Override
  protected void doDraw(MazePainter painter, MazePaintPropertiesProvider properties)
  {
    final float width  = BOX_SIZE;
    final float height = BOX_SIZE;

    final float deltaR = 0.5f*width/(numInRing.length+Math.abs(centerRadius));
    final float aspect = height/width;
    final float startR = Math.abs(centerRadius)*deltaR;

    final float strokeSize = deltaR*
            (float)Math.min(1.0,
                            2*Math.PI*(centerRadius == 0  ?  1  :  Math.abs(centerRadius))/numInRing[0]);

    drawBackgroundAndWay(painter, properties);

    painter.setStroke(new BasicStroke(strokeSize / 10));

    if (properties.isShowingCellBorders()) {
      try {
        painter.startPainting(MazePainter.PaintObjectType.CellBorders);
        painter.setPaint(properties.getCellBorderPaint());

        int pos = 0;
        for (int r = 0;  r < numInRing.length;  ++r) {
          float deltaA = 360.0f/numInRing[r];
          float radius = r*deltaR+startR;
          for (int c = 0;  c < numInRing[r];  ++c, ++pos) {
            RingMazeCell cell = cells.get(pos);
            MazeCell inner = cell.getInner();
            if (inner != null  &&  cell.isConnected(inner)) {
              // draw inner arc
              painter.drawArc(0.5f * width - radius, 0.5f * height - radius * aspect,
                              2 * radius, 2 * radius * aspect,
                              c * deltaA, deltaA);
            }
            if (cell.isConnected(cell.getClockwise())) {
              float cos = (float)Math.cos(Math.toRadians(c*deltaA));
              float sin = (float)-Math.sin(Math.toRadians(c*deltaA));
              painter.drawLine(0.5f * width + radius * cos, 0.5f * height + radius * sin * aspect,
                               0.5f * width + (radius + deltaR) * cos, 0.5f * height + (radius + deltaR) * sin * aspect);
            }
          }
        }
      } finally {
        painter.endPainting(MazePainter.PaintObjectType.CellBorders);
      }
    }

    try {
      painter.startPainting(MazePainter.PaintObjectType.InnerWalls);
    painter.setPaint(properties.getInnerWallsPaint());

    int pos = 0;
    for (int r = 0;  r < numInRing.length;  ++r) {
      float deltaA = 360.0f/numInRing[r];
      float radius = r*deltaR+startR;
      for (int c = 0;  c < numInRing[r];  ++c, ++pos) {
        RingMazeCell cell = cells.get(pos);
        MazeCell inner = cell.getInner();
        if (inner != null  &&  !cell.isConnected(inner)) {
          // draw inner arc
          painter.drawArc(0.5f * width - radius, 0.5f * height - radius * aspect,
                          2 * radius, 2 * radius * aspect,
                          c * deltaA, deltaA);
        }
        if (!cell.isConnected(cell.getClockwise())) {
          float cos = (float)Math.cos(Math.toRadians(c*deltaA));
          float sin = (float)-Math.sin(Math.toRadians(c*deltaA));
          painter.drawLine(0.5f * width + radius * cos, 0.5f * height + radius * sin * aspect,
                           0.5f * width + (radius + deltaR) * cos, 0.5f * height + (radius + deltaR) * sin * aspect);
        }
      }
    }
    } finally {
      painter.endPainting(MazePainter.PaintObjectType.InnerWalls);
    }

    try {
      painter.startPainting(MazePainter.PaintObjectType.OuterWalls);
      painter.setPaint(properties.getOuterWallPaint());
      painter.setStroke(new BasicStroke(strokeSize / 5));
      painter.draw(outShape);
    } finally {
      painter.endPainting(MazePainter.PaintObjectType.OuterWalls);
    }
  }

  /**
   *  Get the preferred aspect ratio (width/height) for this maze.
   *  @return preferred aspect ratio
   */
  public float getPreferredAspectRatio()
  {
    return 1;
  }

  /**
   * Get the necessary insets depending on the paint properties.
   * Usually the insets are necessary to allow for the thick border line to be drawn completely.
   *
   * @param properties paint properties
   * @param scaling    scaling used when painting
   * @return insets
   */
  public Insets getInsets(MazePaintPropertiesProvider properties, float scaling)
  {
    final float delta = 0.5f * BOX_SIZE / (numInRing.length+Math.abs(centerRadius));

    final float strokeSize = delta*
            (float)Math.min(1.0,
                            2*Math.PI*(centerRadius == 0  ?  1  :  Math.abs(centerRadius))/numInRing[0]);
    int margin = (int)Math.ceil(scaling*strokeSize/10);
    return new Insets(margin, margin, margin, margin);
  }

  /**
   *  Get the number of cells of this maze.
   *  @return the number of cells
   */
  public int getNumCells()
  {
    return centerCell != null  ?  cells.size()+1  :  cells.size();
  }

  /**
   * Get the cell with the given id.
   *
   * @param id cell id
   * @return the cell with the given id or <code>null</code> if there is no such cell
   * @see MazeCell#getID()
   */
  public MazeCell getCellByID(int id)
  {
    if (id == CENTER_CELL_ID) {
      return centerCell;
    }
    else if (id >= 0  &&  id < cells.size()) {
      return cells.get(id);
    }
    else {
      return null;
    }
  }

  /**
   *  Has this maze a center cell?
   *  @return <code>true</code>: maze has center cell, else <code>false</code>
   */
  public boolean hasCenterCell()
  {
    return centerCell != null;
  }

  /**
   *  Set some useful default way points.
   */
  public void setDefaultWayPoints()
  {
    MazeCell[] cells = getCells();
    setWayPoints(cells[hasCenterCell()  ?  cells.length-2 : 0], cells[cells.length-1]);
  }

  /**
   *  Recreate this maze from the setters which define the geometry.
   */
  public void recreateFromDelayedSetters()
  {
    initialize((float)innerCellSizePropertySetter.getValue(),
               numberRingsPropertySetter.getValue(),
               innerRingPropertySetter.getValue());
  }



  /**
   *  Get the property setters and displays for this maze.
   *  @return colletion of property setters
   */
  @Override public Collection<PropertyInformation> getPropertyInformations()
  {
    Collection<PropertyInformation> setters = super.getPropertyInformations();

    setters.add(innerRingPropertySetter);
    setters.add(numberRingsPropertySetter);
    setters.add(innerCellSizePropertySetter);

    return setters;
  }

  /**
   *  Get the shape of the outer border of this maze.
   *  @return the shape defining the outer border
   */
  protected Shape getOuterBorder()
  {
    return outShape;
  }


  /**
   * Load extra data defining the maze from the system access.
   *
   * @param systemAccess system access
   */
  public void loadPersistentData(DataStorage systemAccess)
  {
    initialize((float)systemAccess.getDouble(PROPERTY_INNER_RADIUS, centerRadius),
               systemAccess.getInt(PROPERTY_NR_RINGS, numInRing.length),
               systemAccess.getInt(PROPERTY_NR_INNERMOST_RING, numInRing[0]));
    loadSeedWayAndVersion(systemAccess, MAZE_TYPE);
  }

  /**
   * Store extra data defining the maze to the system access.
   *
   * @param systemAccess system access
   */
  public void storePersistentData(DataStorage systemAccess)
  {
    systemAccess.setDouble(PROPERTY_INNER_RADIUS, centerRadius);
    systemAccess.setInt(PROPERTY_NR_RINGS, numInRing.length);
    systemAccess.setInt(PROPERTY_NR_INNERMOST_RING, numInRing[0]);
    storeSeedWayAndVersion(systemAccess, MAZE_TYPE);
  }

  /**
   *  Test code.
   *  @param args unsued
   */
  public static void main(String[] args)
  {
    JFrame frame = new JFrame("TEST");
    final CircularMaze maze = new CircularMaze(-2, 12, 50);
    //final CircularMaze maze = new CircularMaze(2, 12, 30);
    final MazePaintProperties properties = MazePaintProperties.getDrawMazePaintProperties(null);
    properties.setShowingCellBorders(false);
    properties.setShowingSolution(true);
    final MazeCanvas drawarea = new MazeCanvas(maze, properties,
                                               MazePrintProperties.getPrintMazePaintProperties(null));
    maze.createMaze();
    maze.setDefaultWayPoints();
    drawarea.addMouseListener(new MouseAdapter() {
      @Override public void mouseClicked(MouseEvent e)
      {
        try {
          drawarea.setCursor(Cursor.getPredefinedCursor(Cursor.WAIT_CURSOR));
          maze.createMaze();
        } finally {
          drawarea.redraw();
          drawarea.setCursor(Cursor.getDefaultCursor());
        }
      }
    });
    frame.getContentPane().add(drawarea);
    frame.setSize(900, 900);

    frame.setVisible(true);

    frame.addWindowListener(new WindowAdapter() {
      @Override public void windowClosing(WindowEvent e)
      {
        System.exit(0);
      }
    });

  }

}

