// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: ApplicationSystemAccess.java,v $
// History:	       Revision 1.7  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.6  2009/09/24 17:32:20  rammi
// History:	       Finetuning dialog placement
// History:
// History:	       Revision 1.5  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.4  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.3  2004/11/02 20:53:53  rammi
// History:	       Added keyboard shortcuts etc via actions
// History:	
// History:	       Revision 1.2  2004/10/31 22:07:43  rammi
// History:	       Fixed problem with default ways on first execution
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import de.caff.gimmix.I18n;

import javax.swing.*;
import javax.swing.filechooser.FileFilter;
import java.awt.*;
import java.awt.print.Printable;
import java.awt.print.PrinterException;
import java.awt.print.PrinterJob;
import java.io.*;
import java.util.prefs.BackingStoreException;
import java.util.prefs.Preferences;

/**
 *  System access if running as an application.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.7 $
 */
class ApplicationSystemAccess
        implements SystemAccess
{
  /** Shortcut for mazeFilterDescription i18n resource key. */
  public static final String RESOURCE_MAZE_FILTER_DESCRIPTION = "APPL_ACCESS:mazeFilterDescription";
  /** Special return value for key search. */
  private static final String UNEXPECTED_VALUE = ";-) UnExPeCtEd (-;";
  //** The print request attribute set. */
  //private PrintRequestAttributeSet printRequestAttributes = new HashPrintRequestAttributeSet();
  /** The main frame of the application. */
  private JFrame mainFrame;

  /** The preferences object. */
  private final Preferences preferences;

  private static class ExtensionFileFilter
          extends FileFilter
  {
    /** The possible extensions. */
    private final String[] extensions;
    /** The description. */
    private final String   description;

    /**
     *  Constructor.
     *  @param filetype file type
     */
    private ExtensionFileFilter(FileType filetype)
    {
      String[] extensions = filetype.getExtensions();
      this.extensions = new String[extensions.length];
      for (int e = this.extensions.length - 1;  e >= 0;  --e) {
        this.extensions[e] = "." + extensions[e].toLowerCase();
      }
      description = filetype.getType();
    }

    /**
     * Whether the given file is accepted by this filter.
     */
    public boolean accept(File f)
    {
      if (f.isDirectory()) {
        return true;
      }
      String name = f.getName();
      for (int e = extensions.length-1;  e >= 0;  --e) {
        if (name.toLowerCase().endsWith(extensions[e])) {
          return true;
        }
      }
      return false;
    }

    /**
     * The description of this filter. For example: "JPG and GIF Images"
     *
     * @return the decription
     * @see javax.swing.filechooser.FileView#getName
     */
    public String getDescription()
    {
      StringBuilder b = new StringBuilder(description);
      b.append(" [");
      for (int e = 0;  e < extensions.length;  ++e) {
        b.append('*').append(extensions[e]);
        if (e < extensions.length - 1) {
          b.append(',');
        }
        b.append("]");
      }
      return b.toString();
    }
  }

  /**
   *  Constructor.
   */
  public ApplicationSystemAccess()
  {
    this.preferences = Preferences.userNodeForPackage(getClass());
  }

  /**
   * Print the given printable.
   *
   * @param printable printable to be printed
   * @return <code>true</code>: printing was successful,
   *         <code>false</code>: printing failed or was canceled
   */
  public boolean print(Printable printable)
  {
    PrinterJob job = PrinterJob.getPrinterJob();
    if (job.printDialog(/*printRequestAttributes*/)) {
      job.setPrintable(printable);

      try {
        job.print();
        return true;
      } catch (PrinterException x) {
        JOptionPane.showConfirmDialog(null,
                                      I18n.format(MultiMazeControlPanel.RESOURCE_PRINT_ERROR, x.getMessage()),
                                      MultiMazeControlPanel.RESOURCE_PRINT_ERROR+MazeResourceBundle.TITLE,
                                      JOptionPane.ERROR_MESSAGE);
      }
    }
    return false;
  }

  /**
   * Get a persistent string value.
   * A persistent string is a string which should survive the
   * end of the program.
   *
   * @param key          key for the string
   * @param defaultValue default value which is returned if the string is not defined
   * @return the string value defined by the key or the default value
   */
  public String getString(String key, String defaultValue)
  {
    return preferences.get(key, defaultValue);
  }

  /**
   * Set a persistent string.
   * The string should be stored so it may
   * be read again in this or a later session.
   *
   * @param key   key for the string
   * @param value the string value to store
   */
  public void setString(String key, String value)
  {
    preferences.put(key, value);
  }

  /**
   * Get a persistent boolean value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the boolean value defined by the key of the default value
   */
  public boolean getBoolean(String key, boolean defaultValue)
  {
    return preferences.getBoolean(key, defaultValue);
  }

  /**
   * Set a persistent boolean.
   *
   * @param key   key for the boolean
   * @param value value of the boolean
   */
  public void setBoolean(String key, boolean value)
  {
    preferences.putBoolean(key, value);
  }

  /**
   * Get a persistent color value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the color value deg´fined by the key of the default value
   */
  public Color getColor(String key, Color defaultValue)
  {
    return Stringizer.colorFromString(preferences.get(key, null), defaultValue);
  }

  /**
   * Set a persistent color.
   *
   * @param key   key for the color
   * @param value value of the color
   */
  public void setColor(String key, Color value)
  {
    preferences.put(key, Stringizer.colorToString(value));
  }

  /**
   * Get a persistent integer value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the integer value defined by the key of the default value
   */
  public int getInt(String key, int defaultValue)
  {
    return preferences.getInt(key, defaultValue);
  }

  /**
   * Set a persistent integer value.
   *
   * @param key   key for the int
   * @param value value of the int
   */
  public void setInt(String key, int value)
  {
    preferences.putInt(key, value);
  }

  /**
   * Get a persistent long integer value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the long integer value defined by the key of the default value
   */
  public long getLong(String key, long defaultValue)
  {
    return preferences.getLong(key, defaultValue);
  }

  /**
   * Set a persistent long integer value.
   *
   * @param key   key for the long int
   * @param value value of the long int
   */
  public void setLong(String key, long value)
  {
    preferences.putLong(key, value);
  }

  /**
   * Get a persistent double value.
   *
   * @param key          the key of the value
   * @param defaultValue the default value if the key is not defined
   * @return the double value defined by the key of the default value
   */
  public double getDouble(String key, double defaultValue)
  {
    return preferences.getDouble(key, defaultValue);
  }

  /**
   * Set a persistent double value.
   *
   * @param key   key for the int
   * @param value value of the int
   */
  public void setDouble(String key, double value)
  {
    preferences.putDouble(key, value);
  }

  /**
   * Is his key defined?
   *
   * @param key key to look for
   * @return <code>true</code> if the key is defined, <code>false</code> otherwise
   */
  public boolean hasKey(String key)
  {
    return preferences.get(key, UNEXPECTED_VALUE) != UNEXPECTED_VALUE;
  }

  /**
   * Store all persistent data.
   * This is called when the process is about to shutdown.
   */
  public void storePersistentData()
  {
    try {
      preferences.flush();
    } catch (BackingStoreException e) {
      e.printStackTrace();
    }
  }

  /**
   * Get the contents of a user selected file.
   *
   * @param filetype type of file to be used
   * @return byte array or <code>null</code> if the user canceled the operation
   * @throws java.io.IOException on errors during file access
   */
  public byte[] getUserFileContent(FileType filetype) throws IOException
  {
    JFileChooser chooser = new JFileChooser();
    chooser.setFileFilter(new ExtensionFileFilter(filetype));
    if (chooser.showOpenDialog(mainFrame) == JFileChooser.APPROVE_OPTION) {
      ByteArrayOutputStream bos = new ByteArrayOutputStream();
      FileInputStream fis = new FileInputStream(chooser.getSelectedFile());
      int c;
      while ((c = fis.read()) != -1) {
        bos.write(c);
      }
      fis.close();
      bos.close();
      return bos.toByteArray();
    }
    return null;  //To change body of implemented methods use File | Settings | File Templates.
  }

  /**
   * Save a byte array into a user selected file.
   *
   * @param content  the content of the file
   * @param filetype type of file to be used
   * @return <code>true</code> if the user saved the file, <code>false</code> otherwise
   * @throws java.io.IOException on errors during file access
   */
  public boolean saveUserFileContent(byte[] content, FileType filetype) throws IOException
  {
    JFileChooser chooser = new JFileChooser();
    chooser.setFileFilter(new ExtensionFileFilter(filetype));
    if (chooser.showSaveDialog(mainFrame) == JFileChooser.APPROVE_OPTION) {
      FileOutputStream fos = new FileOutputStream(chooser.getSelectedFile());
      fos.write(content);
      fos.close();
      return true;
    }
    return false;
  }

  /**
   * Sets the main frame of the application.
   * This is useful for centering dialogs.
   *
   * @param frame application's main frame
   */
  public void setMainFrame(JFrame frame)
  {
    mainFrame = frame;
  }
}
