// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2004-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: AbstractBasicMaze.java,v $
// History:	       Revision 1.9  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.8  2009/09/24 16:43:31  rammi
// History:	       Added image saving.
// History:
// History:	       Revision 1.7  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.6  2005/04/19 22:15:14  rammi
// History:	       Fixed some Thread problems
// History:	
// History:	       Revision 1.5  2005/04/19 20:00:44  rammi
// History:	       no message
// History:	
// History:	       Revision 1.4  2004/11/02 20:53:53  rammi
// History:	       Added keyboard shortcuts etc via actions
// History:	
// History:	       Revision 1.3  2004/10/31 22:07:43  rammi
// History:	       Fixed problem with default ways on first execution
// History:	
// History:	       Revision 1.2  2004/10/31 12:47:11  rammi
// History:	       Added versioned algorithms.
// History:	       Changed painting to fixed size BOX_SIZE.
// History:	
// History:	       Revision 1.1.1.1  2004/10/25 14:47:54  rammi
// History:	       Initial version
// History:	
//=============================================================================
package de.caff.maze;

import de.caff.gimmix.I18n;
import de.caff.gimmix.Worker;

import javax.swing.*;
import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.awt.geom.Point2D;
import java.beans.PropertyChangeEvent;
import java.beans.PropertyChangeListener;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedList;
import java.util.concurrent.Semaphore;

/**
 *  The abstract basis for a maze.
 *
 *  This class defines a lot of useful stuff common to all mazes.
 *  The live cycle of a maze mainly falls in three parts:
 *  <ol>
 *   <li> <b>Geometry creation</b>
 *    First a geometry is created which is in principle a description of maze cells
 *    and their neighbours. As long as the geometric properties are not changed
 *    this description is constant.
 *   </li>
 *   <li> <b>Maze creation</b>
 *    Second these cells have to be connected to create the real maze. The algorithm
 *    used creates exactly one way between any two cells of the geometry.
 *    This class allows the definition of two <b>way points</b> and calculates
 *    the resulting way between them (called the <b>solution</b>).
 *    This is usually repeated more often than part 1.
 *   </li>
 *   <li> <b>Drawing</b>
 *    The maze is drawn. This part is usually done more often than part 2.
 *   </li>
 *  </ol>
 *
 *  The extending classes need not care for step 2. Creation of a random maze
 *  and calculation of the solution is done here. The drawing of the solution
 *  and the way points may be delegated from extending classes, too.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.9 $
 */
public abstract class AbstractBasicMaze
        extends MazePropertyOwner
        implements Maze
{
  /**
   *  The size of the box to which the maze is drawn.
   *  A size of 1.0 would be more clearly but there are some bugs in java2d which
   *  make this a better choice.
   */
  public static final float BOX_SIZE = 1000;

  /** The property used when the maze changes. */
  public static final String PROPERTY_MAZE                 = "MAZE:maze";
  /** The property used when the way points (start and/or end) changes. */
  public static final String PROPERTY_WAY_POINTS           = "MAZE:wayPoints";
  /** The property used when the way changes. */
  public static final String PROPERTY_WAY                  = "MAZE:way";
  /** The property used when the seed changes. */
  public static final String PROPERTY_SEED                 = "MAZE:seed";
  /** The property used when the number of cells is changed. */
  public static final String PROPERTY_NUMBER_CELLS         = "MAZE:numberCells";
  /** The property used when the length of the solution chnages. */
  public static final String PROPERTY_SOLUTION_LENGTH      = "MAZE:solutionLength";
  /** The property used when the time needed for the creation is newly calculated. */
  public static final String PROPERTY_CREATION_TIME        = "MAZE:creationTime";

  /** Internal property used for storing: way start id. */
  private static final String PROPERTY_WAY_START_ID        = "MAZE:wayStart";
  /** Internal property used for storing: way end id. */
  private static final String PROPERTY_WAY_END_ID          = "MAZE:wayEnd";
  /** Internal property used for storing the version of maze creation algorithm. */
  private static final String PROPERTY_MAZE_ALGORITHM      = "MAZE:algorithmVersion";

  /** The cell where the way starts (may be <code>null</code>). */
  private MazeCell wayStart;
  /** The cell where the way ends (may be <code>null</code>). */
  private MazeCell wayEnd;
  /** The solution, a way from {@link #wayStart} (included) to {#link wayEnd} (included). */
  private Collection<MazeCell> way = null;

  /** The seed of the current maze. */
  private long     seed;
  /** The version of the creation algorithm. */
  private int      toolVersion = MazeTool.getCurrentMazeAlgorithmVersion();

  private static class ProgressShowerWrapper
          implements ProgressShower
  {
    private String note;
    private int maxValue;
    private int value = -1;
    private ProgressShower wrapped;

    /**
     * Start the action.
     *
     * @param note     note to display with the progress
     * @param maxValue maximum value
     */
    public synchronized void start(String note, int maxValue)
    {
      this.note = note;
      this.maxValue = maxValue;
      this.value = 0;
      if (wrapped != null) {
        wrapped.start(note, maxValue);
      }
    }

    /**
     * Set the current progress.
     *
     * @param value current progress
     * @return <code>true</code>: user requested an abort of the action,
     *         <code>false</code>: no abort requested, go on
     */
    public boolean setProgress(int value)
    {
      this.value = value;
      return wrapped != null && wrapped.setProgress(value);
    }

    /**
     * End the action.
     */
    public synchronized void end()
    {
      this.value = -1;
      if (wrapped != null) {
        wrapped.end();
      }
    }

    public synchronized void setWrapped(ProgressShower wrapped)
    {
      if (wrapped != null &&  value != -1) {
        wrapped.start(note, maxValue);
        wrapped.setProgress(value);
      }
      this.wrapped = wrapped;
    }

  }

  /** Wrapper for the progress shower. */
  private final ProgressShowerWrapper progressShowerWrapper = new ProgressShowerWrapper();

  /** Seed property setter. */
  private final SeedDelayedPropertyInformation seedSetter = new SeedDelayedPropertyInformation();

  /** The time for the creation of this maze. */
  private long creationTimeMillis = 0;

  /** The lock set during a recreation operation. */
  private Semaphore creationLock = new Semaphore(1, true);

  /** Listeners for finished maze creation. */
  private Collection<MazeFinishedListener> finishedListener = new LinkedList<MazeFinishedListener>();

  /**
   *  Default constructor.
   */
  protected AbstractBasicMaze()
  {
    addPropertyChangeListener(seedSetter);
  }

  /**
   *  Get a deep copy of this maze geometry.
   *  @return deep copy
   */
  protected abstract AbstractBasicMaze getGeometryClone();

  /**
   *  Get a clone of this maze.
   *  This may be a costly operation.
   *  @return a clone of this maze
   */
  public AbstractBasicMaze getClone()
  {
    final AbstractBasicMaze clone = getGeometryClone();
    final MazeCell start = getWayStart();
    if (start != null) {
      clone.setWayStart(clone.getCellByID(start.getID()));
    }
    final MazeCell end = getWayEnd();
    if (end != null) {
      clone.setWayEnd(clone.getCellByID(end.getID()));
    }
    clone.setSeed(getSeed());
    clone.toolVersion = this.toolVersion;
    return clone;
  }

  /**
   *  Is a recreation running?
   *  @return <code>true</code> during a recreation phase
   */
  public boolean isDuringRecreation()
  {
    return creationLock.availablePermits() == 0;
  }

  /**
   *  Solve this maze.
   *  This calculates a way between the way points.
   *  @see #setWayPoints(MazeCell, MazeCell)
   *  @see #setWayStart(MazeCell)
   *  @see #getWayStart()
   *  @see #setWayEnd(MazeCell)
   *  @see #getWayEnd()
   *  @see #getWay()
   */
  protected void solve()
  {
    setWay(MazeTool.solveMaze(wayStart, wayEnd));
  }

  /**
   *  Set the way.
   *  @param newWay new way
   */
  private void setWay(Collection<MazeCell> newWay)
  {
    Collection<MazeCell> oldWay = way;
    way = newWay;
    if (newWay != oldWay) {
      firePropertyChange(PROPERTY_WAY, oldWay, way);
    }
  }


  /**
   *  Set the way start and end points.
   *  Implicitely creates a solution.
   *  @param start  way start
   *  @param end    way end
   */
  public void setWayPoints(MazeCell start, MazeCell end)
  {
    if (start != wayStart  ||  end != wayEnd) {
      MazeCell[] oldWay = { wayStart, wayEnd };
      wayStart = start;
      wayEnd   = end;
      solve();
      firePropertyChange(PROPERTY_WAY_POINTS, oldWay, new MazeCell[] { start, end });
    }
  }

  /**
   *  Set the start cell of the way.
   *  @param start start cell
   */
  public void setWayStart(MazeCell start)
  {
    if (start != wayStart) {
      setWayPoints(start, getWayEnd());
    }
  }

  /**
   *  Set the end cell of the way.
   *  @param end  end cell
   */
  public void setWayEnd(MazeCell end)
  {
    if (end != wayEnd) {
      setWayPoints(getWayStart(), end);
    }
  }

  /**
   *  Get the start cell of the way.
   *  @return start cell
   */
  public MazeCell getWayStart()
  {
    return wayStart;
  }

  /**
   *  Get the end cell of the way.
   *  @return end cell
   */
  public MazeCell getWayEnd()
  {
    return wayEnd;
  }

  /**
   *  Get the solution way.
   *  @return the solution or <code>null</code> if there is no solution
   */
  public Collection<MazeCell> getWay()
  {
    return way;
  }

  /**
   *  Resets the internal data.
   *  Should be overwritten, and overwritung methods should call this.
   */
  public void reset()
  {
    setWay(null);
  }

  /**
   *  Create the maze with a random seed.
   */
  public void createMaze()
  {
    createMaze(System.currentTimeMillis());
  }

  /**
   *  Recreate the maze with the current seed.
   */
  protected void recreateMaze()
  {
    if (!isDuringRecreation()) {
      createMaze(getSeed());
    }
  }

  private void runLocked(final Runnable runnable)
  {
    new Thread(new Runnable() {
      public void run()
      {
        Throwable exception = null;
        try {
          creationLock.acquireUninterruptibly();
          runnable.run();
        } catch (Throwable x) {
          exception = x;
        } finally {
          creationLock.release();
        }
        fireFinished(exception);
      }
    }).start();
  }

  /**
   *  Set all values from the setters of the maze,
   *  than recreate it.
   *  @see #recreateFromDelayedSetters()
   */
  public void setFromSetters()
  {
    if (isDuringRecreation()) {
      return;
    }
    runLocked(new Runnable() {
      public void run()
      {
        recreateFromDelayedSetters();
        if (seedSetter.useSeedValue()) {
          _createMaze(seedSetter.getSeedValue(), MazeTool.LATEST_VERSION);
        }
        else {
          _createMaze(System.currentTimeMillis(), MazeTool.LATEST_VERSION);
        }
      }
    });
  }

  /**
   * Set the progress shower which is called during maze creation.
   * @param progressShower progress shower
   */
  public void setProgressShower(ProgressShower progressShower)
  {
    progressShowerWrapper.setWrapped(progressShower);
  }

  /**
   *  This is called during the call of {@link #setFromSetters()}
   *  and should be used to recreate the geometry of the maze from
   *  the setters of the geometric properties.
   */
  protected abstract void recreateFromDelayedSetters();

  /**
   *  Create a maze with the given randomSeed.
   *  @param randomSeed   random randomSeed
   */
  public void createMaze(final long randomSeed)
  {
    createMaze(randomSeed, MazeTool.LATEST_VERSION);
  }

  private void _createMaze(long randomSeed, int version)
  {
    //System.out.println("\nStart creation:                  "+System.currentTimeMillis());
    toolVersion = version == MazeTool.LATEST_VERSION ? MazeTool.getCurrentMazeAlgorithmVersion() : version;
    long start = System.currentTimeMillis();
    try {
      setSeed(randomSeed);
      Collection oldWay = getWay();
      final MazeTool mazeTool = MazeTool.getMazeTool(version);
      if (mazeTool == null) {
        // todo: raise error
      }
      else {
        boolean created = mazeTool.createMaze(AbstractBasicMaze.this, progressShowerWrapper, randomSeed);
        if (created) {
          firePropertyChange(PROPERTY_MAZE, null, this);
          way = MazeTool.solveMaze(wayStart, wayEnd);
          firePropertyChange(PROPERTY_WAY, oldWay, way);
          Long lastCreationTime = Long.valueOf(creationTimeMillis);
          creationTimeMillis = System.currentTimeMillis() - start;

          firePropertyChange(PROPERTY_CREATION_TIME, lastCreationTime, Long.valueOf(creationTimeMillis));
        }
        else {
          reset();
          solve();
        }
      }
    } finally {
      // System.out.println("End creation:                    "+System.currentTimeMillis());
      progressShowerWrapper.end();
    }

  }

  /**
   *  Create a maze with the given randomSeed using a special version of the creation algorithm.
   *  @param randomSeed    random randomSeed
   *  @param version maze creation algorithm version
   */
  public void createMaze(final long randomSeed, final int version)
  {
    if (!isDuringRecreation()) {
      runLocked(new Runnable() {
        public void run()
        {
          _createMaze(randomSeed, version);
        }
      });
    }
  }


  /**
   *  Get the seed which with the current maze is created.
   *  @return the seed
   */
  public long getSeed()
  {
    return seed;
  }

  /**
   *  Set the seed which which the current maze was created.
   *  @param seed seed for random number creation
   */
  private void setSeed(long seed)
  {
    if (seed != this.seed) {
      long oldSeed = this.seed;
      this.seed = seed;
      firePropertyChange(PROPERTY_SEED, Long.valueOf(oldSeed), Long.valueOf(seed));
    }
  }

  /**
   *  Get the creation time of this maze.
   *  @return creation time in milliseconds
   */
  public long getCreationTimeMillis()
  {
    return creationTimeMillis;
  }

  /**
   *  Helper method creating an int value from a string from an array, used
   *  in test code in <code>main()</code> methods of extending classes.
   *  @param args  array of strings
   *  @param arg   index into the array
   *  @param defaultValue  value to return if <code>args[arg]</code> is not defining an int
   *  @return integer value, either from <code>args[arg]</code> or the default value
   */
  public static int arg2int(String[] args, int arg, int defaultValue)
  {
    try {
      return Integer.parseInt(args[arg]);
    } catch (Exception x) {
      return defaultValue;
    }
  }


  /**
   *  Set some useful default way points.
   */
  public abstract void setDefaultWayPoints();


  /**
   *   A delayed setter for integer values.
   */
  protected abstract static class IntegerDelayedPropertyInformation
          extends AbstractPropertyInformation
  {
    /** Spinner widget to set the integer. */
    private JSpinner spinner;

    /**
     *  Create a setter with the given name and values.
     *  @param name     human readable property name
     *  @param minimum  minimum value for the property
     *  @param maximum  maximum value for the property
     *  @param step     step of spinner
     */
    protected IntegerDelayedPropertyInformation(String name, int minimum, int maximum, int step)
    {
      super(name);
      spinner = new JSpinner(new SpinnerNumberModel(minimum, minimum, maximum, step));
    }

    /**
     *  Reset the value displayed in the setter component to the value used in the maze.
     */
    @Override public void forget()
    {
      spinner.setValue(new Integer(getMazeValue()));
    }

    /**
     *  Get a component used for setting the property.
     *  @return setter component
     */
    public JComponent getSetterComponent()
    {
      spinner.setValue(new Integer(getMazeValue()));
      return spinner;
    }

    /**
     *  Get the current value of the setter component.
     *  @return setter component value
     */
    public int getValue()
    {
      return ((Integer)spinner.getValue()).intValue();
    }

    public Object getPropertyValue()
    {
      return Integer.valueOf(getMazeValue());
    }

    /**
     *  Get the property value used in the maze.
     *  @return maze value
     */
    protected abstract int getMazeValue();
  }

  /**
   *   A delayed setter for double values.
   */
  protected abstract static class DoubleDelayedPropertyInformation
          extends AbstractPropertyInformation
  {
    /** Spinner widget to set the double. */
    private JSpinner spinner;

    /**
     *  Create a setter with the given name and values.
     *  @param name     human readable property name
     *  @param minimum  minimum value for the property
     *  @param maximum  maximum value for the property
     *  @param step     step of spinner
     */
    protected DoubleDelayedPropertyInformation(String name, double minimum, double maximum, double step)
    {
      super(name);
      spinner = new JSpinner(new SpinnerNumberModel(minimum, minimum, maximum, step));
    }

    /**
     *  Reset the value displayed in the setter component to the value used in the maze.
     */
    @Override public void forget()
    {
      spinner.setValue(new Double(getMazeValue()));
    }

    /**
     *  Get a component used for setting the property.
     *  @return setter component
     */
    public JComponent getSetterComponent()
    {
      spinner.setValue(new Double(getMazeValue()));
      return spinner;
    }

    /**
     *  Get the current value of the setter component.
     *  @return setter component value
     */
    public double getValue()
    {
      return ((Double)spinner.getValue()).doubleValue();
    }

    public Object getPropertyValue()
    {
      return Double.valueOf(getMazeValue());
    }

    /**
     *  Get the property value used in the maze.
     *  @return maze value
     */
    protected abstract double getMazeValue();
  }

  /**
   *  Get a internally used string describing the maze type.
   *  @return maze type
   */
  public abstract String getMazeType();

  /**
   *  Draws the background.
   *  Draw the solution if it is wished by the user.
   *  Draws the way points, too.
   *  This should be called early in their <code>draw</code> routine, before any
   *  borders are drawn, but after the shapes are (re)created.
   *  @param painter painter to draw to
   *  @param properties  graphical maze properties
   */
  protected void drawBackgroundAndWay(MazePainter painter, MazePaintPropertiesProvider properties)
  {
    Paint bg = properties.getBackgroundPaint();
    if (bg != null) {
      try {
        painter.startPainting(MazePainter.PaintObjectType.Background);
        painter.setPaint(bg);
        painter.fill(getOuterBorder());
      } finally {
        painter.endPainting(MazePainter.PaintObjectType.Background);
      }
    }

    if (properties.isShowingSolution()) {
      Collection<MazeCell> wayCells = getWay();

      if (wayCells != null) {
        try {
          painter.startPainting(MazePainter.PaintObjectType.Solution);
          painter.setPaint(properties.getSolutionPaint());

          for (MazeCell cell: wayCells) {
            painter.fill(cell.getShape());
          }
        } finally {
          painter.endPainting(MazePainter.PaintObjectType.Solution);
        }
      }
    }
    if (getWayStart() != null) {
      try {
        painter.startPainting(MazePainter.PaintObjectType.WayStart);
        painter.setPaint(properties.getWayStartPaint());
        painter.fill(getWayStart().getShape());
      } finally {
        painter.endPainting(MazePainter.PaintObjectType.WayStart);
      }
    }
    if (getWayEnd() != null) {
      try {
        painter.startPainting(MazePainter.PaintObjectType.WayEnd);
        painter.setPaint(properties.getWayEndPaint());
        painter.fill(getWayEnd().getShape());
      } finally {
        painter.endPainting(MazePainter.PaintObjectType.WayEnd);
      }
    }
  }

  /**
   * Do the actual drawing.
   * The call to this method is embedded in the the calls to
   * {@link de.caff.maze.MazePainter#startPaintingMaze(Maze)} and
   * {@link de.caff.maze.MazePainter#endPaintingMaze()}.
   *
   * @param painter    painter to draw to
   * @param properties access to properties for drawing (colors etc)
   */
  protected abstract void doDraw(MazePainter painter, MazePaintPropertiesProvider properties);

  /**
   * Draw this maze.
   *
   * @param painter    painter to draw to
   * @param properties access to properties for drawing (colors etc)
   */
  public void draw(MazePainter painter, MazePaintPropertiesProvider properties)
  {
    painter.startPaintingMaze(this);
    try {
      doDraw(painter, properties);
    } finally {
      painter.endPaintingMaze();
    }
  }

  /**
   *  Get the cell at a given point. This method uses the second worst algorithm (first checking
   *  the outer border of the maze, and if necessary checking each
   *  cell shape whether it contains the point), so extending
   *  classes are encouraged to use their better understanding of the underlying geometry
   *  to calculate the target cell faster.
   *  @param position  cell position
   *  @return cell at position or <code>null</code> if there is no cell at the given position
   */
  public MazeCell getCellAt(Point2D position)
  {
    if (getOuterBorder().contains(position)) {
      final MazeCell[] cells = getCells();
      for (int c = 0;  c < cells.length;  ++c) {
        if (cells[c].getShape().contains(position)) {
          return cells[c];
        }
      }
    }
    return null;
  }

  public String getInfo(boolean onlyVariable)
  {
    StringBuilder info = new StringBuilder(I18n.getString(getMazeType())).append(" (");

    boolean first = true;
    for (PropertyInformation information: getPropertyInformations()) {
      if (!onlyVariable  ||  !information.isInformational()) {
        if (first) {
          first = false;
        }
        else {
          info.append(", ");
        }

        info.append(information.getLocalizedShortDescription()).append(": ").append(information.getPropertyValue());
      }
    }
    info.append(')');
    return info.toString();
  }

  /**
   *  Get the borders of the maze as a shape.
   *  @return outer border
   */
  protected abstract Shape getOuterBorder();

  /**
   *  Get the property setters and displays for this maze.
   *
   *  Overriding classes should include this collection first.
   *  @return colletion of property setters
   */
  public Collection<PropertyInformation> getPropertyInformations()
  {
    Collection<PropertyInformation> setters = new ArrayList<PropertyInformation>(4);

    setters.add(seedSetter);
    setters.add(new InfoPropertyDisplay(PROPERTY_NUMBER_CELLS,
                                        PROPERTY_MAZE,
                                        this) {
      protected Object getValue()
      {
        MazeCell[] cells = getCells();
        if (cells == null) {
          return "?";
        }
        else {
          return Integer.valueOf(cells.length);
        }
      }

    });
    setters.add(new InfoPropertyDisplay(PROPERTY_SOLUTION_LENGTH,
                                        PROPERTY_WAY,
                                        this) {
      protected Object getValue()
      {
        Collection way = getWay();
        if (way == null) {
          return "?";
        }
        else {
          return Integer.valueOf(way.size());
        }
      }

    });
    setters.add(new InfoPropertyDisplay(PROPERTY_CREATION_TIME,
                                        PROPERTY_CREATION_TIME,
                                        this) {
      protected Object getValue()
      {
        return Long.valueOf(getCreationTimeMillis());
      }

    });

    return setters;
  }

  /**
   *  A delayed property setter for the seed.
   */
  private class SeedDelayedPropertyInformation
          extends AbstractPropertyInformation
          implements PropertyChangeListener
  {
    /** Panel keeping together the text field and the radio button. */
    private final JPanel panel = new JPanel(new BorderLayout());
    /** Text field for seed input. */
    private final LongTextField seedField = new LongTextField(getSeed(), 12);
    /** Radiobutton for setting whether the seed value is used for maze creation. */
    private final JRadioButton useButton = new JRadioButton();

    /**
     *  Constructor.
     */
    public SeedDelayedPropertyInformation()
    {
      super(PROPERTY_SEED);

      panel.add(seedField, BorderLayout.CENTER);
      panel.add(useButton, BorderLayout.EAST);
      useButton.addActionListener(new ActionListener()
      {
        public void actionPerformed(ActionEvent e)
        {
          seedField.setEnabled(useButton.isSelected());
        }
      });
      seedField.setEnabled(false);
    }

    /**
     *  Get the component to set the property.
     *  @return proiperty setter component
     */
    public JComponent getSetterComponent()
    {
      return panel;
    }

    /**
     *  Get the value displayed in the seed field.
     *  @return seed value
     */
    public long getSeedValue()
    {
      return seedField.getValue();
    }

    public Object getPropertyValue()
    {
      return Long.valueOf(getSeedValue());
    }

    /**
     *  Reset the seed value to the one of the maze.
     */
    @Override public void forget()
    {
      seedField.setValue(getSeed());
    }

    /**
     *  Use the seed value for maze creation?
     *  @return the answer
     */
    public boolean useSeedValue()
    {
      return useButton.isSelected();
    }

    /**
     *  Internally called when any property of the maze is changed so
     *  the seed is updated.
     *  @param evt property change event
     */
    public void propertyChange(PropertyChangeEvent evt)
    {
      if (evt.getPropertyName().equals(PROPERTY_SEED)) {
        seedField.setValue(getSeed());
      }
    }
  }

  /**
   *  Load the seed and way points from the given data storage.
   *  @param dataStorage data storage to load from
   *  @param prefix key prefix unique for each maze
   */
  protected void loadSeedWayAndVersion(DataStorage dataStorage, String prefix)
  {
    final String seedKey = prefix+':'+PROPERTY_SEED;
    if (dataStorage.hasKey(seedKey)) {
      // something is stored
      long newSeed = dataStorage.getLong(seedKey, getSeed());
      int version  = dataStorage.getInt(prefix+':'+PROPERTY_MAZE_ALGORITHM, 0);
      if (MazeTool.getMazeTool(version) == null) {
        // todo: raise error
      }
      final MazeCell start = getCellByID(dataStorage.getInt(prefix+":"+PROPERTY_WAY_START_ID, Integer.MIN_VALUE));
      final MazeCell end   = getCellByID(dataStorage.getInt(prefix+":"+PROPERTY_WAY_END_ID,   Integer.MIN_VALUE));
      setWayPoints(start, end);
      createMaze(newSeed, version);
    }
  }

  /**
   *  Store the seed and way points to the given data storage.
   *  @param dataStorage data storage to load from
   *  @param prefix key prefix unique for each maze
   */
  protected void storeSeedWayAndVersion(DataStorage dataStorage, String prefix)
  {
    dataStorage.setLong(prefix+':'+PROPERTY_SEED, getSeed());
    dataStorage.setInt(prefix+':'+PROPERTY_MAZE_ALGORITHM, toolVersion);
    final MazeCell start = getWayStart();
    if (start != null) {
      dataStorage.setInt(prefix+":"+PROPERTY_WAY_START_ID, start.getID());
    }
    final MazeCell end = getWayEnd();
    if (end != null) {
      dataStorage.setInt(prefix+":"+PROPERTY_WAY_END_ID, end.getID());
    }
  }

  /**
   * Add a maze finished listener which is called when the maze (re)creation is finished.
   *
   * @param listener listener to add
   */
  public void addMazeFinishedListener(MazeFinishedListener listener)
  {
    synchronized(finishedListener) {
      finishedListener.add(listener);
    }
  }

  /**
   * Remove a maze finished listener which was called when the maze (re)creation is finished.
   *
   * @param listener listener to remove
   */
  public void removeMazeFinishedListener(MazeFinishedListener listener)
  {
    synchronized(finishedListener) {
      finishedListener.remove(listener);
    }
  }

  /**
   *  Fire the finished event.
   *  @param exception exception, if any ioccured during the creation
   */
  private void fireFinished(final Throwable exception)
  {
    Runnable runnable;
    synchronized(finishedListener) {
      runnable = new Runnable() {
        private Collection<MazeFinishedListener> listeners = new ArrayList<MazeFinishedListener>(finishedListener);

        public void run()
        {
          for (MazeFinishedListener listener: listeners) {
            listener.finished(AbstractBasicMaze.this, exception);
          }
        }
      };
    }
    Worker.invokeInEventDispatchThread(runnable);
  }
}
