// ============================================================================
// File:               $File$
//
// Project:            DXF viewer
//
// Purpose:            A Runnable which knows of exceptions.
//
// Author:             Rammi
// ----------------------------------------------------------------------------
// Copyright Notice:   (c) 2003-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: Worker.java,v $
// History:	       Revision 1.3  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.2  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.1  2005/04/19 22:15:14  rammi
// History:	       Fixed some Thread problems
// History:	
// History:	       Revision 1.3  2005/02/07 20:40:11  rammi
// History:	       Improved error handling
// History:	
// History:	       Revision 1.2  2004/07/22 14:36:01  rammi
// History:	       changes for DXF creation and writing
// History:	
// History:	       Revision 1.1  2003/01/23 12:00:59  rammi
// History:	       First release.
// History:	
//=============================================================================

package de.caff.gimmix;

import javax.swing.*;
import java.util.*;

/**
 *  A Runnable which knows of exceptions.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.3 $
 */
public abstract class Worker
        implements Runnable
{
  /** The exception catched (if any). */
  private Throwable catched;
  /** The listeners for finished work. */
  private List<KnockOffListener> knockOffListeners = new LinkedList<KnockOffListener>();

  /**
   *  Default constructor.
   */
  public Worker()
  {
  }

  /**
   *  Constructor taking an intial knock off listener.
   *  @param listener knock off listener
   */
  public Worker(KnockOffListener listener)
  {
    _addKnockOffListener(listener);
  }



  /**
   * When an object implementing interface <code>Runnable</code> is used
   * to create a thread, starting the thread causes the object's
   * <code>run</code> method to be called in that separately executing
   * thread.
   * <p>
   * The general contract of the method <code>run</code> is that it may
   * take any action whatsoever.
   *
   * @see     Thread#run()
   */
  public void run()
  {
    try {
      catched = null;
      execute();
    } catch (Throwable x) {
      catched = x;
    } finally {
      informKnockOffListeners();
    }
  }

  /**
   *  Implement this in extending classes to do the work.
   *  @throws Exception any exception thrown during work
   */
  protected abstract void execute() throws Exception;

  /**
   *  Get the thrwoable catched during work, if there happened one.
   *  @return the throwable or <code>null</code> if there didn't occure any
   */
  public Throwable getCatched()
  {
    return catched;
  }

  /**
   *  This throws the catched exception if there occured one.
   *  @throws Throwable exception catched during execute of work
   */
  public void rethrow()
          throws Throwable
  {
    if (catched != null) {
      throw catched;
    }
  }

  /**
   *  Add a knock off listener.
   *  The knock off listener will be called in the AWT event thread after the worker has finished.
   *  @param listener new listener
   */
  public void addKnockOffListener(KnockOffListener listener)
  {
    _addKnockOffListener(listener);
  }

  /**
   *  Add a knock off listener.
   *  The knock off listener will be called in the AWT event thread after the worker has finished.
   *  This is the internal implementation which cannot be overwritten by extending classes.
   *  @param listener new listener
   */
  private void _addKnockOffListener(KnockOffListener listener)
  {
    if (listener != null) {
      synchronized (knockOffListeners) {
        knockOffListeners.add(listener);
      }
    }
  }

  /**
   *  Remove a knock off listener.
   *  @param listener listener to remove
   */
  public void removeKnockOffListener(KnockOffListener listener)
  {
    if (listener != null) {
      synchronized (knockOffListeners) {
        knockOffListeners.remove(listener);
      }
    }
  }

  /**
   *  Call all knockoff listeners.
   */
  protected void informKnockOffListeners()
  {
    invokeInEventDispatchThread(new Runnable() {
      public void run()
      {
        Collection<KnockOffListener> knockOffListenersCopy;
        synchronized (knockOffListeners) {
          knockOffListenersCopy = new ArrayList<KnockOffListener>(knockOffListeners);
        }
        for (KnockOffListener listener: knockOffListenersCopy) {
          listener.knockedOff(Worker.this);
        }
      }
    });
  }

  /**
   *  Invoke in the AWT thread.
   *  @param run runnable which has to be invoked in the AWT event thread
   */
  public static void invokeInEventDispatchThread(Runnable run)
  {
    if (SwingUtilities.isEventDispatchThread()) {
      run.run();
    }
    else {
      SwingUtilities.invokeLater(run);
    }
  }

}
