// ============================================================================
// File:               Utility.java
//
// Project:            General.
//
// Purpose:            Different useful functions.
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2002-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: Utility.java,v $
// History:	       Revision 1.3  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.2  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.1.1.1  2004/10/25 14:47:55  rammi
// History:	       Initial version
// History:	
// History:	       Revision 1.9  2004/07/22 14:36:01  rammi
// History:	       changes for DXF creation and writing
// History:	
// History:	       Revision 1.8  2004/06/23 07:58:53  rammi
// History:	       Allowed objects instead of strings as format arguments
// History:	
// History:	       Revision 1.7  2003/09/22 08:42:29  rammi
// History:	       Fixed comments
// History:	
// History:	       Revision 1.6  2003/02/10 18:57:53  rammi
// History:	       Improved printing
// History:	
// History:	       Revision 1.5  2002/11/18 20:05:03  rammi
// History:	       Finetuning of comments and removal of superfluous code.
// History:	
// History:	       Revision 1.4  2002/04/19 08:55:33  rammi
// History:	       Removed escape references
// History:	
// History:	       Revision 1.3  2002/04/03 15:03:34  rammi
// History:	       Added possibility to define non-standard paths for image resources
// History:	
// History:	       Revision 1.2  2002/03/04 19:37:43  rammi
// History:	       added CVS variables
// History:	
//=============================================================================

package de.caff.gimmix;

import java.applet.Applet;
import java.awt.*;
import java.awt.image.ImageProducer;
import java.io.BufferedReader;
import java.io.File;
import java.io.InputStreamReader;
import java.net.URL;
import java.util.ResourceBundle;

/**
 *  Utility contains some helpful functionality.
 *
 *  Main thing to know is that the {@link Utility#setApplet(java.applet.Applet)}
 *  method has to be called explicitely in the <code>init()</code> method of an Applet
 *  for everything to work smoothly.
 *  @author Rammi
 *  @version $Revision: 1.3 $
 */  
public class Utility
{
  /** Used for access to prepareImage method. */
  private static Component preparer = new Canvas();
  /** (too?) simple */
  private static boolean   weAreOnDOS = (File.separatorChar == '\\');
  /** Directory to search for images and other resources */
  private static String    resourceDir   = "resources/";
  /** Applet if we are running in one. */
  private static Applet	   applet = null;

  /** Debugging mode. */
  private static boolean   debugging = false;

  /**
   *  Set the debugging mode.
   *  @param mode new mode
   */
  public static void setDebug(boolean mode) {
    debugging = mode;
  }

  /**
   *  Get the debug mode.
   *  @return debug mode
   */
  public static boolean isDebug() {
    return debugging;
  }

  /**
   *  Load an image and prepare a representation. Used for static images to be loaded
   *  in an very early stage of program execution.
   *  @param   path   path of the image file
   *  @return  the loaded image
   */
  public static Image loadImage(String path) {
    return loadImage(path, preparer);
  }

  /**
   *  Load an image and prepare a representation. Used for static images to be loaded
   *  in an very early stage of program execution.
   *  @param   path      path of the image file
   *  @param   renderer  renderer used to render the image
   *  @return  the loaded image
   */
  public static Image loadImage(String path, Component renderer) {
    if (resourceDir != null  &&  !path.startsWith("/")) {
      path = resourceDir + /*File.separator +*/ path;
    }
    return (new Utility()).loadAnImage(path, renderer);
  }

  /**
   *  Loads an image from a jar file. Be careful to always use /
   *  for dirs packed in jar!
   *  @param   path   path of file (e.g.. images/icon.gif)
   *  @param   renderer  component used for image rendering
   *  @return  the image
   */
  private Image loadAnImage(String path, Component renderer) {
    Image img = null;
    try {
      URL url = getClass().getResource(path);
      if (url == null) {
	// workaround for netscape problem
	if (applet != null) {
	  url = new URL(applet.getDocumentBase(), "de/caff/gimmicks/"+path);
	}
	else {
	  return null;
	}
      }

      if (applet != null) {
	img = applet.getImage(url);
      }
      else {
	img = Toolkit.getDefaultToolkit().createImage( (ImageProducer) url.getContent() );
      }
    } catch (Exception x) {
      debug(x);
    }
          
    if (img != null) {
      /* --- load it NOW --- */
      renderer.prepareImage(img, null);
    }

    return img;
  } 


  /**
   *  Load a text file into a string. 
   *  @param   path   name of the text file
   *  @return  the loaded text
   */
  public static String loadText(String path) {
    if (resourceDir != null) {
      path = resourceDir + /*File.separator +*/ path;
    }
    return (new Utility()).loadAText(path);
  }

  /**
   *  Loads a text file from a jar file. Be careful to always use /
   *  for dirs packed in jar!
   *  @param   path   path of file (e.g.. images/foo.txt)
   *  @return  the text
   */
  private String loadAText(String path) {
    String txt = "";
    try {
      String line;
      //      System.out.println("Loading "+path);
      
      //      System.out.println("URL = "+url);
      BufferedReader reader = new BufferedReader(new InputStreamReader(getClass().getResourceAsStream(path)));
      while ((line = reader.readLine()) != null) {
	txt += line +"\n";
      }
      reader.close();
    } catch (Exception x) {
      debug(x);
    }
          
    return txt;
  } 


  /**
   *  Test wether our System is a DOS.
   *  @return   true   we are on DOS
   */
  public static boolean areWeOnDOS() {
    return weAreOnDOS;
  }


  /**
   *  Set the resource directory.
   *  @param   dir   the image drirectory
   */
  public static void setResourceDir(String dir) {
    resourceDir = dir;
  }


  /**
   *  Compile a formatted string with maximum 10 args.
   *  <pre>
   *  Special signs:
   *     %#  where hash is a digit from 0 to 9 means insert arg #
   *     &#64;#  where hash is a digit from 0 to 9 means insert localized arg #
   *     %%  means %
   *     &#64;&@64;  means &#64; 
   *  </pre>
   *  @param   tag    resource tag for format string
   *  @param   args   arguments for insertion
   *  @param   res    active resource bundle
   *  @return  String with inserted args.
   */
  public static String compileString(String tag, Object[] args, ResourceBundle res) {
    String       format  = res.getString(tag);
    StringBuffer ret     = new StringBuffer(format.length());
    int          i;
    char         c;

    for (i = 0;   i < format.length();   i++) {
      c = format.charAt(i);
      
      if (c == '%'   ||   c == '@') {
	int argNum = -1;

        if (i < format.length()-1) {
	  // this implies that there are never more than 10 args
	  switch (format.charAt(i+1)) {
	  case '%':  
	    if (c == '%') { // "%%" means "%"
	      ret.append('%');
	      i++;
	    }
	    break;

	  case '@':
	    if (c == '@') { // "@@" means "@"
	      ret.append("@");
	      i++;
	    }
	    break;

	  case '0':
	    argNum = 0;
	    break;

	  case '1':
	    argNum = 1;
	    break;

	  case '2':
	    argNum = 2;
	    break;

	  case '3':
	    argNum = 3;
	    break;
	    
	  case '4':
	    argNum = 4;
	    break;

	  case '5':
	    argNum = 5;
	    break;

	  case '6':
	    argNum = 6;
	    break;

	  case '7':
	    argNum = 7;
	    break;
	    
	  case '8':
	    argNum = 8;
	    break;

	  case '9':
	    argNum = 9;
	    break;

	  default:
	    break;
	  }
	}
	if (argNum >= 0   &&   argNum < args.length) {
	  if (c == '%') {
	    // arg is a non-localized string
	    ret.append(args[argNum]);
	  }
	  else { // c == '@'
	    // arg is a tag for localization
	    ret.append(res.getString(args[argNum].toString()));
	  }
	  i++;
	}
      }
      else {
	ret.append(c);
      }
    }

    return new String(ret);
  }


  /**
   *  Method to get the frame parent of any component.
   *  @param   comp   the component to search the frame for
   *  @return  the frame parent of the component
   */
  public static Frame getFrame(Component comp) {
    for (   ;  comp != null;   comp = comp.getParent()) {
      if (comp instanceof Frame) {
	return (Frame)comp;
      }
    }
    /* --- Not found. Ugly workaround: --- */
    return new Frame();
  }


  /**
   *  Compare two byte arrays.
   *  Compare <code>len</code> bytes from array 1 starting with offset 1 
   *  with <code>len</code> bytes from array 2 starting with offset 2.
   *  Will return always <code>true</code> for <code>len &le;= 0</code>
   *  @param arr1    array 1
   *  @param off1    offset 1
   *  @param arr2    array 2
   *  @param off2    offset 2
   *  @param len     length to compare
   *  @return <code>true</code> if both chunks are equal<br>
   *          <code>false</code> otherwise
   */
  public static boolean equalBytes(byte[] arr1, int off1, byte[] arr2, int off2, int len) {
    while (len-- > 0) {
      //      System.out.println(arr1[off1] + " == "+arr2[off2]);
      if (arr1[off1++] != arr2[off2++]) {
	//	System.out.println();
	return false;		// not equal
      }
    }
    return true;		// equal
  }


  /**
   *  Set the applet we are running in (if any).
   *  @param applet   applet we are running in (if <code>null</code> then we
   *                  are running in an application
   */
  public static void setApplet(Applet applet) {
    Utility.applet = applet;
  }

  /**
   *  Get the applet we are running in (if any).
   *  @return applet or <code>null</code>
   */
  public static Applet getApplet() {
    return applet;
  }

  /**
   *  Are we running an applet?
   *  @return the answer
   */
  public static boolean areWeInAnApplet() {
    return applet != null;
  }

  /**
   *  Look for a boolean applet parameter or application property.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set) or the default
   */
  public static boolean getBooleanParameter(String key, boolean def) {
    String value = getStringParameter(key, null);
    if (value != null) {
      return "true".equals(value.toLowerCase());
    }
    else {
      return def;
    }
  }

  /**
   *  Look for a String applet parameter or application property.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set) or the default
   */
  public static String getStringParameter(String key, String def) {
    try {
      String value = areWeInAnApplet()  ?
	applet.getParameter(key)  :
	System.getProperty(key);

      if (value != null) {
	return value;
      }
    } catch (Exception x) {
      // do nothing
      debug(x);
    }

    // === return default ===
    return def;
  }

  /**
   *  Look for a color applet parameter or application property.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set) or the default
   */
  public static Color getColorParameter(String key, Color def) {
    String value = getStringParameter(key, null);
    if (value != null) {
      // try to decode color
      try {
	return Color.decode(value);
      } catch (Exception x) {
	// nothing
	debug(x);
      }
    }

    return def;
  }

  /**
   *  Look for a integer applet parameter or application property.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set) or the default
   */
  public static int getIntParameter(String key, int def) {
    return getIntParameter(key, def, 10);
  }


  /**
   *  Look for an integer applet parameter or application property.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @param  base number base
   *  @return the parameter value (if set) or the default
   */
  public static int getIntParameter(String key, int def, int base) {
    String value = getStringParameter(key, null);
    if (value != null) {
      try {
	return Integer.parseInt(value, base);
      } catch (NumberFormatException x) {
	// nothing
	debug(x);
      }
    }

    return def;
  }

  /**
   *  Look for a double applet parameter or application property.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set) or the default
   */
  public static double getDoubleParameter(String key, double def) {
    String value = getStringParameter(key, null);
    if (value != null) {
      try {
	return Double.valueOf(value).doubleValue();
      } catch (NumberFormatException x) {
	// nothing
	debug(x);
      }
    }

    return def;
  }

  /**
   *  Look for a float applet parameter or application property.
   *  @param  key  parameter key
   *  @param  def  default value
   *  @return the parameter value (if set) or the default
   */
  public static float getFloatParameter(String key, float def) {
    String value = getStringParameter(key, null);
    if (value != null) {
      try {
	return Float.valueOf(value).floatValue();
      } catch (NumberFormatException x) {
	// nothing
	debug(x);
      }
    }

    return def;
  }

  /**
   *  Print message if debug mode is on.
   *  @param  x  object which's toString is called
   */
  public static void debug(Object x) {
    if (debugging) {
      System.out.println(x == null  ?  "<null>"  :  x.toString());
    }
  }

  /**
   *  Print the stack trace if debug mode is on.
   *  @param  x  exception
   */
  public static void debug(Throwable x) {
    if (debugging) {
      x.printStackTrace();
    }
  }

  /**
   *  Print a given property to the console. Catch possible Security exceptions.
   *  Does nothing if not in debug mode.
   *  @param prop poperty name
   */
  public static void printProperty(String prop) {
    try {
      debug(prop+"="+System.getProperty(prop));
    } catch (Throwable x) {
      // empty
    }
  }

  /**
   *  In debug mode: print properties to console.
   */
  public static void printProperties() {
    if (Utility.isDebug()) {
      String[] useful = new String[] {
	"java.version",
	"java.vendor",
	"os.name",
	"os.arch",
	"os.version"
      };

      for (int u = 0;  u < useful.length;  ++u) {
	printProperty(useful[u]);
      }
    }
  }

  /**
   *  An equal function which accepts globbing.
   *  This method accepts the glob chars <tt>'*'</tt>
   *  (for any number of chars) and <tt>'?'</tt> (any single char).
   *  @param  mask   glob mask (containing special chars)
   *  @param  str    string to be checked against mask
   *  @return wether the string matches the mask
   */
  public static boolean globEquals(String mask, String str)
  {
    int maskLen = mask.length();
    int strLen  = str.length();
    if (maskLen > 0) {
      char first = mask.charAt(0);
      switch (first) {
      case '*':
	return 
	  globEquals(mask.substring(1), str) ||
	  (strLen > 0  &&  globEquals(mask, str.substring(1)));

      case '?':
	return strLen > 0  &&
	  globEquals(mask.substring(1), str.substring(1));

      default:
	return strLen > 0   &&
	  first == str.charAt(0)   &&
	  globEquals(mask.substring(1), str.substring(1));
      }
    }
    else {
      return maskLen == strLen;
    }
  }
}

