// ============================================================================
// File:               ResourceBundleCollection.java
//
// Project:            I18n
//
// Purpose:            Handle a collection of resource bundles to allow
//                     parts of an applictaion (e.g. beans) to maintain
//                     there own resource bundle but access all these bundles
//                     as an unit.
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2002-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: ResourceBundleCollection.java,v $
// History:	       Revision 1.3  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.2  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.1.1.1  2004/10/25 14:47:55  rammi
// History:	       Initial version
// History:	
// History:	       Revision 1.4  2004/07/22 14:36:01  rammi
// History:	       changes for DXF creation and writing
// History:	
// History:	       Revision 1.3  2003/09/22 08:42:43  rammi
// History:	       Fixed comments
// History:	
// History:	       Revision 1.2  2002/03/04 19:37:43  rammi
// History:	       added CVS variables
// History:	
//=============================================================================

package de.caff.gimmix;

import java.util.*;

/**
 *  Handle a collection of resource bundles. This allows
 *  parts of an application (e.g. beans) to maintain
 *  their own resource bundle but access to all these bundles
 *  is handled as an unit.
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.3 $
 */
public class ResourceBundleCollection
        extends ResourceBundle
{
  /** The collected resource bundles. */
  private List<ResourceBundle>     resourceBundles = new ArrayList<ResourceBundle>();
  /** The collected resources. */
  private HashMap<String,Object>  resources       = new HashMap<String,Object>();

  /**
   *  Add an resource bundle.
   *  <em>If this bundle defines an already defined resource,
   *  the old resource will be overwritten!</em>
   *  @param  bundle   resource bundle to add
   */
  public void addResourceBundle(ResourceBundle bundle) {
    resourceBundles.add(bundle);
    addEntries(bundle);
  }

  /**
   *  Remove a resource bundle.
   *  @param  bundle   resource bundle to remove
   */
  public void removeResourceBundle(ResourceBundle bundle) {
    if (resourceBundles.remove(bundle)) {
      // really removed
      // --- now we need to create a new Hashtable because we cannot
      // --- easily just remove the key/value pairs due to correct 
      // --- handleing of multiple entries
      resources = new HashMap<String,Object>();
      for (ResourceBundle resourceBundle: resourceBundles) {
	addEntries(resourceBundle);
      }
    }
  }

  /**
   *  Add the key/value pairs from a bundle to the collected list.
   *  @param  bundle   resource bundle for which the entries should be added
   */
  private void addEntries(ResourceBundle bundle) {
    for (Enumeration e = bundle.getKeys();   e.hasMoreElements();  ) {
      String key = (String)e.nextElement();
      if (I18n.DEBUG) {
	if (resources.get(key) != null) {
	  System.err.println("key '"+key+"'redefined in resource bundle "+bundle);
	}
      }
      resources.put(key, bundle.getObject(key));
    }
  }

  /**
   *  Must have method for getting the value for a key.
   *  @param  key  see description of java.util.ResourceBundle
   *  @return the value for the given key or <code>null</code>
   */
  protected Object handleGetObject(String key) {
    return resources.get(key);
  }

  /**
   *  Return an Enumeration of the keys.
   *  @return enumeration of all keys from the collected tables.
   */
  public Enumeration<String> getKeys() {
    return new Vector<String>(resources.keySet()).elements();
  }
}
