// ============================================================================
// File:               I18n.java
//
// Project:            I18n support.
//
// Purpose:            Application specific support for internationalisation.
//
// Author:             Rammi
//-----------------------------------------------------------------------------
// Copyright Notice:   (c) 2002-2006  Rammi (rammi@caff.de)
//
//                     This code was part of the irrGardener maze creation tool
//                     (see http://caff.de/maze/)
//                     and may be used and changed without restrictions
//                     since December 19, 2006.
//                     No guarantees are given.
//
// Latest change:      $Date: 2012/06/07 18:36:39 $
//
// History:	       $Log: I18n.java,v $
// History:	       Revision 1.3  2012/06/07 18:36:39  rammi
// History:	       FIxed typo in copyright comment.
// History:	       Added vector format outputs to DXF and SVG.
// History:
// History:	       Revision 1.2  2006/12/19 16:12:00  rammi
// History:	       Opened the code
// History:
// History:	       Revision 1.1.1.1  2004/10/25 14:47:55  rammi
// History:	       Initial version
// History:	
// History:	       Revision 1.6  2004/07/30 10:59:43  rammi
// History:	       global cleaning up
// History:	
// History:	       Revision 1.5  2004/07/22 14:36:01  rammi
// History:	       changes for DXF creation and writing
// History:	
// History:	       Revision 1.4  2004/06/23 08:01:21  rammi
// History:	       Allowed objects instead of strings as format arguments
// History:	
// History:	       Revision 1.3  2003/01/08 16:51:40  rammi
// History:	       Snapshot backup
// History:	
// History:	       Revision 1.2  2002/03/04 19:37:43  rammi
// History:	       added CVS variables
// History:	
//=============================================================================

package de.caff.gimmix;

import java.lang.ref.WeakReference;
import java.util.*;

/**
 *  Application specific support for internationalisation.
 *  This allows support for several resource classes for
 *  one application but <b>has the big disadvantage that
 *  separate resource bundles may  not define the same
 *  keys</b>.  Beware of that! You can switch DEBUG on
 *  to see if this happens.<p>
 *  This design should be overworked when possible.
 *  <p>
 *  To allow for enhancements the previous static design
 *  is now switched to something more object-oriented.
 *  This allows to take advantages of the new possibilities
 *  introduced with Java 1.2.
 *
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 *  @version $Revision: 1.3 $
 */
public class I18n
{
  /** Switch on to see whether there are key clashes. */
  static boolean     DEBUG            = false;

  /** 
   *  A singleton of the real I18n class which manages the
   *  I18n handling.
   */
  private static final I18n i18n = new I18n();

  /** The resource bundles. */
  protected Map<Locale, ResourceBundleCollection> resourceBundles  = new HashMap<Locale, ResourceBundleCollection>();
  /** The resource bases. */
  protected List<String>    appResourceBases = new ArrayList<String>();
  /** The default locale. */
  protected Locale    defaultLocale    = Locale.getDefault();

  /**
   *  Collection of known localizables, which have to be informed
   *  of localization changes.
   *  <p>
   *  The localizables are only weakly referenced here.
   */
  protected List<WeakReference<Localizable>> localizables     = new ArrayList<WeakReference<Localizable>>();



  /**
   *  Add an application specific resource class base name.
   *  This should be called by any application/applet before
   *  any other i18n specific code is used.
   *  @param  base  base class name for resources
   *  @see java.util.ResourceBundle
   */
  public static void addAppResourceBase(String base) {
    if (i18n != null) {
      i18n._addAppResourceBase(base);
    }
  }

  /**
   *  Add an application specific resource class base name.
   *  This should be called by any application/applet before
   *  any other i18n specific code is used.
   *  @param  base  base class name for resources
   *  @see java.util.ResourceBundle
   */
  protected void _addAppResourceBase(String base)
  {
    appResourceBases.add(base);
    resourceBundles.clear();
  }

  /**
   *  Set the locale to be used as a default for the application.
   *  @param  l   locale to be used as default
   */
  public static void setDefaultLocale(Locale l) {
    i18n._setDefaultLocale(l);
  }


  /**
   *  Set the locale to be used as a default for the application.
   *  @param  l   locale to be used as default
   */
  protected void _setDefaultLocale(Locale l)
  {
    if (!defaultLocale.equals(l)) {
      defaultLocale = l;
      _fireLocaleChanged(l);
    }
  }


  /**
   *  Get the locale to be used as a default for the application.
   *  @return   locale to be used as default
   */
  public static Locale getDefaultLocale() {
    return i18n._getDefaultLocale();
  }

  /**
   *  Get the locale to be used as a default for the application.
   *  @return   locale to be used as default
   */
  protected Locale _getDefaultLocale()
  {
    return defaultLocale;
  }

  /**
   *  Get a ResourceBundle for a locale.
   *  @param  l  locale
   *  @return ResourceBundle for that Locale
   *  @exception MissingResourceException  when no appResourceBase is set
   */
  private static final ResourceBundle getBundle(Locale l)
    throws MissingResourceException 
  {
    return i18n._getBundle(l);
  }

  /**
   *  Get a ResourceBundle for a locale.
   *  @param  l  locale
   *  @return ResourceBundle for that Locale
   *  @exception MissingResourceException  when no appResourceBase is set
   */
  protected ResourceBundle _getBundle(Locale l)
    throws MissingResourceException
  {
    if (l == null) {
      l = defaultLocale;
    }
    ResourceBundle res = resourceBundles.get(l);
    if (res == null) {
      // this resource is unknown yet
      if (appResourceBases.size() == 0) {
	throw new MissingResourceException("No application specific resource base defined",
					   "<unknown>",
					   "");
      }

      ResourceBundleCollection collect = new ResourceBundleCollection();

      for (String bundle: appResourceBases) {
	collect.addResourceBundle(ResourceBundle.getBundle(bundle, l));
      }

      resourceBundles.put(l, collect);

      return collect;
    }
    else {
      return res;
    }

  }


  /**
   *  Get a string for the default locale.
   *  @param   tag     resource tag
   *  @return  localized string
   *  @throws  MissingResourceException    when no appResourceBase is set
   */
  public static String getString(String tag)
          throws MissingResourceException
  {
    return getString(tag, null);
  }


  /**
   *  Get a String specified by a Locale.
   *  @param   tag     resource tag
   *  @param   l       Locale to be used
   *  @return  localized String
   *  @exception MissingResourceException  when no appResourceBase is set
   *  @see java.util.ResourceBundle#getString
   */
  public static String getString(String tag, Locale l)
          throws MissingResourceException
  {
    // System.out.println("Tag = "+tag+", l = "+(l != null ? l.getDisplayName() :  "<unknown>")+", returning \'"+ getBundle(l).getString(tag)+"'");
    return getBundle(l).getString(tag);
  }

  /**
   *  Compile a String with a format using the default locale.
   *  @see de.caff.gimmix.Utility#compileString
   *  @param  tag   resource tag of format
   *  @param  args  arguments
   *  @return  localized String
   *  @exception MissingResourceException  when no appResourceBase is set
   */
  public static String format(String tag, Object ... args)
    throws MissingResourceException 
  {
    return format(null, tag, args);
  }

  /**
   *  Compile a String with a format.
   *  @see de.caff.gimmix.Utility#compileString
   *  @param  l     locale
   *  @param  tag   resource tag of format
   *  @param  args  arguments
   *  @return  localized String
   *  @exception MissingResourceException  when no appResourceBase is set
   */
  public static String format(Locale l, String tag, Object ... args)
    throws MissingResourceException
  {
    return Utility.compileString(tag, args, getBundle(l));
  }

  /**
   *  Add a listener for localization changes.
   *  @param  localizable  listener for changes
   */
  public static void addLocalizationChangeListener(Localizable localizable)
  {
    i18n._addLocalizationChangeListener(localizable);
  }

  /**
   *  Add a listener for localization changes.
   *  @param  localizable  listener for changes
   */
  protected void _addLocalizationChangeListener(Localizable localizable)
  {
    localizables.add(new WeakReference<Localizable>(localizable));
    localizable.setLocale(_getDefaultLocale());
  }

  /**
   *  Remove a listener for localization changes.
   *  @param  localizable  listener to be removed
   */
  public static void removeLocalizationChangeListener(Localizable localizable)
  {
    i18n._removeLocalizationChangeListener(localizable);
  }

  /**
   *  Remove a listener for localization changes.
   *  @param  localizable  listener to be removed
   */
  protected void _removeLocalizationChangeListener(Localizable localizable)
  {
    for (ListIterator<WeakReference<Localizable>> it = localizables.listIterator(); it.hasNext(); ) {
      WeakReference<Localizable> ref = it.next();
      if (localizable.equals(ref.get())) {
        it.remove();
        return;
      }
    }
  }

  /**
   *  Tell all registered localizables of localization changes.
   *  @param  locale  new locale
   */
  protected void _fireLocaleChanged(Locale locale)
  {
    for (ListIterator<WeakReference<Localizable>> it = localizables.listIterator(); it.hasNext(); ) {
      WeakReference<Localizable> ref = it.next();
      Localizable loc = ref.get();
      if (loc != null) {
	loc.setLocale(locale);
      }
      else {
	// no longer referenced
	it.remove();
      }
    }
  }
}

