// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.version;

import de.caff.annotation.NotNull;
import de.caff.generics.Indexable;
import de.caff.version.SemVer;
import junit.framework.TestCase;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 27, 2021
 */
public class SemVerTest
        extends TestCase
{
  private static final String[] ORDER1 = {
          "1.0.0-alpha",
          "1.0.0-alpha.1",
          "1.0.0-alpha.beta",
          "1.0.0-beta",
          "1.0.0-beta.2",
          "1.0.0-beta.11",
          "1.0.0-rc.1",
          "1.0.0"
  };
  private static final String[] ORDER2 = {
          "1.0.0",
          "2.0.0",
          "2.1.0",
          "2.1.1",
          "17.2.9",
          "17.13.0"
  };

  private static final String[] ORDER3 = {
          "1.0.0-alpha",
          "1.0.0-beta",
          "1.0.0-rc.1",
          "1.0.0-rc.2",
          "1.0.0"
  };

  public void testParseToString()
  {
    checkParseOkay("1.0.0-alpha+001");
    checkParseOkay("1.0.0+20130313144700");
    checkParseOkay("1.0.0-beta+exp.sha.5114f85");
    checkParseOkay("1.0.0+21AF26D3---117B344092BD");
    checkParseOkay("1.0.0");
    checkParseOkay("2.0.0");
    checkParseOkay("2.1.0");
    checkParseOkay("2.1.1");
    for (String s : ORDER1) {
      checkParseOkay(s);
    }
    for (String s : ORDER2) {
      checkParseOkay(s);
    }
    for (String s : ORDER3) {
      checkParseOkay(s);
    }
  }

  private void checkParseOkay(@NotNull String txt)
  {
    final SemVer ver = SemVer.parse(txt);
    assertNotNull(txt, ver);
    assertEquals(txt, ver.toString());
  }

  public void testParseInvalid()
  {
    checkParseBroken("");
    checkParseBroken("1");
    checkParseBroken("1.");
    checkParseBroken("1.0");
    checkParseBroken("1.0.");
    checkParseBroken("1.0.0.");
    checkParseBroken("1.0.0-");
    checkParseBroken("1.0.0+");
    checkParseBroken("v1.0.0");
    checkParseBroken("1.0.0-alpha+beta+gamma");
    checkParseBroken("1.0.0-alpha.+beta");
    checkParseBroken("1.0.0-alpha.+beta.");
  }

  private void checkParseBroken(@NotNull String txt)
  {
    assertNull(txt, SemVer.parse(txt));
  }

  public void testOrder()
  {
    checkOrder(ORDER1);
    checkOrder(ORDER2);
    checkOrder(ORDER3);
  }

  private void checkOrder(@NotNull String ... texts)
  {
    assertTrue(texts.length >= 2);
    final Indexable<SemVer> incoming = Indexable.viewArray(texts).view(SemVer::parse);
    final Indexable<SemVer> ordered = incoming.sorted(SemVer::compareTo);
    assertEquals(incoming, ordered);
  }

  public void testBoundaries()
  {
    final SemVer ver = new SemVer(Integer.MAX_VALUE, Integer.MAX_VALUE, Integer.MAX_VALUE);
    checkThrowing(ver::nextMajor, ArithmeticException.class);
    checkThrowing(ver::nextMinor, ArithmeticException.class);
    checkThrowing(ver::nextPatch, ArithmeticException.class);
  }

  private <X extends Throwable> void checkThrowing(@NotNull Runnable code, @NotNull Class<X> exception)
  {
    try {
      code.run();
    } catch (Throwable x) {
      if (!exception.isInstance(x)) {
        fail("No exception "+ exception.getTypeName() + "raised!");
      }
    }
  }

  public void testUpdate()
  {
    final SemVer base = SemVer.parse("1.2.3-alpha+doodledah");
    assertNotNull(base);
    assertEquals("2.0.0", base.nextMajor().toString());
    assertEquals("1.3.0", base.nextMinor().toString());
    assertEquals("1.2.4", base.nextPatch().toString());
    assertEquals("1.2.3-beta", base.withPrerelease(SemVer.Prerelease.BETA).toString());
    assertEquals("1.2.3-alpha+hugglebax", base.withBuildString("hugglebax").toString());
  }

  public void testConstructChecks()
  {
    checkThrowing(() -> new SemVer(-1, 0, 0), IllegalArgumentException.class);
    checkThrowing(() -> new SemVer(0, Integer.MIN_VALUE, 0), IllegalArgumentException.class);
    checkThrowing(() -> new SemVer(0, 0, -13), IllegalArgumentException.class);
    checkThrowing(() -> new SemVer(0, 0, 0, ""), IllegalArgumentException.class);
    checkThrowing(() -> new SemVer(0, 0, 0, "1..no"), IllegalArgumentException.class);
    checkThrowing(() -> new SemVer(0, 0, 0, "+1.no"), IllegalArgumentException.class);
  }
}
