// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.range;

import de.caff.generics.Empty;
import de.caff.generics.Types;
import junit.framework.TestCase;

import java.util.Collections;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since March 02, 2020
 */
public class RangeTest
        extends TestCase
{
  public void testIndexes()
  {
    assertTrue(Types.areEqual(Range.indexes(4),
                              Types.asList(0, 1, 2, 3)));
    assertTrue(Types.areEqual(Range.indexes(Empty.STRING_ARRAY),
                              Collections.emptyList()));
  }

  public void testIntStep1()
  {
    assertTrue(Types.areEqual(Range.of(0, 6),
                              Types.asList(0, 1, 2, 3, 4, 5, 6)));
    assertTrue(Types.areEqual(Range.of(4, -3),
                              Types.asList(4, 3, 2, 1, 0, -1, -2, -3)));

    assertTrue(Types.areEqual(Range.of(1, 1),
                              Types.asList(1)));
    assertTrue(Types.areEqual(Range.of(Integer.MAX_VALUE - 3, Integer.MAX_VALUE),
                              Types.asList(Integer.MAX_VALUE - 3,
                                           Integer.MAX_VALUE - 2,
                                           Integer.MAX_VALUE - 1,
                                           Integer.MAX_VALUE)));
    assertTrue(Types.areEqual(Range.of(Integer.MIN_VALUE + 3, Integer.MIN_VALUE),
                              Types.asList(Integer.MIN_VALUE + 3,
                                           Integer.MIN_VALUE + 2,
                                           Integer.MIN_VALUE + 1,
                                           Integer.MIN_VALUE)));
    // int Range allows iterating over the complete int32 range
    final Iterable<Integer> maxRange = Range.of(Integer.MIN_VALUE, Integer.MAX_VALUE);
    long count = 0;
    final long securityBounds = 16L + (0x01L << 32);  
    for (Integer i : maxRange) {
      if (++count > securityBounds) {
        break;
      }
    }
    assertEquals(0x01L << 32, count);
  }

  public void testIntLarge()
  {
    int idx = Integer.MIN_VALUE;
    for (int i : Range.of(Integer.MIN_VALUE, Integer.MAX_VALUE)) {
      assertEquals(idx++, i);
    }
    assertEquals(Integer.MIN_VALUE, idx);

    idx = Integer.MAX_VALUE;
    for (int i : Range.of(Integer.MAX_VALUE, Integer.MIN_VALUE)) {
      assertEquals(idx--, i);
    }
    assertEquals(Integer.MAX_VALUE, idx);
  }

  public void testInt()
  {
    assertTrue(Types.areEqual(Range.of(1, 3, 2),
                              Types.asList(1, 3)));
    assertTrue(Types.areEqual(Range.of(1, 4, 2),
                              Types.asList(1, 3)));

    assertTrue(Types.areEqual(Range.of(17, -13, -5),
                              Types.asList(17, 12, 7, 2, -3, -8, -13)));
    assertTrue(Types.areEqual(Range.of(17, -16, -5),
                              Types.asList(17, 12, 7, 2, -3, -8, -13)));
  }

  public void testLongStep1()
  {
    assertTrue(Types.areEqual(Range.of(0L, 6L),
                              Types.asList(0L, 1L, 2L, 3L, 4L, 5L, 6L)));
    assertTrue(Types.areEqual(Range.of(4L, -3L),
                              Types.asList(4L, 3L, 2L, 1L, 0L, -1L, -2L, -3L)));

    assertTrue(Types.areEqual(Range.of(1L, 1L),
                              Types.asList(1L)));
    assertTrue(Types.areEqual(Range.of(Integer.MAX_VALUE - 3L, Integer.MAX_VALUE),
                              Types.asList(Integer.MAX_VALUE - 3L,
                                           Integer.MAX_VALUE - 2L,
                                           Integer.MAX_VALUE - 1L,
                                           (long)Integer.MAX_VALUE)));
    assertTrue(Types.areEqual(Range.of(Integer.MIN_VALUE + 3L, Integer.MIN_VALUE),
                              Types.asList(Integer.MIN_VALUE + 3L,
                                           Integer.MIN_VALUE + 2L,
                                           Integer.MIN_VALUE + 1L,
                                           (long)Integer.MIN_VALUE)));
  }

  public void testLongLarge()
  {
    final long step = 0x4_00_00_00_00L;
    long idx = Long.MIN_VALUE;
    for (long l : Range.of(Long.MIN_VALUE, Long.MAX_VALUE, step)) {
      assertEquals(idx, l);
      idx += step;
    }
    assertEquals(Long.MIN_VALUE, idx);

    idx = Long.MAX_VALUE;
    for (long l : Range.of(Long.MAX_VALUE, Long.MIN_VALUE, -step)) {
      assertEquals(idx, l);
      idx -= step;
    }
    assertEquals(Long.MAX_VALUE, idx);
  }

  public void testLong()
  {
    assertTrue(Types.areEqual(Range.of(1L, 3L, 2L),
                              Types.asList(1L, 3L)));
    assertTrue(Types.areEqual(Range.of(1L, 4L, 2L),
                              Types.asList(1L, 3L)));

    assertTrue(Types.areEqual(Range.of(17L, -13L, -5L),
                              Types.asList(17L, 12L, 7L, 2L, -3L, -8L, -13L)));
    assertTrue(Types.areEqual(Range.of(17L, -16L, -5L),
                              Types.asList(17L, 12L, 7L, 2L, -3L, -8L, -13L)));
  }

}
