// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.annotation.NotNull;
import de.caff.generics.function.DoubleOperator0;
import de.caff.generics.function.IntOperator0;
import de.caff.generics.function.LongOperator0;
import junit.framework.TestCase;

import static de.caff.generics.mda.MultiIndexLinearizer.OPEN;

/**
 * Speed tests for multi-dimensional arrays.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 11, 2019
 */
public class SpeedTest
        extends TestCase
{
  private static final String FORMAT = "%-30s %16d\n";
  static final int SIZE_4D = 64;
  static final int SIZE_3D = 256;
  static final int SIZE_2D = SIZE_4D * SIZE_4D;
  static final int SIZE_1D = SIZE_2D * SIZE_2D;
  static final long LCOUNT = ((long)SIZE_1D * (long)(SIZE_1D - 1)) / 2;
  static final int COUNT = (int)LCOUNT;
  static final double DCOUNT = (double)LCOUNT;
  
  private static int[] INT_ARRAY_1D;
  private static int[][] INT_ARRAY_2D;
  private static int[][][] INT_ARRAY_3D;
  private static int[][][][] INT_ARRAY_4D;
  private static MultiDimensionalIntArray INT_MDA_1D;
  private static MultiDimensionalIntArray INT_MDA_2D;
  private static MultiDimensionalIntArray INT_MDA_3D;
  private static MultiDimensionalIntArray INT_MDA_4D;
  private static OneDimensionalIntArray INT_1D_ARR;
  private static TwoDimensionalIntArray INT_2D_ARR;
  
  private static long[] LONG_ARRAY_1D;
  private static long[][] LONG_ARRAY_2D;
  private static long[][][] LONG_ARRAY_3D;
  private static long[][][][] LONG_ARRAY_4D;
  private static MultiDimensionalLongArray LONG_MDA_1D;
  private static MultiDimensionalLongArray LONG_MDA_2D;
  private static MultiDimensionalLongArray LONG_MDA_3D;
  private static MultiDimensionalLongArray LONG_MDA_4D;
  private static OneDimensionalLongArray LONG_1D_ARR;
  private static TwoDimensionalLongArray LONG_2D_ARR;
  
  private static double[] DOUBLE_ARRAY_1D;
  private static double[][] DOUBLE_ARRAY_2D;
  private static double[][][] DOUBLE_ARRAY_3D;
  private static double[][][][] DOUBLE_ARRAY_4D;
  private static MultiDimensionalDoubleArray DOUBLE_MDA_1D;
  private static MultiDimensionalDoubleArray DOUBLE_MDA_2D;
  private static MultiDimensionalDoubleArray DOUBLE_MDA_3D;
  private static MultiDimensionalDoubleArray DOUBLE_MDA_4D;
  private static OneDimensionalDoubleArray DOUBLE_1D_ARR;
  private static TwoDimensionalDoubleArray DOUBLE_2D_ARR;
  
  
  private static void time(@NotNull String name, @NotNull Runnable r)
  {
    final long start = System.nanoTime();
    r.run();
    final long end = System.nanoTime();
    System.out.printf(FORMAT, name, end - start);
  }

  private static void time(@NotNull String name, int repeat, @NotNull IntOperator0 r)
  {
    final long start = System.nanoTime();
    for (int i = 0;  i < repeat;  ++i) {
      assertEquals(r.applyAsInt(), COUNT);
    }
    final long end = System.nanoTime();
    System.out.printf(FORMAT, name, (end - start) / repeat);
  }

  private static void time(@NotNull String name, int repeat, @NotNull LongOperator0 r)
  {
    final long start = System.nanoTime();
    for (int i = 0;  i < repeat;  ++i) {
      assertEquals(r.applyAsLong(), LCOUNT);
    }
    final long end = System.nanoTime();
    System.out.printf(FORMAT, name, (end - start) / repeat);
  }

  private static void time(@NotNull String name, int repeat, @NotNull DoubleOperator0 r)
  {
    final long start = System.nanoTime();
    for (int i = 0;  i < repeat;  ++i) {
      assertEquals(r.applyAsDouble(), DCOUNT);
    }
    final long end = System.nanoTime();
    System.out.printf(FORMAT, name, (end - start) / repeat);
  }

  public void testInt1D()
  {
    time("Create INT ARR 1D", SpeedTest::getIntArray1D);
    time("Access INT ARR 1D", 16, SpeedTest::runIntArray1D);
    
    time("Create INT MDA 1D", SpeedTest::getIntMda1D);
    time("Access INT MDA 1D", 16, SpeedTest::runIntMda1D);

    time("Create INT 1D", SpeedTest::getLong1dArray);
    time("Access INT 1D", SpeedTest::runInt1dArr);
  }
  
  private static int[] getIntArray1D()
  {
    if (INT_ARRAY_1D == null) {
      INT_ARRAY_1D = new int[SIZE_1D];
      for (int i = 0;  i < SIZE_1D;  ++i) {
        INT_ARRAY_1D[i] = i;
      }
    }
    return INT_ARRAY_1D;
  }

  private static int runIntArray1D()
  {
    final int[] array = getIntArray1D();
    int count = 0;
    for (int i = 0;  i < SIZE_1D;  ++i) {
      count += array[i];
    }
    return count;
  }
  

  private static int runIntMda1D()
  {
    final MultiDimensionalIntArray array = getIntMda1D();
    int count = 0;
    for (int i = 0;  i < SIZE_1D;  ++i) {
      count += array.getValue(i);
    }
    return count;
  }

  @NotNull
  private static MultiDimensionalIntArray getIntMda1D()
  {
    if (INT_MDA_1D == null) {
      INT_MDA_1D = new MultiDimensionalIntArray(SIZE_1D);
      for (int i = 0; i < SIZE_1D; ++i) {
        INT_MDA_1D.setValue(i, i);
      }
    }
    return INT_MDA_1D;
  }

  public void testInt2D()
  {
    time("Create INT ARR 2D", SpeedTest::createIntArray2D);
    time("Access INT ARR 2D", 16, SpeedTest::runIntArray2D);
    
    time("Create INT MDA 2D", SpeedTest::createIntMda2D);
    time("Access INT MDA 2D", 16, SpeedTest::runIntMda2D);
    
    time("Create INT 2D", SpeedTest::createInt2dArr);
    time("Access INT 2D", 16, SpeedTest::runInt2dArr);
  }

  private static int runIntArray2D()
  {
    final int[][] array = createIntArray2D();
    int count = 0;
    for (int i = 0;  i < SIZE_2D;  ++i) {
      for (int j = 0;  j < SIZE_2D;  ++j) {
        count += array[i][j];
      }
    }
    return count;
  }

  private static int[][] createIntArray2D()
  {
    if (INT_ARRAY_2D == null) {
      INT_ARRAY_2D = new int[SIZE_2D][SIZE_2D];
      for (int i = 0; i < SIZE_2D; ++i) {
        for (int j = 0; j < SIZE_2D; ++j) {
          INT_ARRAY_2D[i][j] = i * SIZE_2D + j;
        }
      }
    }
    return INT_ARRAY_2D;
  }

  private static int runIntMda2D()
  {
    final MultiDimensionalIntArray array = createIntMda2D();
    int count = 0;
    for (int i = 0;  i < SIZE_2D;  ++i) {
      for (int j = 0;  j < SIZE_2D;  ++j) {
        count += array.getValue(i, j);
      }
    }
    return count;
  }

  @NotNull
  private static MultiDimensionalIntArray createIntMda2D()
  {
    if (INT_MDA_2D == null) {
      INT_MDA_2D = new MultiDimensionalIntArray(SIZE_2D, SIZE_2D);
      for (int i = 0; i < SIZE_2D; ++i) {
        for (int j = 0; j < SIZE_2D; ++j) {
          INT_MDA_2D.setValue(i * SIZE_2D + j, i, j);
        }
      }
    }
    return INT_MDA_2D;
  }

  private static int runInt2dArr()
  {
    final TwoDimensionalIntAccess array = createInt2dArr();
    int count = 0;
    for (int i = 0;  i < SIZE_2D;  ++i) {
      for (int j = 0;  j < SIZE_2D;  ++j) {
        count += array.getValueAt(i, j);
      }
    }
    return count;
  }

  @NotNull
  private static TwoDimensionalIntAccess createInt2dArr()
  {
    if (INT_2D_ARR == null) {
      INT_2D_ARR = new TwoDimensionalIntArray(SIZE_2D, SIZE_2D);
      for (int i = 0; i < SIZE_2D; ++i) {
        for (int j = 0; j < SIZE_2D; ++j) {
          INT_2D_ARR.setValueAt(i * SIZE_2D + j, i, j);
        }
      }
    }
    return INT_2D_ARR;
  }

  public void testInt3D()
  {
    time("Create INT ARR 3D", SpeedTest::createIntArray3D);
    time("Access INT ARR 3D", 16, SpeedTest::runIntArray3D);
    
    time("Create INT MDA 3D", SpeedTest::createIntMda3D);
    time("Access INT MDA 3D", 16, SpeedTest::runIntMda3D);
  }

  private static int runIntArray3D()
  {
    final int[][][] array = createIntArray3D();
    int count = 0;
    for (int i = 0;  i < SIZE_3D;  ++i) {
      for (int j = 0;  j < SIZE_3D;  ++j) {
        for (int k = 0;  k < SIZE_3D;  ++k) {
          count += array[i][j][k];
        }
      }
    }
    return count;
  }

  private static int[][][] createIntArray3D()
  {
    if (INT_ARRAY_3D == null) {
      INT_ARRAY_3D = new int[SIZE_3D][SIZE_3D][SIZE_3D];
      for (int i = 0; i < SIZE_3D; ++i) {
        for (int j = 0; j < SIZE_3D; ++j) {
          final int base = i * SIZE_3D * SIZE_3D + j * SIZE_3D;
          for (int k = 0; k < SIZE_3D; ++k) {
            INT_ARRAY_3D[i][j][k] = base + k;
          }
        }
      }
    }
    return INT_ARRAY_3D;
  }

  private static int runIntMda3D()
  {
    final MultiDimensionalIntArray array = createIntMda3D();
    int count = 0;
    for (int i = 0;  i < SIZE_3D;  ++i) {
      for (int j = 0;  j < SIZE_3D;  ++j) {
        for (int k = 0;  k < SIZE_3D;  ++k) {
          count += array.getValue(i, j, k);
        }
      }
    }
    return count;
  }

  @NotNull
  private static MultiDimensionalIntArray createIntMda3D()
  {
    if (INT_MDA_3D == null) {
      INT_MDA_3D = new MultiDimensionalIntArray(SIZE_3D, SIZE_3D, SIZE_3D);
      for (int i = 0; i < SIZE_3D; ++i) {
        for (int j = 0; j < SIZE_3D; ++j) {
          final int base = i * SIZE_3D * SIZE_3D + j * SIZE_3D;
          for (int k = 0; k < SIZE_3D; ++k) {
            INT_MDA_3D.setValue(base + k, i, j, k);
          }
        }
      }
    }
    return INT_MDA_3D;
  }


  public void testInt4D()
  {
    time("Create INT ARR 4D", SpeedTest::createIntArray4D);
    time("Access INT ARR 4D", 16, SpeedTest::runIntArray4D);
    
    time("Create INT MDA 4D", SpeedTest::createIntMda4D);
    time("Access INT MDA 4D", 16, SpeedTest::runIntMda4D);
  }

  private static int runIntArray4D()
  {
    final int[][][][] array = createIntArray4D();
    int count = 0;
    for (int i = 0;  i < SIZE_4D;  ++i) {
      for (int j = 0;  j < SIZE_4D;  ++j) {
        for (int k = 0;  k < SIZE_4D;  ++k) {
          for (int l = 0;  l < SIZE_4D;  ++l) {
            count += array[i][j][k][l];
          }
        }
      }
    }
    return count;
  }

  private static int[][][][] createIntArray4D()
  {
    if (INT_ARRAY_4D == null) {
      INT_ARRAY_4D = new int[SIZE_4D][SIZE_4D][SIZE_4D][SIZE_4D];
      for (int i = 0; i < SIZE_4D; ++i) {
        for (int j = 0; j < SIZE_4D; ++j) {
          final int base = i * SIZE_4D * SIZE_4D + j * SIZE_4D;
          for (int k = 0; k < SIZE_4D; ++k) {
            final int base2 = (base + k) * SIZE_4D;
            for (int l = 0; l < SIZE_4D; ++l) {
              INT_ARRAY_4D[i][j][k][l] = base2 + l;
            }
          }
        }
      }
    }
    return INT_ARRAY_4D;
  }

  private static int runIntMda4D()
  {
    final MultiDimensionalIntArray array = createIntMda4D();
    int count = 0;
    for (int i = 0;  i < SIZE_4D;  ++i) {
      for (int j = 0;  j < SIZE_4D;  ++j) {
        for (int k = 0;  k < SIZE_4D;  ++k) {
          for (int l = 0;  l < SIZE_4D;  ++l) {
            count += array.getValue(i, j, k, l);
          }
        }
      }
    }
    return count;
  }

  @NotNull
  private static MultiDimensionalIntArray createIntMda4D()
  {
    if (INT_MDA_4D == null) {
      INT_MDA_4D = new MultiDimensionalIntArray(SIZE_4D, SIZE_4D, SIZE_4D, SIZE_4D);
      for (int i = 0; i < SIZE_4D; ++i) {
        for (int j = 0; j < SIZE_4D; ++j) {
          final int base = i * SIZE_4D * SIZE_4D + j * SIZE_4D;
          for (int k = 0; k < SIZE_4D; ++k) {
            final int base2 = (base + k) * SIZE_4D;
            for (int l = 0; l < SIZE_4D; ++l) {
              INT_MDA_4D.setValue(base2 + l, i, j, k, l);
            }
          }
        }
      }
    }
    return INT_MDA_4D;
  }

  private static int runInt1dArr()
  {
    final OneDimensionalIntAccess array = getInt1dArray();
    int count = 0;
    for (int i = 0;  i < SIZE_1D;  ++i) {
      count += array.getValueAt(i);
    }
    return count;
  }

  @NotNull
  private static OneDimensionalIntAccess getInt1dArray()
  {
    if (INT_1D_ARR == null) {
      INT_1D_ARR = new OneDimensionalIntArray(SIZE_1D);
      for (int i = 0;  i < SIZE_1D;  ++i) {
        INT_1D_ARR.setValueAt(i, i);
      }
    }
    return INT_1D_ARR;
  }



  public void testLong1D()
  {
    time("Create LONG ARR 1D", SpeedTest::getLongArray1D);
    time("Access LONG ARR 1D", 16, SpeedTest::runLongArray1D);
    
    time("Create LONG MDA 1D", SpeedTest::getLongMda1D);
    time("Access LONG MDA 1D", 16, SpeedTest::runLongMda1D);
    
    time("Create LONG 1D", SpeedTest::getLong1dArray);
    time("Access LONG 1D", 16, SpeedTest::runLong1dArr);
  }
  
  private static long[] getLongArray1D()
  {
    if (LONG_ARRAY_1D == null) {
      LONG_ARRAY_1D = new long[SIZE_1D];
      for (int i = 0;  i < SIZE_1D;  ++i) {
        LONG_ARRAY_1D[i] = i;
      }
    }
    return LONG_ARRAY_1D;
  }

  private static long runLongArray1D()
  {
    final long[] array = getLongArray1D();
    long count = 0;
    for (int i = 0;  i < SIZE_1D;  ++i) {
      count += array[i];
    }
    return count;
  }
  

  private static long runLongMda1D()
  {
    final MultiDimensionalLongArray array = getLongMda1D();
    long count = 0;
    for (int i = 0;  i < SIZE_1D;  ++i) {
      count += array.getValueAt(i);
    }
    return count;
  }

  @NotNull
  private static MultiDimensionalLongArray getLongMda1D()
  {
    if (LONG_MDA_1D == null) {
      LONG_MDA_1D = new MultiDimensionalLongArray(SIZE_1D);
      for (int i = 0; i < SIZE_1D; ++i) {
        LONG_MDA_1D.setValueAt(i, i);
      }
    }
    return LONG_MDA_1D;
  }

  private static long runLong1dArr()
  {
    final OneDimensionalLongAccess array = getLong1dArray();
    long count = 0;
    for (int i = 0;  i < SIZE_1D;  ++i) {
      count += array.getValueAt(i);
    }
    return count;
  }

  @NotNull
  private static OneDimensionalLongAccess getLong1dArray()
  {
    if (LONG_1D_ARR == null) {
      LONG_1D_ARR = new OneDimensionalLongArray(SIZE_1D);
      for (int i = 0;  i < SIZE_1D;  ++i) {
        LONG_1D_ARR.setValueAt(i, i);
      }
    }
    return LONG_1D_ARR;
  }

  public void testLong2D()
  {
    time("Create LONG ARR 2D", SpeedTest::createLongArray2D);
    time("Access LONG ARR 2D", 16, SpeedTest::runLongArray2D);
    
    time("Create LONG MDA 2D", SpeedTest::createLongMda2D);
    time("Access LONG MDA 2D", 16, SpeedTest::runLongMda2D);
    
    time("Create LONG 2D", SpeedTest::createLong2dArr);
    time("Access LONG 2D", 16, SpeedTest::runLong2dArr);
  }

  private static long runLongArray2D()
  {
    final long[][] array = createLongArray2D();
    long count = 0;
    for (int i = 0;  i < SIZE_2D;  ++i) {
      for (int j = 0;  j < SIZE_2D;  ++j) {
        count += array[i][j];
      }
    }
    return count;
  }

  private static long[][] createLongArray2D()
  {
    if (LONG_ARRAY_2D == null) {
      LONG_ARRAY_2D = new long[SIZE_2D][SIZE_2D];
      for (int i = 0; i < SIZE_2D; ++i) {
        for (int j = 0; j < SIZE_2D; ++j) {
          LONG_ARRAY_2D[i][j] = i * SIZE_2D + j;
        }
      }
    }
    return LONG_ARRAY_2D;
  }

  private static long runLongMda2D()
  {
    final MultiDimensionalLongArray array = createLongMda2D();
    long count = 0;
    for (int i = 0;  i < SIZE_2D;  ++i) {
      for (int j = 0;  j < SIZE_2D;  ++j) {
        count += array.getValueAt(i, j);
      }
    }
    return count;
  }

  @NotNull
  private static MultiDimensionalLongArray createLongMda2D()
  {
    if (LONG_MDA_2D == null) {
      LONG_MDA_2D = new MultiDimensionalLongArray(SIZE_2D, SIZE_2D);
      for (int i = 0; i < SIZE_2D; ++i) {
        for (int j = 0; j < SIZE_2D; ++j) {
          LONG_MDA_2D.setValueAt(i * SIZE_2D + j, i, j);
        }
      }
    }
    return LONG_MDA_2D;
  }

  private static long runLong2dArr()
  {
    final TwoDimensionalLongAccess array = createLong2dArr();
    long count = 0;
    for (int i = 0;  i < SIZE_2D;  ++i) {
      for (int j = 0;  j < SIZE_2D;  ++j) {
        count += array.getValueAt(i, j);
      }
    }
    return count;
  }

  @NotNull
  private static TwoDimensionalLongAccess createLong2dArr()
  {
    if (LONG_2D_ARR == null) {
      LONG_2D_ARR = new TwoDimensionalLongArray(SIZE_2D, SIZE_2D);
      for (int i = 0; i < SIZE_2D; ++i) {
        for (int j = 0; j < SIZE_2D; ++j) {
          LONG_2D_ARR.setValueAt(i * SIZE_2D + j, i, j);
        }
      }
    }
    return LONG_2D_ARR;
  }

  public void testLong3D()
  {
    time("Create LONG ARR 3D", SpeedTest::createLongArray3D);
    time("Access LONG ARR 3D", 16, SpeedTest::runLongArray3D);
    
    time("Create LONG MDA 3D", SpeedTest::createLongMda3D);
    time("Access LONG MDA 3D", 16, SpeedTest::runLongMda3D);
  }

  private static long runLongArray3D()
  {
    final long[][][] array = createLongArray3D();
    long count = 0;
    for (int i = 0;  i < SIZE_3D;  ++i) {
      for (int j = 0;  j < SIZE_3D;  ++j) {
        for (int k = 0;  k < SIZE_3D;  ++k) {
          count += array[i][j][k];
        }
      }
    }
    return count;
  }

  private static long[][][] createLongArray3D()
  {
    if (LONG_ARRAY_3D == null) {
      LONG_ARRAY_3D = new long[SIZE_3D][SIZE_3D][SIZE_3D];
      for (int i = 0; i < SIZE_3D; ++i) {
        for (int j = 0; j < SIZE_3D; ++j) {
          final int base = i * SIZE_3D * SIZE_3D + j * SIZE_3D;
          for (int k = 0; k < SIZE_3D; ++k) {
            LONG_ARRAY_3D[i][j][k] = base + k;
          }
        }
      }
    }
    return LONG_ARRAY_3D;
  }

  private static long runLongMda3D()
  {
    final MultiDimensionalLongArray array = createLongMda3D();
    long count = 0;
    for (int i = 0;  i < SIZE_3D;  ++i) {
      for (int j = 0;  j < SIZE_3D;  ++j) {
        for (int k = 0;  k < SIZE_3D;  ++k) {
          count += array.getValueAt(i, j, k);
        }
      }
    }
    return count;
  }

  @NotNull
  private static MultiDimensionalLongArray createLongMda3D()
  {
    if (LONG_MDA_3D == null) {
      LONG_MDA_3D = new MultiDimensionalLongArray(SIZE_3D, SIZE_3D, SIZE_3D);
      for (int i = 0; i < SIZE_3D; ++i) {
        for (int j = 0; j < SIZE_3D; ++j) {
          final int base = i * SIZE_3D * SIZE_3D + j * SIZE_3D;
          for (int k = 0; k < SIZE_3D; ++k) {
            LONG_MDA_3D.setValueAt(base + k, i, j, k);
          }
        }
      }
    }
    return LONG_MDA_3D;
  }


  public void testLong4D()
  {
    time("Create LONG ARR 4D", SpeedTest::createLongArray4D);
    time("Access LONG ARR 4D", 16, SpeedTest::runLongArray4D);
    
    time("Create LONG MDA 4D", SpeedTest::createLongMda4D);
    time("Access LONG MDA 4D", 16, SpeedTest::runLongMda4D);
  }

  private static long runLongArray4D()
  {
    final long[][][][] array = createLongArray4D();
    long count = 0;
    for (int i = 0;  i < SIZE_4D;  ++i) {
      for (int j = 0;  j < SIZE_4D;  ++j) {
        for (int k = 0;  k < SIZE_4D;  ++k) {
          for (int l = 0;  l < SIZE_4D;  ++l) {
            count += array[i][j][k][l];
          }
        }
      }
    }
    return count;
  }

  private static long[][][][] createLongArray4D()
  {
    if (LONG_ARRAY_4D == null) {
      LONG_ARRAY_4D = new long[SIZE_4D][SIZE_4D][SIZE_4D][SIZE_4D];
      for (int i = 0; i < SIZE_4D; ++i) {
        for (int j = 0; j < SIZE_4D; ++j) {
          final int base = i * SIZE_4D * SIZE_4D + j * SIZE_4D;
          for (int k = 0; k < SIZE_4D; ++k) {
            final int base2 = (base + k) * SIZE_4D;
            for (int l = 0; l < SIZE_4D; ++l) {
              LONG_ARRAY_4D[i][j][k][l] = base2 + l;
            }
          }
        }
      }
    }
    return LONG_ARRAY_4D;
  }

  private static long runLongMda4D()
  {
    final MultiDimensionalLongArray array = createLongMda4D();
    long count = 0;
    for (int i = 0;  i < SIZE_4D;  ++i) {
      for (int j = 0;  j < SIZE_4D;  ++j) {
        for (int k = 0;  k < SIZE_4D;  ++k) {
          for (int l = 0;  l < SIZE_4D;  ++l) {
            count += array.getValueAt(i, j, k, l);
          }
        }
      }
    }
    return count;
  }

  @NotNull
  private static MultiDimensionalLongArray createLongMda4D()
  {
    if (LONG_MDA_4D == null) {
      LONG_MDA_4D = new MultiDimensionalLongArray(SIZE_4D, SIZE_4D, SIZE_4D, SIZE_4D);
      for (int i = 0; i < SIZE_4D; ++i) {
        for (int j = 0; j < SIZE_4D; ++j) {
          final int base = i * SIZE_4D * SIZE_4D + j * SIZE_4D;
          for (int k = 0; k < SIZE_4D; ++k) {
            final int base2 = (base + k) * SIZE_4D;
            for (int l = 0; l < SIZE_4D; ++l) {
              LONG_MDA_4D.setValueAt(base2 + l, i, j, k, l);
            }
          }
        }
      }
    }
    return LONG_MDA_4D;
  }

  public void testDouble1D()
  {
    time("Create DOUBLE ARR 1D", SpeedTest::getDoubleArray1D);
    time("Access DOUBLE ARR 1D", 16, SpeedTest::runDoubleArray1D);
    
    time("Create DOUBLE MDA 1D", SpeedTest::getDoubleMda1D);
    time("Access DOUBLE MDA 1D", 16, SpeedTest::runDoubleMda1D);
    
    time("Create DOUBLE 1D", SpeedTest::getDouble1dArray);
    time("Access DOUBLE 1D", 16, SpeedTest::runDouble1dArr);
  }
  
  private static double[] getDoubleArray1D()
  {
    if (DOUBLE_ARRAY_1D == null) {
      DOUBLE_ARRAY_1D = new double[SIZE_1D];
      for (int i = 0;  i < SIZE_1D;  ++i) {
        DOUBLE_ARRAY_1D[i] = i;
      }
    }
    return DOUBLE_ARRAY_1D;
  }

  private static double runDoubleArray1D()
  {
    final double[] array = getDoubleArray1D();
    double count = 0;
    for (int i = 0;  i < SIZE_1D;  ++i) {
      count += array[i];
    }
    return count;
  }
  

  private static double runDoubleMda1D()
  {
    final MultiDimensionalDoubleArray array = getDoubleMda1D();
    double count = 0;
    for (int i = 0;  i < SIZE_1D;  ++i) {
      count += array.getValue(i);
    }
    return count;
  }

  @NotNull
  private static MultiDimensionalDoubleArray getDoubleMda1D()
  {
    if (DOUBLE_MDA_1D == null) {
      DOUBLE_MDA_1D = new MultiDimensionalDoubleArray(SIZE_1D);
      for (int i = 0; i < SIZE_1D; ++i) {
        DOUBLE_MDA_1D.setValue(i, i);
      }
    }
    return DOUBLE_MDA_1D;
  }

  private static double runDouble1dArr()
  {
    final OneDimensionalDoubleAccess array = getDouble1dArray();
    double count = 0;
    for (int i = 0;  i < SIZE_1D;  ++i) {
      count += array.getValueAt(i);
    }
    return count;
  }

  @NotNull
  private static OneDimensionalDoubleAccess getDouble1dArray()
  {
    if (DOUBLE_1D_ARR == null) {
      DOUBLE_1D_ARR = new OneDimensionalDoubleArray(SIZE_1D);
      for (int i = 0;  i < SIZE_1D;  ++i) {
        DOUBLE_1D_ARR.setValueAt(i, i);
      }
    }
    return DOUBLE_1D_ARR;
  }

  public void testDouble2D()
  {
    time("Create DOUBLE ARR 2D", SpeedTest::createDoubleArray2D);
    time("Access DOUBLE ARR 2D", 16, SpeedTest::runDoubleArray2D);
    
    time("Create DOUBLE MDA 2D", SpeedTest::createDoubleMda2D);
    time("Access DOUBLE MDA 2D", 16, SpeedTest::runDoubleMda2D);

    time("Create DOUBLE 2D", SpeedTest::createDouble2dArr);
    time("Access DOUBLE 2D", 16, SpeedTest::runDouble2dArr);
  }

  private static double runDoubleArray2D()
  {
    final double[][] array = createDoubleArray2D();
    double count = 0;
    for (int i = 0;  i < SIZE_2D;  ++i) {
      for (int j = 0;  j < SIZE_2D;  ++j) {
        count += array[i][j];
      }
    }
    return count;
  }

  private static double[][] createDoubleArray2D()
  {
    if (DOUBLE_ARRAY_2D == null) {
      DOUBLE_ARRAY_2D = new double[SIZE_2D][SIZE_2D];
      for (int i = 0; i < SIZE_2D; ++i) {
        for (int j = 0; j < SIZE_2D; ++j) {
          DOUBLE_ARRAY_2D[i][j] = i * SIZE_2D + j;
        }
      }
    }
    return DOUBLE_ARRAY_2D;
  }

  private static double runDoubleMda2D()
  {
    final MultiDimensionalDoubleArray array = createDoubleMda2D();
    double count = 0;
    for (int i = 0;  i < SIZE_2D;  ++i) {
      for (int j = 0;  j < SIZE_2D;  ++j) {
        count += array.getValue(i, j);
      }
    }
    return count;
  }

  @NotNull
  private static MultiDimensionalDoubleArray createDoubleMda2D()
  {
    if (DOUBLE_MDA_2D == null) {
      DOUBLE_MDA_2D = new MultiDimensionalDoubleArray(SIZE_2D, SIZE_2D);
      for (int i = 0; i < SIZE_2D; ++i) {
        for (int j = 0; j < SIZE_2D; ++j) {
          DOUBLE_MDA_2D.setValue(i * SIZE_2D + j, i, j);
        }
      }
    }
    return DOUBLE_MDA_2D;
  }

  private static double runDouble2dArr()
  {
    final TwoDimensionalDoubleAccess array = createDouble2dArr();
    double count = 0;
    for (int i = 0;  i < SIZE_2D;  ++i) {
      for (int j = 0;  j < SIZE_2D;  ++j) {
        count += array.getValueAt(i, j);
      }
    }
    return count;
  }

  @NotNull
  private static TwoDimensionalDoubleAccess createDouble2dArr()
  {
    if (DOUBLE_2D_ARR == null) {
      DOUBLE_2D_ARR = new TwoDimensionalDoubleArray(SIZE_2D, SIZE_2D);
      for (int i = 0; i < SIZE_2D; ++i) {
        for (int j = 0; j < SIZE_2D; ++j) {
          DOUBLE_2D_ARR.setValueAt(i * SIZE_2D + j, i, j);
        }
      }
    }
    return DOUBLE_2D_ARR;
  }

  public void testDouble3D()
  {
    time("Create DOUBLE ARR 3D", SpeedTest::createDoubleArray3D);
    time("Access DOUBLE ARR 3D", 16, SpeedTest::runDoubleArray3D);
    
    time("Create DOUBLE MDA 3D", SpeedTest::createDoubleMda3D);
    time("Access DOUBLE MDA 3D", 16, SpeedTest::runDoubleMda3D);
  }

  private static double runDoubleArray3D()
  {
    final double[][][] array = createDoubleArray3D();
    double count = 0;
    for (int i = 0;  i < SIZE_3D;  ++i) {
      for (int j = 0;  j < SIZE_3D;  ++j) {
        for (int k = 0;  k < SIZE_3D;  ++k) {
          count += array[i][j][k];
        }
      }
    }
    return count;
  }

  private static double[][][] createDoubleArray3D()
  {
    if (DOUBLE_ARRAY_3D == null) {
      DOUBLE_ARRAY_3D = new double[SIZE_3D][SIZE_3D][SIZE_3D];
      for (int i = 0; i < SIZE_3D; ++i) {
        for (int j = 0; j < SIZE_3D; ++j) {
          final int base = i * SIZE_3D * SIZE_3D + j * SIZE_3D;
          for (int k = 0; k < SIZE_3D; ++k) {
            DOUBLE_ARRAY_3D[i][j][k] = base + k;
          }
        }
      }
    }
    return DOUBLE_ARRAY_3D;
  }

  private static double runDoubleMda3D()
  {
    final MultiDimensionalDoubleArray array = createDoubleMda3D();
    double count = 0;
    for (int i = 0;  i < SIZE_3D;  ++i) {
      for (int j = 0;  j < SIZE_3D;  ++j) {
        for (int k = 0;  k < SIZE_3D;  ++k) {
          count += array.getValue(i, j, k);
        }
      }
    }
    return count;
  }

  @NotNull
  private static MultiDimensionalDoubleArray createDoubleMda3D()
  {
    if (DOUBLE_MDA_3D == null) {
      DOUBLE_MDA_3D = new MultiDimensionalDoubleArray(SIZE_3D, SIZE_3D, SIZE_3D);
      for (int i = 0; i < SIZE_3D; ++i) {
        for (int j = 0; j < SIZE_3D; ++j) {
          final int base = i * SIZE_3D * SIZE_3D + j * SIZE_3D;
          for (int k = 0; k < SIZE_3D; ++k) {
            DOUBLE_MDA_3D.setValue(base + k, i, j, k);
          }
        }
      }
    }
    return DOUBLE_MDA_3D;
  }


  public void testDouble4D()
  {
    time("Create DOUBLE ARR 4D", SpeedTest::createDoubleArray4D);
    time("Access DOUBLE ARR 4D", 16, SpeedTest::runDoubleArray4D);
    
    time("Create DOUBLE MDA 4D", SpeedTest::createDoubleMda4D);
    time("Access DOUBLE MDA 4D", 16, SpeedTest::runDoubleMda4D);
  }

  private static double runDoubleArray4D()
  {
    final double[][][][] array = createDoubleArray4D();
    double count = 0;
    for (int i = 0;  i < SIZE_4D;  ++i) {
      for (int j = 0;  j < SIZE_4D;  ++j) {
        for (int k = 0;  k < SIZE_4D;  ++k) {
          for (int l = 0;  l < SIZE_4D;  ++l) {
            count += array[i][j][k][l];
          }
        }
      }
    }
    return count;
  }

  private static double[][][][] createDoubleArray4D()
  {
    if (DOUBLE_ARRAY_4D == null) {
      DOUBLE_ARRAY_4D = new double[SIZE_4D][SIZE_4D][SIZE_4D][SIZE_4D];
      for (int i = 0; i < SIZE_4D; ++i) {
        for (int j = 0; j < SIZE_4D; ++j) {
          final int base = i * SIZE_4D * SIZE_4D + j * SIZE_4D;
          for (int k = 0; k < SIZE_4D; ++k) {
            final int base2 = (base + k) * SIZE_4D;
            for (int l = 0; l < SIZE_4D; ++l) {
              DOUBLE_ARRAY_4D[i][j][k][l] = base2 + l;
            }
          }
        }
      }
    }
    return DOUBLE_ARRAY_4D;
  }

  private static double runDoubleMda4D()
  {
    final MultiDimensionalDoubleArray array = createDoubleMda4D();
    double count = 0;
    for (int i = 0;  i < SIZE_4D;  ++i) {
      for (int j = 0;  j < SIZE_4D;  ++j) {
        for (int k = 0;  k < SIZE_4D;  ++k) {
          for (int l = 0;  l < SIZE_4D;  ++l) {
            count += array.getValue(i, j, k, l);
          }
        }
      }
    }
    return count;
  }

  @NotNull
  private static MultiDimensionalDoubleArray createDoubleMda4D()
  {
    if (DOUBLE_MDA_4D == null) {
      DOUBLE_MDA_4D = new MultiDimensionalDoubleArray(SIZE_4D, SIZE_4D, SIZE_4D, SIZE_4D);
      for (int i = 0; i < SIZE_4D; ++i) {
        for (int j = 0; j < SIZE_4D; ++j) {
          final int base = i * SIZE_4D * SIZE_4D + j * SIZE_4D;
          for (int k = 0; k < SIZE_4D; ++k) {
            final int base2 = (base + k) * SIZE_4D;
            for (int l = 0; l < SIZE_4D; ++l) {
              DOUBLE_MDA_4D.setValue(base2 + l, i, j, k, l);
            }
          }
        }
      }
    }
    return DOUBLE_MDA_4D;
  }

  public void testExtract()
  {
    time("Create DOUBLE ARR 4D", SpeedTest::createDoubleArray4D);
    time("Xtract DOUBLE ARR 4D -> 1D", 16, () -> extractDoubleArray4D(41, 29, 17));

    time("Create DOUBLE MDA 4D", SpeedTest::createDoubleMda4D);
    time("Xccess DOUBLE MDA 4D -> 1D" , 16, () -> extractDoubleMda4D(41, 29, 17));

  }

  private static double extractDoubleArray4D(int i2, int i3, int i4)
  {
    final double[][][][] array = createDoubleArray4D();
    final double[] result = new double[SIZE_4D];
    for (int i = 0;  i < SIZE_4D;  ++i) {
      result[i] = array[i][i2][i3][i4]; 
    }
    double count = 0;
    for (int i = 0;  i < SIZE_4D;  ++i) {
      count += result[i]; 
    }
    return DCOUNT;
  }

  public double extractDoubleMda4D(int i2, int i3, int i4)
  {
    final MultiDimensionalDoubleArray array = createDoubleMda4D();
    final MultiDimensionalDoubleArray result = array.sub(OPEN, i2, i3, i4);
    double count = 0;
    for (int i = 0;  i < SIZE_4D;  ++i) {
      count += result.getElement(i);
    }
    return DCOUNT;
  }
}
