// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.mda;

import de.caff.generics.Types;
import junit.framework.TestCase;

import java.util.List;

import static de.caff.generics.mda.MultiIndexLinearizer.OPEN;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 02, 2019
 */
public class MultiDimTest
        extends TestCase
{
  private static final List<Integer> LINEAR =
          // z:         0   1   // x     y
          Types.asList(0, 1,  // 0     0
                       2, 3,  // 0     1
                       4, 5,  // 0     2

                       6, 7,  // 1     0
                       8, 9,  // 1     1
                       10, 11, // 1     2

                       12, 13, // 2     0
                       14, 15, // 2     1
                       16, 17, // 2     2

                       18, 19, // 3     0
                       20, 21, // 3     1
                       22, 23);// 3     2

  public void testBasic()
  {
    final MultiDimensionalArray<Integer> array432 = new MultiDimensionalArray<>(4, 3, 2);
    array432.setFrom(LINEAR);

    assertTrue(Types.areEqual(LINEAR, array432.toList()));

    assertEquals(array432.getElement(3, 2, 1).intValue(), 23);

    assertTrue(Types.areEqual(Types.asList(6,7,  8,9, 10,11),
                              array432.sub(1).toList()));

    assertTrue(Types.areEqual(Types.asList(4,5, 10,11, 16,17, 22,23),
                              array432.sub(OPEN, 2)));

    assertTrue(Types.areEqual(Types.asList(0,2,4, 6,8,10, 12,14,16, 18,20,22),
                              array432.sub(OPEN, OPEN, 0)));

    assertTrue(Types.areEqual(Types.asList(20, 21),
                              array432.sub(3, 1)));

    assertEquals(array432.sub(3, 1), array432.sub(3).sub(1));;
  }

  public void testUserSub()
  {
    final MultiDimensionalArray<Integer> array432 = new MultiDimensionalArray<>(4, 3, 2);
    array432.setAll(idx -> array432.getIndexLinearizer().toLinear(idx));

    assertTrue(Types.areEqual(LINEAR, array432.toList()));

    final MultiDimensionalArray<Integer> rotated =
            new MultiDimensionalArray<>(array432,
                                        idx -> new int[] { idx[1], idx[2], idx[0] },
                                        2, 4, 3);
    for (int x = 0;  x < 4;  ++x) {
      for (int y = 0;  y < 3;  ++y) {
        for (int z = 0;  z < 2;  ++z) {
          assertSame(array432.getElement(x, y, z),
                     rotated.getElement(z, x, y));
        }
      }
    }
  }
}
