// ============================================================================
// File:               TestTimSort
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            1/16/23 3:09 PM
//=============================================================================
package de.caff.generics.algorithm;

import de.caff.annotation.NotNull;
import de.caff.generics.*;
import de.caff.generics.function.IntOrdering;
import de.caff.generics.function.Ordering;
import junit.framework.TestCase;

import java.util.Arrays;
import java.util.Comparator;
import java.util.Random;

/**
 * Test our implementation of TimSort.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 16, 2023
 */
public class TestTimSort
        extends TestCase
{
  @NotNull
  static String[] makeRandom(long seed, int len, int width)
  {
    assert width > 0  &&  width <= 16;
    final Random random = new Random(seed);
    final String[] result = new String[len];
    for (int i = 0;  i < len;  ++i) {
      result[i] = String.format("%016x", random.nextLong()).substring(0, width);
    }
    return result;
  }

  private static void checkRandom(long seed, int len)
  {
    checkRandom(seed, len, 16);
  }

  private static void checkRandom(long seed, int len, int width)
  {
    System.out.printf("seed=%d, len=%d, width=%d\n", seed, len, width);
    checkArray(makeRandom(seed, len, width));
  }

  /**
   * Find the first difference between the correctly sorted array
   * and our buggy implementation.
   * @param refArray  correctly sorted reference array
   * @param testArray hopefully sorted array
   * @return location of the first difference, or {@code -1} if there are no
   *         differences
   */
  static int findFirstDiff(@NotNull String[] refArray,
                           @NotNull String[] testArray)
  {
    assertEquals(refArray.length, testArray.length);
    for (int i = 0;  i < refArray.length;  ++i) {
      if (!refArray[i].equals(testArray[i])) {
        return i;
      }
    }
    return -1;
  }

  private static void checkArray(@NotNull String[] array)
  {
    final String[] forIndexable = array.clone();

    Arrays.sort(array, String::compareTo);
    TimSort.sort(MutableIndexable.viewArray(forIndexable), Ordering.natural());
    assertEquals(-1, findFirstDiff(array, forIndexable));
  }

  /**
   * Test 2^16 simple random arrays with no same elements expected.
   */
  public void testSimple()
  {
    for (int len = 1;  len < 0x10000;  ++len) {
      checkRandom(len, len);
    }
  }

  public void testSimple257()
  {
    checkRandom(257, 257);
  }

  /**
   * Test 2^16 random arrays with a decent probability of equal elements.
   */
  public void testDecentDup()
  {
    for (int len = 0;  len < 0x10;  ++len) {
      checkRandom(len, len, 1);
    }
    for (int len = 0x10;  len < 0x100;  ++len) {
      checkRandom(len, len, 2);
    }
    for (int len = 0x100;  len < 0x1000;  ++len) {
      checkRandom(len, len, 3);
    }
    for (int len = 0x1000;  len < 0x10000;  ++len) {
      checkRandom(len, len, 4);
    }
  }

  public void testHighDup()
  {
    for (int len = 0x10;  len < 0x10000;  ++len) {
      checkRandom(len, len, 2);
    }
  }

  static long measureDuration(@NotNull Runnable operation)
  {
    final long start = System.nanoTime();
    operation.run();
    return System.nanoTime() - start;
  }


  /**
   * Test the duration of the de.caff vs the Java implementation.
   * Running this as a unit test is stupid (e.g. assertions are enabled),
   * but this is only a quick hit to see how we perform.
   */
  public void testDuration()
  {
    System.out.println("\nSTRING");
    // todo: include warmup
    final int size = 0x100000;
    final int repeats = 16;
    final String[] array = makeRandom(42, size, 6);
    final String[] work = new String[size];
    final MutableIndexable<String> mi = MutableIndexable.viewArray(work);

    long durationJava = 0;
    long durationDeCaffTS = 0;
    long durationDeCaffDPQ = 0;
    for (int i = 0;  i < repeats;  ++i) {
      System.arraycopy(array, 0, work, 0, size);
      durationDeCaffTS += measureDuration(() -> TimSort.sort(mi));

      System.arraycopy(array, 0, work, 0, size);
      durationDeCaffDPQ += measureDuration(() -> DualPivotQuicksort.sort(mi, Ordering.natural()));

      System.arraycopy(array, 0, work, 0, size);
      durationJava += measureDuration(() -> Arrays.sort(work, Comparator.naturalOrder()));
    }

    // make sure this is not compiled away
    System.arraycopy(work, 0, array, 0, size);

    System.out.printf("Java:         %10d ns\n", durationJava / repeats);
    System.out.printf("DeCaff (TS):  %10d ns\n", durationDeCaffTS / repeats);
    System.out.printf("DeCaff (DPQ): %10d ns\n", durationDeCaffDPQ / repeats);
    System.out.printf("Penalty (TS vs J):   %.2f %%\n", durationDeCaffTS * 100.0 / durationJava - 100);
    System.out.printf("Penalty (DPQ vs J):  %.2f %%\n", durationDeCaffDPQ * 100.0 / durationJava - 100);
    System.out.printf("Penalty (DPQ vs TS): %.2f %%\n", durationDeCaffDPQ * 100.0 / durationDeCaffTS - 100);
  }

  @NotNull
  static double[] makeRandomDouble(int seed, int len)
  {
    final Random random = new Random(seed);
    final double[] result = new double[len];
    for (int i = 0;  i < len;  ++i) {
      result[i] = random.nextDouble();
    }
    return result;
  }

  public void testDurationDouble()
  {
    System.out.println("\nDOUBLE");
    final int size = 0x100000;
    final int repeats = 16;
    final double[] array = makeRandomDouble(43, size);
    final double[] work = new double[size];
    final MutableDoubleIndexable mi = MutableDoubleIndexable.viewArray(work);

    long durationJava = 0;
    long durationDeCaff = 0;
    for (int i = 0;  i < repeats;  ++i) {
      System.arraycopy(array, 0, work, 0, size);
      durationDeCaff += measureDuration(() -> TimSortDouble.sort(mi));

      System.arraycopy(array, 0, work, 0, size);
      durationJava += measureDuration(() -> Arrays.sort(work));
    }

    // make sure this is not compiled away
    System.arraycopy(work, 0, array, 0, size);

    System.out.printf("Java:    %10d ns\n", durationJava / repeats);
    System.out.printf("DeCaff:  %10d ns\n", durationDeCaff / repeats);
    System.out.printf("Penalty: %.2f %%\n", durationDeCaff * 100.0 / durationJava - 100);

  }

  @NotNull
  static float[] makeRandomFloat(int seed, int len)
  {
    final Random random = new Random(seed);
    final float[] result = new float[len];
    for (int i = 0;  i < len;  ++i) {
      result[i] = random.nextFloat();
    }
    return result;
  }

  public void testDurationFloat()
  {
    System.out.println("\nFLOAT");
    final int size = 0x100000;
    final int repeats = 16;
    final float[] array = makeRandomFloat(44, size);
    final float[] work = new float[size];
    final MutableFloatIndexable mi = MutableFloatIndexable.viewArray(work);

    long durationJava = 0;
    long durationDeCaff = 0;
    for (int i = 0;  i < repeats;  ++i) {
      System.arraycopy(array, 0, work, 0, size);
      durationDeCaff += measureDuration(() -> TimSortFloat.sort(mi));

      System.arraycopy(array, 0, work, 0, size);
      durationJava += measureDuration(() -> Arrays.sort(work));
    }

    // make sure this is not compiled away
    System.arraycopy(work, 0, array, 0, size);

    System.out.printf("Java:    %10d ns\n", durationJava / repeats);
    System.out.printf("DeCaff:  %10d ns\n", durationDeCaff / repeats);
    System.out.printf("Penalty: %.2f %%\n", durationDeCaff * 100.0 / durationJava - 100);

  }

  @NotNull
  static int[] makeRandomInt(int seed, int len)
  {
    final Random random = new Random(seed);
    final int[] result = new int[len];
    for (int i = 0;  i < len;  ++i) {
      result[i] = random.nextInt();
    }
    return result;
  }

  public void testDurationInt()
  {
    System.out.println("\nINT");
    final int size = 0x100000;
    final int repeats = 16;
    final int[] array = makeRandomInt(45, size);
    final int[] work = new int[size];
    final MutableIntIndexable mi = MutableIntIndexable.viewArray(work);

    long durationJava = 0;
    long durationDeCaffDPQ = 0;
    long durationDeCaffDPQ2 = 0;
    long durationDeCaffTS = 0;
    for (int i = 0;  i < repeats;  ++i) {
      System.arraycopy(array, 0, work, 0, size);
      durationDeCaffTS += measureDuration(() -> TimSortInt.sort(mi));

      System.arraycopy(array, 0, work, 0, size);
      durationDeCaffDPQ += measureDuration(() -> DualPivotQuicksort.sort(work, IntOrdering.ASCENDING));

      System.arraycopy(array, 0, work, 0, size);
      durationDeCaffDPQ2 += measureDuration(() -> DualPivotQuicksort.sort(mi));

      System.arraycopy(array, 0, work, 0, size);
      durationJava += measureDuration(() -> Arrays.sort(work));
    }

    // make sure this is not compiled away
    System.arraycopy(work, 0, array, 0, size);

    System.out.printf("Java:                %10d ns\n", durationJava / repeats);
    System.out.printf("DeCaff (TS):         %10d ns\n", durationDeCaffTS / repeats);
    System.out.printf("DeCaff (DPQ,array):  %10d ns\n", durationDeCaffDPQ / repeats);
    System.out.printf("DeCaff (DPQ,index):  %10d ns\n", durationDeCaffDPQ2 / repeats);
    System.out.printf("Penalty (TS):        %.2f %%\n", durationDeCaffTS * 100.0 / durationJava - 100);
    System.out.printf("Penalty (DPQ,arr):   %.2f %%\n", durationDeCaffDPQ * 100.0 / durationJava - 100);
    System.out.printf("Penalty (DPQ,idx):   %.2f %%\n", durationDeCaffDPQ2 * 100.0 / durationJava - 100);
    System.out.printf("Penalty (idx:arr):   %.2f %%\n", durationDeCaffDPQ2 * 100.0 / durationDeCaffDPQ - 100);

  }

  @NotNull
  static long[] makeRandomLong(int seed, int len)
  {
    final Random random = new Random(seed);
    final long[] result = new long[len];
    for (int i = 0;  i < len;  ++i) {
      result[i] = random.nextLong();
    }
    return result;
  }

  public void testDurationLong()
  {
    System.out.println("\nLONG");
    final int size = 0x100000;
    final int repeats = 16;
    final long[] array = makeRandomLong(46, size);
    final long[] work = new long[size];
    final MutableLongIndexable mi = MutableLongIndexable.viewArray(work);

    long durationJava = 0;
    long durationDeCaff = 0;
    for (int i = 0;  i < repeats;  ++i) {
      System.arraycopy(array, 0, work, 0, size);
      durationDeCaff += measureDuration(() -> TimSortLong.sort(mi));

      System.arraycopy(array, 0, work, 0, size);
      durationJava += measureDuration(() -> Arrays.sort(work));
    }

    // make sure this is not compiled away
    System.arraycopy(work, 0, array, 0, size);

    System.out.printf("Java:         %10d ns\n", durationJava / repeats);
    System.out.printf("DeCaff (TS):  %10d ns\n", durationDeCaff / repeats);
    System.out.printf("Penalty: %.2f %%\n", durationDeCaff * 100.0 / durationJava - 100);
  }

  public static void main(String[] args)
  {
    final TestTimSort testTimSort = new TestTimSort();

    //testTimSort.testDuration();
    //testTimSort.testDurationDouble();
    //testTimSort.testDurationFloat();
    //testTimSort.testDurationLong();
    testTimSort.testDurationInt();
  }

  //STRING
  //Java:     314041551 ns
  //DeCaff:   315798779 ns
  //Penalty: 0.56 %
  //
  //DOUBLE
  //Java:      71004227 ns
  //DeCaff:   116592583 ns
  //Penalty: 64.21 %
  //
  //FLOAT
  //Java:      73000467 ns
  //DeCaff:   123605992 ns
  //Penalty: 69.32 %
  //
  //LONG
  //Java:      52397024 ns
  //DeCaff:    91555861 ns
  //Penalty: 74.73 %
  //
  //INT
  //Java:      56203854 ns
  //DeCaff:    92335674 ns
  //Penalty: 64.29 %
  //
  //SHORT (w/ removed TimSortShort)
  //Java:       1626256 ns
  //DeCaff:    89863576 ns
  //Penalty: 5425.80 %
  //
  //BYTE (w/ removed TimSortByte)
  //Java:       1031414 ns
  //DeCaff:    75569849 ns
  //Penalty: 7226.82 %
  //
  //CHAR (w/ removed TimSortCHar)
  //Java:       1571332 ns
  //DeCaff:    86886888 ns
  //Penalty: 5429.50 %
}
