// ============================================================================
// File:               TestFastSort
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            1/25/23 3:36 PM
//=============================================================================
package de.caff.generics.algorithm;

import de.caff.annotation.NotNull;
import de.caff.generics.MutableByteIndexable;
import de.caff.generics.MutableCharIndexable;
import junit.framework.TestCase;

import java.util.Arrays;
import java.util.Random;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 25, 2023
 */
public class TestFastSort
        extends TestCase
{
  @NotNull
  static byte[] makeRandomBytes(long seed, int len)
  {
    final Random random = new Random(seed);
    final byte[] result = new byte[len];
    random.nextBytes(result);
    return result;
  }

  @NotNull
  static short[] makeRandomShorts(long seed, int len)
  {
    final Random random = new Random(seed);
    final short[] result = new short[len];
    for (int i = 0;  i < len;  ++i) {
      result[i] = (short)random.nextInt();
    }
    return result;
  }

  @NotNull
  static char[] makeRandomChars(long seed, int len)
  {
    final Random random = new Random(seed);
    final char[] result = new char[len];
    for (int i = 0;  i < len;  ++i) {
      result[i] = (char)random.nextInt();
    }
    return result;
  }

  private static void checkRandom(long seed, int len)
  {
    checkArray(makeRandomBytes(seed, len));
  }

  private static void checkArray(@NotNull byte[] array)
  {
    final byte[] forIndexable = array.clone();

    Arrays.sort(array);
    FastSortByte.sort(MutableByteIndexable.viewArray(forIndexable));
    assertEquals(-1, findFirstDiff(array, forIndexable));
  }

  /**
   * Find the first difference between the correctly sorted array
   * and our buggy implementation.
   * @param refArray  correctly sorted reference array
   * @param testArray hopefully sorted array
   * @return location of the first difference, or {@code -1} if there are no
   *         differences
   */
  private static int findFirstDiff(@NotNull byte[] refArray,
                                   @NotNull byte[] testArray)
  {
    assertEquals(refArray.length, testArray.length);
    for (int i = 0;  i < refArray.length;  ++i) {
      if (refArray[i] != testArray[i]) {
        return i;
      }
    }
    return -1;
  }


  public void testByteSort()
  {
    for (int len = 1;  len < 0x10000;  ++len) {
      System.out.println(len);
      checkRandom(len, len);
    }
  }

  public void testByteSort5()
  {
    checkRandom(5, 5);
  }

  public void testDurationByte()
  {
    System.out.println("\nBYTE");
    final int size = 0x100000;
    final int repeats = 16;
    final byte[] array = makeRandomBytes(47, size);
    final byte[] work = new byte[size];
    final MutableByteIndexable mi = MutableByteIndexable.viewArray(work);

    long durationJava = 0;
    long durationDeCaff = 0;
    for (int i = 0;  i < repeats;  ++i) {
      System.arraycopy(array, 0, work, 0, size);
      durationDeCaff += TestTimSort.measureDuration(() -> FastSortByte.sort(mi));

      System.arraycopy(array, 0, work, 0, size);
      durationJava += TestTimSort.measureDuration(() -> Arrays.sort(work));
    }

    // make sure this is not compiled away
    System.arraycopy(work, 0, array, 0, size);

    System.out.printf("Java:    %10d ns\n", durationJava / repeats);
    System.out.printf("DeCaff:  %10d ns\n", durationDeCaff / repeats);
    System.out.printf("Penalty: %.2f %%\n", durationDeCaff * 100.0 / durationJava - 100);
  }

  public void testDurationChar()
  {
    if (true) return;  // this test fails w/ StackOverflowError
    System.out.println("\nCHAR");
    final int size = 0x100000;
    final int repeats = 16;
    final char[] array = makeRandomChars(48, size);
    final char[] work = new char[size];
    final MutableCharIndexable mi = MutableCharIndexable.viewArray(work);

    long durationJava = 0;
    long durationDeCaff = 0;
    for (int i = 0;  i < repeats;  ++i) {
      System.arraycopy(array, 0, work, 0, size);
      durationDeCaff += TestTimSort.measureDuration(() -> FastSortChar.sort(mi));

      System.arraycopy(array, 0, work, 0, size);
      durationJava += TestTimSort.measureDuration(() -> Arrays.sort(work));
    }

    // make sure this is not compiled away
    System.arraycopy(work, 0, array, 0, size);

    System.out.printf("Java:    %10d ns\n", durationJava / repeats);
    System.out.printf("DeCaff:  %10d ns\n", durationDeCaff / repeats);
    System.out.printf("Penalty: %.2f %%\n", durationDeCaff * 100.0 / durationJava - 100);
  }

  public static void main(String[] args)
  {
    final TestFastSort tfs = new TestFastSort();
    tfs.testDurationByte();
    tfs.testDurationChar();
  }
}
