// ============================================================================
// File:               TestDualPivotQuicksort
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            1/26/23 2:34 PM
//=============================================================================
package de.caff.generics.algorithm;

import de.caff.annotation.NotNull;
import de.caff.generics.*;
import de.caff.generics.function.*;
import junit.framework.TestCase;

import java.util.*;

import static de.caff.generics.algorithm.TestTimSort.measureDuration;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 26, 2023
 */
public class TestDualPivotQuicksort
        extends TestCase
{
  public void testIntAscending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final int[] array = TestTimSort.makeRandomInt(i, i);
      final int[] test = array.clone();

      Arrays.sort(array);
      DualPivotQuicksort.sort(test, IntOrdering.ASCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testIntDescending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final int[] array = TestTimSort.makeRandomInt(i, i);
      final int[] test = array.clone();

      Arrays.sort(array);
      MutableIntIndexable.viewArray(array).revert();
      DualPivotQuicksort.sort(test, IntOrdering.DESCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testLongAscending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final long[] array = TestTimSort.makeRandomLong(i, i);
      final long[] test = array.clone();

      Arrays.sort(array);
      DualPivotQuicksort.sort(test, LongOrdering.ASCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testLongDescending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final long[] array = TestTimSort.makeRandomLong(i, i);
      final long[] test = array.clone();

      Arrays.sort(array);
      MutableLongIndexable.viewArray(array).revert();
      DualPivotQuicksort.sort(test, LongOrdering.DESCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testULongAscending47()
  {
    final long[] array = TestTimSort.makeRandomLong(47, 47);
    final long[] test = array.clone();

    Arrays.sort(array);
    int split = Arrays.binarySearch(array, 0);
    if (split < 0) {
      split = -split - 1;
    }
    if (split < array.length) {
      final long[] tmp = new long[split];
      System.arraycopy(array, 0, tmp, 0, split);
      System.arraycopy(array, split, array, 0, array.length - split);
      System.arraycopy(tmp, 0, array, array.length - split, tmp.length);
    }
    else {
      final long[] tmp = new long[array.length - split];
      System.arraycopy(array, split, tmp, 0, tmp.length);
      System.arraycopy(array, 0, array, tmp.length, split);
      System.arraycopy(tmp, 0, array, 0, tmp.length);
    }
    DualPivotQuicksort.sort(test, LongOrdering.UNSIGNED_ASCENDING);

    assertEquals(-1, findFirstDiff(array, test));
  }

  public void testShortAscending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final short[] array = TestFastSort.makeRandomShorts(i, i);
      final short[] test = array.clone();

      Arrays.sort(array);
      DualPivotQuicksort.sort(test, ShortOrdering.ASCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testShortDescending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final short[] array = TestFastSort.makeRandomShorts(i, i);
      final short[] test = array.clone();

      Arrays.sort(array);
      MutableShortIndexable.viewArray(array).revert();
      DualPivotQuicksort.sort(test, ShortOrdering.DESCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testCharAscending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final char[] array = TestFastSort.makeRandomChars(i, i);
      final char[] test = array.clone();

      Arrays.sort(array);
      DualPivotQuicksort.sort(test, CharOrdering.ASCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testCharDescending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final char[] array = TestFastSort.makeRandomChars(i, i);
      final char[] test = array.clone();

      Arrays.sort(array);
      MutableCharIndexable.viewArray(array).revert();
      DualPivotQuicksort.sort(test, CharOrdering.DESCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testByteAscending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final byte[] array = TestFastSort.makeRandomBytes(i, i);
      final byte[] test = array.clone();

      Arrays.sort(array);
      DualPivotQuicksort.sort(test, ByteOrdering.ASCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testByteDescending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final byte[] array = TestFastSort.makeRandomBytes(i, i);
      final byte[] test = array.clone();

      Arrays.sort(array);
      MutableByteIndexable.viewArray(array).revert();
      DualPivotQuicksort.sort(test, ByteOrdering.DESCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testFloatAscending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final float[] array = TestTimSort.makeRandomFloat(i, i);
      final float[] test = array.clone();

      Arrays.sort(array);
      DualPivotQuicksort.sort(test, FloatOrdering.STANDARD_ASCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testFloatDescending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final float[] array = TestTimSort.makeRandomFloat(i, i);
      final float[] test = array.clone();

      Arrays.sort(array);
      MutableFloatIndexable.viewArray(array).revert();
      DualPivotQuicksort.sort(test, FloatOrdering.STANDARD_DESCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testDoubleAscending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final double[] array = TestTimSort.makeRandomDouble(i, i);
      final double[] test = array.clone();

      Arrays.sort(array);
      DualPivotQuicksort.sort(test, DoubleOrdering.STANDARD_ASCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testDoubleDescending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final double[] array = TestTimSort.makeRandomDouble(i, i);
      final double[] test = array.clone();

      Arrays.sort(array);
      MutableDoubleIndexable.viewArray(array).revert();
      DualPivotQuicksort.sort(test, DoubleOrdering.STANDARD_DESCENDING);

      assertEquals(-1, findFirstDiff(array, test));
    }
  }

  public void testStringAscending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final String[] array = TestTimSort.makeRandom(i, i, 16);
      final String[] test = array.clone();

      Arrays.sort(array);
      DualPivotQuicksort.sort(MutableIndexable.viewArray(test), Ordering.natural());

      assertEquals(-1, TestTimSort.findFirstDiff(array, test));
    }
  }

  public void testStringDescending()
  {
    for (int i = 1;   i < 0x10000;  ++i) {
      final String[] array = TestTimSort.makeRandom(i, i, 16);
      final String[] test = array.clone();

      Arrays.sort(array);
      MutableIndexable.viewArray(array).revert();
      DualPivotQuicksort.sort(MutableIndexable.viewArray(test),
                              Ordering.inverseNatural());

      assertEquals(-1, TestTimSort.findFirstDiff(array, test));
    }
  }

  /**
   * Find the first difference between the correctly sorted array
   * and our buggy implementation.
   * @param refArray  correctly sorted reference array
   * @param testArray hopefully sorted array
   * @return location of the first difference, or {@code -1} if there are no
   *         differences
   */
  static int findFirstDiff(@NotNull int[] refArray,
                           @NotNull int[] testArray)
  {
    assertEquals(refArray.length, testArray.length);
    for (int i = 0;  i < refArray.length;  ++i) {
      if (refArray[i] != testArray[i]) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Find the first difference between the correctly sorted array
   * and our buggy implementation.
   * @param refArray  correctly sorted reference array
   * @param testArray hopefully sorted array
   * @return location of the first difference, or {@code -1} if there are no
   *         differences
   */
  static int findFirstDiff(@NotNull long[] refArray,
                           @NotNull long[] testArray)
  {
    assertEquals(refArray.length, testArray.length);
    for (int i = 0;  i < refArray.length;  ++i) {
      if (refArray[i] != testArray[i]) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Find the first difference between the correctly sorted array
   * and our buggy implementation.
   * @param refArray  correctly sorted reference array
   * @param testArray hopefully sorted array
   * @return location of the first difference, or {@code -1} if there are no
   *         differences
   */
  static int findFirstDiff(@NotNull short[] refArray,
                           @NotNull short[] testArray)
  {
    assertEquals(refArray.length, testArray.length);
    for (int i = 0;  i < refArray.length;  ++i) {
      if (refArray[i] != testArray[i]) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Find the first difference between the correctly sorted array
   * and our buggy implementation.
   * @param refArray  correctly sorted reference array
   * @param testArray hopefully sorted array
   * @return location of the first difference, or {@code -1} if there are no
   *         differences
   */
  static int findFirstDiff(@NotNull char[] refArray,
                           @NotNull char[] testArray)
  {
    assertEquals(refArray.length, testArray.length);
    for (int i = 0;  i < refArray.length;  ++i) {
      if (refArray[i] != testArray[i]) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Find the first difference between the correctly sorted array
   * and our buggy implementation.
   * @param refArray  correctly sorted reference array
   * @param testArray hopefully sorted array
   * @return location of the first difference, or {@code -1} if there are no
   *         differences
   */
  static int findFirstDiff(@NotNull byte[] refArray,
                           @NotNull byte[] testArray)
  {
    assertEquals(refArray.length, testArray.length);
    for (int i = 0;  i < refArray.length;  ++i) {
      if (refArray[i] != testArray[i]) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Find the first difference between the correctly sorted array
   * and our buggy implementation.
   * @param refArray  correctly sorted reference array
   * @param testArray hopefully sorted array
   * @return location of the first difference, or {@code -1} if there are no
   *         differences
   */
  static int findFirstDiff(@NotNull float[] refArray,
                           @NotNull float[] testArray)
  {
    assertEquals(refArray.length, testArray.length);
    for (int i = 0;  i < refArray.length;  ++i) {
      if (refArray[i] != testArray[i]) {
        return i;
      }
    }
    return -1;
  }

  /**
   * Find the first difference between the correctly sorted array
   * and our buggy implementation.
   * @param refArray  correctly sorted reference array
   * @param testArray hopefully sorted array
   * @return location of the first difference, or {@code -1} if there are no
   *         differences
   */
  static int findFirstDiff(@NotNull double[] refArray,
                           @NotNull double[] testArray)
  {
    assertEquals(refArray.length, testArray.length);
    for (int i = 0; i < refArray.length; ++i) {
      if (refArray[i] != testArray[i]) {
        return i;
      }
    }
    return -1;
  }

  public static void main(String[] args)
  {
    final int minRepeats = 4;
    final int maxRepeats = 64;
    final int shuffleEach = 8;
    for (int size = 0x100;  size > 0;  size <<= 4) {
      final int repeats = Math.min(maxRepeats,
                                   Math.max(minRepeats, 0x01000000 / size));
      for (int w = 4;  w <= 16;  w <<= 1) {
        final String[] array = TestTimSort.makeRandom(size, size, w);

        testDuration("RANDOM", array, w, repeats);

        // array is now sorted
        testDuration("SORTED", array, w, repeats);

        final int shuffleSize = size / shuffleEach;
        final MutableIndexable<String> forShuffle =
                new MutableIndexable<String>()
                {
                  @Override
                  public void set(int index, String elem)
                  {
                    array[index * shuffleEach] = elem;
                  }

                  @Override
                  public String get(int index)
                  {
                    return array[index * shuffleEach];
                  }

                  @Override
                  public int size()
                  {
                    return shuffleSize;
                  }
                };
        Collections.shuffle(forShuffle.asList(), new Random(size));

        // array is now partly (every shuffleEach'ed element) shuffled
        testDuration("SHUFFLED", array, w, repeats);
        System.out.println("---------------------------------------------------------------------------");
      }
      System.out.println("---------------------------------------------------------------------------");
    }
  }

  @NotNull
  private static LongPair testDuration(@NotNull String title,
                                       @NotNull String[] array,
                                       int width,
                                       int repeats)
  {
    final String[] work = new String[array.length];
    final MutableIndexable<String> mi = MutableIndexable.viewArray(work);

    long durationJava = 0;
    long durationTS = 0;
    long durationDPQ = 0;
    for (int i = 0;  i < repeats;  ++i) {
      System.arraycopy(array, 0, work, 0, array.length);
      durationTS += measureDuration(() -> TimSort.sort(mi));

      System.arraycopy(array, 0, work, 0, array.length);
      durationDPQ += measureDuration(() -> DualPivotQuicksort.sort(mi, Ordering.natural()));

      System.arraycopy(array, 0, work, 0, array.length);
      durationJava += measureDuration(() -> Arrays.sort(work, Comparator.naturalOrder()));
    }

    // make sure this is not compiled away
    System.arraycopy(work, 0, array, 0, array.length);

    System.out.printf("%s (size=%d, width=%d, repeats=%d)\n",
                      title, array.length, width, repeats);
    System.out.printf("Java:         %10d ns\n", durationJava / repeats);
    System.out.printf("DeCaff (TS):  %10d ns\n", durationTS / repeats);
    System.out.printf("DeCaff (DPQ): %10d ns\n", durationDPQ / repeats);
    System.out.printf("Penalty (TS vs J):   %4.2f %%\n", durationTS * 100.0 / durationJava - 100);
    System.out.printf("Penalty (DPQ vs J):  %4.2f %%\n", durationDPQ * 100.0 / durationJava - 100);
    System.out.printf("Penalty (DPQ vs TS): %4.2f %%\n", durationDPQ * 100.0 / durationTS - 100);
    System.out.println();
    
    return new LongPair(durationTS, durationDPQ);
  }
}

//RANDOM (size=256, width=4, repeats=64)
//Java:              59658 ns
//DeCaff (TS):      210222 ns
//DeCaff (DPQ):      96657 ns
//Penalty (TS vs J):   252.38 %
//Penalty (DPQ vs J):  62.02 %
//Penalty (DPQ vs TS): -54.02 %
//
//SORTED (size=256, width=4, repeats=64)
//Java:               4003 ns
//DeCaff (TS):        5033 ns
//DeCaff (DPQ):      14370 ns
//Penalty (TS vs J):   25.72 %
//Penalty (DPQ vs J):  258.91 %
//Penalty (DPQ vs TS): 185.50 %
//
//SHUFFLED (size=256, width=4, repeats=64)
//Java:              10091 ns
//DeCaff (TS):       11270 ns
//DeCaff (DPQ):      19421 ns
//Penalty (TS vs J):   11.69 %
//Penalty (DPQ vs J):  92.45 %
//Penalty (DPQ vs TS): 72.31 %
//
//---------------------------------------------------------------------------
//RANDOM (size=256, width=8, repeats=64)
//Java:              63823 ns
//DeCaff (TS):       25863 ns
//DeCaff (DPQ):      35596 ns
//Penalty (TS vs J):   -59.48 %
//Penalty (DPQ vs J):  -44.23 %
//Penalty (DPQ vs TS): 37.64 %
//
//SORTED (size=256, width=8, repeats=64)
//Java:              10225 ns
//DeCaff (TS):        1244 ns
//DeCaff (DPQ):      10217 ns
//Penalty (TS vs J):   -87.83 %
//Penalty (DPQ vs J):  -0.07 %
//Penalty (DPQ vs TS): 720.97 %
//
//SHUFFLED (size=256, width=8, repeats=64)
//Java:              39527 ns
//DeCaff (TS):       19477 ns
//DeCaff (DPQ):      22000 ns
//Penalty (TS vs J):   -50.73 %
//Penalty (DPQ vs J):  -44.34 %
//Penalty (DPQ vs TS): 12.96 %
//
//---------------------------------------------------------------------------
//RANDOM (size=256, width=16, repeats=64)
//Java:              83818 ns
//DeCaff (TS):       44305 ns
//DeCaff (DPQ):      66708 ns
//Penalty (TS vs J):   -47.14 %
//Penalty (DPQ vs J):  -20.41 %
//Penalty (DPQ vs TS): 50.56 %
//
//SORTED (size=256, width=16, repeats=64)
//Java:               1031 ns
//DeCaff (TS):        1081 ns
//DeCaff (DPQ):       9668 ns
//Penalty (TS vs J):   4.91 %
//Penalty (DPQ vs J):  837.63 %
//Penalty (DPQ vs TS): 793.74 %
//
//SHUFFLED (size=256, width=16, repeats=64)
//Java:              10505 ns
//DeCaff (TS):        5566 ns
//DeCaff (DPQ):      17217 ns
//Penalty (TS vs J):   -47.01 %
//Penalty (DPQ vs J):  63.89 %
//Penalty (DPQ vs TS): 209.29 %
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//RANDOM (size=4096, width=4, repeats=64)
//Java:            1171592 ns
//DeCaff (TS):      625598 ns
//DeCaff (DPQ):    1184562 ns
//Penalty (TS vs J):   -46.60 %
//Penalty (DPQ vs J):  1.11 %
//Penalty (DPQ vs TS): 89.35 %
//
//SORTED (size=4096, width=4, repeats=64)
//Java:              53138 ns
//DeCaff (TS):       52651 ns
//DeCaff (DPQ):     115272 ns
//Penalty (TS vs J):   -0.92 %
//Penalty (DPQ vs J):  116.93 %
//Penalty (DPQ vs TS): 118.93 %
//
//SHUFFLED (size=4096, width=4, repeats=64)
//Java:             200729 ns
//DeCaff (TS):      168393 ns
//DeCaff (DPQ):     962432 ns
//Penalty (TS vs J):   -16.11 %
//Penalty (DPQ vs J):  379.47 %
//Penalty (DPQ vs TS): 471.54 %
//
//---------------------------------------------------------------------------
//RANDOM (size=4096, width=8, repeats=64)
//Java:             645610 ns
//DeCaff (TS):      586992 ns
//DeCaff (DPQ):    1288679 ns
//Penalty (TS vs J):   -9.08 %
//Penalty (DPQ vs J):  99.61 %
//Penalty (DPQ vs TS): 119.54 %
//
//SORTED (size=4096, width=8, repeats=64)
//Java:              20312 ns
//DeCaff (TS):       20468 ns
//DeCaff (DPQ):      20363 ns
//Penalty (TS vs J):   0.76 %
//Penalty (DPQ vs J):  0.25 %
//Penalty (DPQ vs TS): -0.51 %
//
//SHUFFLED (size=4096, width=8, repeats=64)
//Java:              81885 ns
//DeCaff (TS):      122300 ns
//DeCaff (DPQ):     314335 ns
//Penalty (TS vs J):   49.36 %
//Penalty (DPQ vs J):  283.87 %
//Penalty (DPQ vs TS): 157.02 %
//
//---------------------------------------------------------------------------
//RANDOM (size=4096, width=16, repeats=64)
//Java:             572779 ns
//DeCaff (TS):      539749 ns
//DeCaff (DPQ):     580518 ns
//Penalty (TS vs J):   -5.77 %
//Penalty (DPQ vs J):  1.35 %
//Penalty (DPQ vs TS): 7.55 %
//
//SORTED (size=4096, width=16, repeats=64)
//Java:              18655 ns
//DeCaff (TS):       19114 ns
//DeCaff (DPQ):      19280 ns
//Penalty (TS vs J):   2.46 %
//Penalty (DPQ vs J):  3.35 %
//Penalty (DPQ vs TS): 0.87 %
//
//SHUFFLED (size=4096, width=16, repeats=64)
//Java:              76411 ns
//DeCaff (TS):      120579 ns
//DeCaff (DPQ):     303920 ns
//Penalty (TS vs J):   57.80 %
//Penalty (DPQ vs J):  297.74 %
//Penalty (DPQ vs TS): 152.05 %
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//RANDOM (size=65536, width=4, repeats=64)
//Java:           11595085 ns
//DeCaff (TS):    12272749 ns
//DeCaff (DPQ):   15778300 ns
//Penalty (TS vs J):   5.84 %
//Penalty (DPQ vs J):  36.08 %
//Penalty (DPQ vs TS): 28.56 %
//
//SORTED (size=65536, width=4, repeats=64)
//Java:             628216 ns
//DeCaff (TS):      610603 ns
//DeCaff (DPQ):     636669 ns
//Penalty (TS vs J):   -2.80 %
//Penalty (DPQ vs J):  1.35 %
//Penalty (DPQ vs TS): 4.27 %
//
//SHUFFLED (size=65536, width=4, repeats=64)
//Java:            2172887 ns
//DeCaff (TS):     3449569 ns
//DeCaff (DPQ):   10853149 ns
//Penalty (TS vs J):   58.76 %
//Penalty (DPQ vs J):  399.48 %
//Penalty (DPQ vs TS): 214.62 %
//
//---------------------------------------------------------------------------
//RANDOM (size=65536, width=8, repeats=64)
//Java:           12152034 ns
//DeCaff (TS):    12606112 ns
//DeCaff (DPQ):   13916075 ns
//Penalty (TS vs J):   3.74 %
//Penalty (DPQ vs J):  14.52 %
//Penalty (DPQ vs TS): 10.39 %
//
//SORTED (size=65536, width=8, repeats=64)
//Java:             355925 ns
//DeCaff (TS):      356587 ns
//DeCaff (DPQ):     355094 ns
//Penalty (TS vs J):   0.19 %
//Penalty (DPQ vs J):  -0.23 %
//Penalty (DPQ vs TS): -0.42 %
//
//SHUFFLED (size=65536, width=8, repeats=64)
//Java:            1753186 ns
//DeCaff (TS):     3359607 ns
//DeCaff (DPQ):    7915884 ns
//Penalty (TS vs J):   91.63 %
//Penalty (DPQ vs J):  351.51 %
//Penalty (DPQ vs TS): 135.62 %
//
//---------------------------------------------------------------------------
//RANDOM (size=65536, width=16, repeats=64)
//Java:           12147931 ns
//DeCaff (TS):    12651822 ns
//DeCaff (DPQ):   13862027 ns
//Penalty (TS vs J):   4.15 %
//Penalty (DPQ vs J):  14.11 %
//Penalty (DPQ vs TS): 9.57 %
//
//SORTED (size=65536, width=16, repeats=64)
//Java:             341865 ns
//DeCaff (TS):      343254 ns
//DeCaff (DPQ):     340700 ns
//Penalty (TS vs J):   0.41 %
//Penalty (DPQ vs J):  -0.34 %
//Penalty (DPQ vs TS): -0.74 %
//
//SHUFFLED (size=65536, width=16, repeats=64)
//Java:            1729910 ns
//DeCaff (TS):     3343367 ns
//DeCaff (DPQ):    7854263 ns
//Penalty (TS vs J):   93.27 %
//Penalty (DPQ vs J):  354.03 %
//Penalty (DPQ vs TS): 134.92 %
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//RANDOM (size=1048576, width=4, repeats=16)
//Java:          274674889 ns
//DeCaff (TS):   281942871 ns
//DeCaff (DPQ):  355852932 ns
//Penalty (TS vs J):   2.65 %
//Penalty (DPQ vs J):  29.55 %
//Penalty (DPQ vs TS): 26.21 %
//
//SORTED (size=1048576, width=4, repeats=16)
//Java:           27512131 ns
//DeCaff (TS):    25934944 ns
//DeCaff (DPQ):  156205163 ns
//Penalty (TS vs J):   -5.73 %
//Penalty (DPQ vs J):  467.77 %
//Penalty (DPQ vs TS): 502.30 %
//
//SHUFFLED (size=1048576, width=4, repeats=16)
//Java:           61741138 ns
//DeCaff (TS):    99347391 ns
//DeCaff (DPQ):  234707951 ns
//Penalty (TS vs J):   60.91 %
//Penalty (DPQ vs J):  280.15 %
//Penalty (DPQ vs TS): 136.25 %
//
//---------------------------------------------------------------------------
//RANDOM (size=1048576, width=8, repeats=16)
//Java:          317236161 ns
//DeCaff (TS):   325717784 ns
//DeCaff (DPQ):  396830885 ns
//Penalty (TS vs J):   2.67 %
//Penalty (DPQ vs J):  25.09 %
//Penalty (DPQ vs TS): 21.83 %
//
//SORTED (size=1048576, width=8, repeats=16)
//Java:           33897742 ns
//DeCaff (TS):    35475527 ns
//DeCaff (DPQ):   34605083 ns
//Penalty (TS vs J):   4.65 %
//Penalty (DPQ vs J):  2.09 %
//Penalty (DPQ vs TS): -2.45 %
//
//SHUFFLED (size=1048576, width=8, repeats=16)
//Java:           66251387 ns
//DeCaff (TS):   122859066 ns
//DeCaff (DPQ):  280822191 ns
//Penalty (TS vs J):   85.44 %
//Penalty (DPQ vs J):  323.87 %
//Penalty (DPQ vs TS): 128.57 %
//
//---------------------------------------------------------------------------
//RANDOM (size=1048576, width=16, repeats=16)
//Java:          341795173 ns
//DeCaff (TS):   354705132 ns
//DeCaff (DPQ):  426367193 ns
//Penalty (TS vs J):   3.78 %
//Penalty (DPQ vs J):  24.74 %
//Penalty (DPQ vs TS): 20.20 %
//
//SORTED (size=1048576, width=16, repeats=16)
//Java:           27279601 ns
//DeCaff (TS):    27028138 ns
//DeCaff (DPQ):   27102250 ns
//Penalty (TS vs J):   -0.92 %
//Penalty (DPQ vs J):  -0.65 %
//Penalty (DPQ vs TS): 0.27 %
//
//SHUFFLED (size=1048576, width=16, repeats=16)
//Java:           68815004 ns
//DeCaff (TS):   124044914 ns
//DeCaff (DPQ):  293434581 ns
//Penalty (TS vs J):   80.26 %
//Penalty (DPQ vs J):  326.41 %
//Penalty (DPQ vs TS): 136.56 %
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------
//RANDOM (size=16777216, width=4, repeats=4)
//Java:         5460686350 ns
//DeCaff (TS):  7819711552 ns
//DeCaff (DPQ): 10447202596 ns
//Penalty (TS vs J):   43.20 %
//Penalty (DPQ vs J):  91.32 %
//Penalty (DPQ vs TS): 33.60 %
//
//SORTED (size=16777216, width=4, repeats=4)
//Java:          733704403 ns
//DeCaff (TS):   768198435 ns
//DeCaff (DPQ): 5820876550 ns
//Penalty (TS vs J):   4.70 %
//Penalty (DPQ vs J):  693.35 %
//Penalty (DPQ vs TS): 657.73 %
//
//SHUFFLED (size=16777216, width=4, repeats=4)
//Java:         1570069381 ns
//DeCaff (TS):  4231814177 ns
//DeCaff (DPQ): 7779395625 ns
//Penalty (TS vs J):   169.53 %
//Penalty (DPQ vs J):  395.48 %
//Penalty (DPQ vs TS): 83.83 %
//
//---------------------------------------------------------------------------
//RANDOM (size=16777216, width=8, repeats=4)
//Java:         8211812288 ns
//DeCaff (TS):  8291894564 ns
//DeCaff (DPQ): 10458569588 ns
//Penalty (TS vs J):   0.98 %
//Penalty (DPQ vs J):  27.36 %
//Penalty (DPQ vs TS): 26.13 %
//
//SORTED (size=16777216, width=8, repeats=4)
//Java:          524111762 ns
//DeCaff (TS):   500087036 ns
//DeCaff (DPQ):  505228940 ns
//Penalty (TS vs J):   -4.58 %
//Penalty (DPQ vs J):  -3.60 %
//Penalty (DPQ vs TS): 1.03 %
//
//SHUFFLED (size=16777216, width=8, repeats=4)
//Java:         1431401035 ns
//DeCaff (TS):  3463307531 ns
//DeCaff (DPQ): 7316042687 ns
//Penalty (TS vs J):   141.95 %
//Penalty (DPQ vs J):  411.11 %
//Penalty (DPQ vs TS): 111.24 %
//
//---------------------------------------------------------------------------
//RANDOM (size=16777216, width=16, repeats=4)
//Java:         8449341711 ns
//DeCaff (TS):  8522538078 ns
//DeCaff (DPQ): 10612230624 ns
//Penalty (TS vs J):   0.87 %
//Penalty (DPQ vs J):  25.60 %
//Penalty (DPQ vs TS): 24.52 %
//
//SORTED (size=16777216, width=16, repeats=4)
//Java:          476146816 ns
//DeCaff (TS):   465585783 ns
//DeCaff (DPQ):  479641894 ns
//Penalty (TS vs J):   -2.22 %
//Penalty (DPQ vs J):  0.73 %
//Penalty (DPQ vs TS): 3.02 %
//
//SHUFFLED (size=16777216, width=16, repeats=4)
//Java:         1398143737 ns
//DeCaff (TS):  3474768423 ns
//DeCaff (DPQ): 7381541082 ns
//Penalty (TS vs J):   148.53 %
//Penalty (DPQ vs J):  427.95 %
//Penalty (DPQ vs TS): 112.43 %
//
//---------------------------------------------------------------------------
//---------------------------------------------------------------------------