// ============================================================================
// File:               QuickSortPerformanceTests
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            1/26/23 11:51 AM
//=============================================================================
package de.caff.generics.algorithm;

import de.caff.annotation.NotNull;
import de.caff.generics.IntIndexable;
import de.caff.generics.MutableIntIndexable;
import de.caff.generics.Order;
import de.caff.generics.function.IntOrdering;

import java.util.Arrays;

import static de.caff.generics.algorithm.TestTimSort.measureDuration;

/**
 * Test out possible improvements for quicksort.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since January 26, 2023
 */
public class QuickSortPerformanceTests
{
  private static void basicQuickSort(@NotNull MutableIntIndexable elements,
                                     @NotNull IntOrdering order)
  {
    final int size = elements.size();
    if (size <= 1) {
      return;
    }
    if (size == 2) {
      if (order.checkInt(elements.get(0), elements.get(1)) == Order.Descending) {
        elements.swap(0, 1);
      }
      return;
    }

    final int pIndex = basicPartition(elements, order);

    basicQuickSort(elements.headSet(pIndex), order);
    basicQuickSort(elements.tailSet(pIndex + 1), order);
  }

  private static int basicPartition(@NotNull MutableIntIndexable elements,
                                    @NotNull IntOrdering order)
  {
    final int len = elements.size() - 1;
    final int pivot = elements.get(len);
    int i = 0;

    for (int j = 0; j < len; j++) {
      if (order.checkInt(elements.get(j), pivot) == Order.Ascending) {
        elements.swap(i++, j);
      }
    }

    elements.swyp(i, -1);

    return i;
  }
  
  private static void bestOf3QuickSort(@NotNull MutableIntIndexable elements,
                                       @NotNull IntOrdering order)
  {
    final int size = elements.size();
    if (size <= 1) {
      return;
    }
    if (size == 2) {
      if (order.checkInt(elements.get(0), elements.get(1)) == Order.Descending) {
        elements.swap(0, 1);
      }
      return;
    }

    final int pIndex = bestOf3Partition(elements, order);

    bestOf3QuickSort(elements.headSet(pIndex), order);
    bestOf3QuickSort(elements.tailSet(pIndex + 1), order);
  }


  private static int BEST_OF_3_BOUNDARY = 8;
  private static int INSERTION_SORT_THRESHHOLD = 222;

  private static int bestOf3Partition(@NotNull MutableIntIndexable elements,
                                     @NotNull IntOrdering order)
  {
    final int len = elements.size() - 1;
    final int pivot;
    if (len >= BEST_OF_3_BOUNDARY) {
      final int midIndex = len / 2;
      final int ps = elements.get(0);
      final int pm = elements.get(len / 2);
      final int pe = elements.get(len);
      // find middle
      int mask = 0;
      if (order.checkInt(ps, pm) == Order.Ascending) {
        mask |= 0x01;
      }
      if (order.checkInt(ps, pe) == Order.Ascending) {
        mask |= 0x02;
      }
      if (order.checkInt(pm, pe) == Order.Ascending) {
        mask |= 0x04;
      }
      switch (mask) {
      case 0: // CBA
      case 7: // ABC
        elements.swap(midIndex, len);
        pivot = pm;
        break;
      case 1: // BCA
      case 3: // ACB
        pivot = pe;
        break;
      default: // BAC and CAB
        elements.swap(0, len);
        pivot = ps;
        break;
      }
    }
    else {
      pivot = elements.get(len);
    }
    int i = 0;

    for (int j = 0; j < len; j++) {
      if (order.checkInt(elements.get(j), pivot) == Order.Ascending) {
        elements.swap(i++, j);
      }
    }

    elements.swyp(i, -1);

    return i;
  }

  public static void main(String[] args)
  {
    final int sortSize = 65536;
    final int[] random = TestTimSort.makeRandomInt(13, sortSize);
    final int[] ordered = IntIndexable.rangeFromSize(sortSize).toArray();
    final int[][] arrays = { random, ordered };

    final long[] nanosBestOf3 = new long[arrays.length];
    final long[] nanosTim     = new long[arrays.length];
    final long[] nanosJava    = new long[arrays.length];

    final int[] work = new int[sortSize];
    final MutableIntIndexable mii = MutableIntIndexable.viewArray(work);

    for (int i = 0;  i < 100;  ++i) {
      for (int a = 0;  a < arrays.length;  ++a) {
        final int[] array = arrays[a];
        //System.arraycopy(array, 0, work, 0, sortSize);
        //nanosBasic += measureDuration(() -> basicQuickSort(mii, IntOrdering.ASCENDING));

        System.arraycopy(array, 0, work, 0, sortSize);
        nanosBestOf3[a] += measureDuration(() -> bestOf3QuickSort(mii, IntOrdering.ASCENDING));

        System.arraycopy(array, 0, work, 0, sortSize);
        nanosTim[a] += measureDuration(() -> TimSortInt.sort(mii, IntOrdering.ASCENDING));

        System.arraycopy(array, 0, work, 0, sortSize);
        nanosJava[a] += measureDuration(() -> Arrays.sort(work));
      }
    }

    System.out.printf("Best3: %12d %12d %12d\n", nanosBestOf3[0], nanosBestOf3[1], nanosBestOf3[0] + nanosBestOf3[1]);
    System.out.printf("Tim:   %12d %12d %12d\n", nanosTim[0], nanosTim[1], nanosTim[0] + nanosTim[1]);
    System.out.printf("Java:  %12d %12d %12d\n", nanosJava[0], nanosJava[1], nanosJava[0] + nanosJava[1]);
  }

}
