// ============================================================================
// File:               TestSequences
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            11/21/23 9:59 AM
//=============================================================================
package de.caff.generics;

import de.caff.generics.function.Ordering;
import junit.framework.TestCase;

import java.util.Collections;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 21, 2023
 */
public class TestSequences
        extends TestCase
{
  static final Indexable<Integer> INT_SEQ1 = Indexable.viewArray(0, 1, 2, 3);
  static final Indexable<Integer> INT_SEQ2 = INT_SEQ1.withAppendedItem(-1).frozen();

  static final Indexable<String> STR_SEQ1 = Indexable.viewArray("0", "1", "2", "3");

  public void testEquals()
  {
    assertTrue(Sequences.areEqual(Indexable.emptyIndexable(), Collections.emptyList()));
    assertTrue(Sequences.areIdentical(Indexable.emptyIndexable(), Collections.emptyList()));
    assertTrue(Sequences.areEqual(Indexable.emptyIndexable(), Collections.emptyList(),
                                  (e1, e2) -> false));

    assertTrue(Sequences.areEqual(INT_SEQ1, INT_SEQ1));
    assertTrue(Sequences.areIdentical(INT_SEQ2, INT_SEQ2));
    assertTrue(Sequences.areEqual(INT_SEQ1, INT_SEQ2.headSet(-1)));
    assertTrue(Sequences.areIdentical(INT_SEQ1, INT_SEQ2.headSet(-1)));

    assertFalse(Sequences.areEqual(INT_SEQ1, INT_SEQ2));
    assertFalse(Sequences.areIdentical(INT_SEQ1, INT_SEQ2));
    assertFalse(Sequences.areEqual(INT_SEQ1, INT_SEQ1.reverse()));

    assertFalse(Sequences.areEqual(INT_SEQ1, STR_SEQ1));
    assertFalse(Sequences.areIdentical(INT_SEQ1, STR_SEQ1));

    assertTrue(Sequences.areEqual(INT_SEQ1, STR_SEQ1,
                                  (i, s) -> Integer.toString(i).equals(s)));
  }

  public void testCompare()
  {
    assertEquals(Order.Same,
                 Sequences.compare(INT_SEQ1, INT_SEQ1));
    assertEquals(Order.Same,
                 Sequences.compare(INT_SEQ1, INT_SEQ2.headSet(-1)));
    assertEquals(Order.Ascending,
                 Sequences.compare(INT_SEQ1, INT_SEQ2));
    assertEquals(Order.Descending,
                 Sequences.compare(INT_SEQ1, INT_SEQ2, Order.Descending));
    assertEquals(Order.Descending,
                 Sequences.compare(INT_SEQ2, INT_SEQ1));
    assertEquals(Order.Ascending,
                 Sequences.compare(INT_SEQ2, INT_SEQ1, Order.Descending));
    assertEquals(Order.Descending,
                 Sequences.compare(INT_SEQ1.withAppendedItem(4), INT_SEQ2));

    final Ordering<Integer> inv = Ordering.inverseNatural();
    assertEquals(Order.Same,
                 Sequences.compare(INT_SEQ1, INT_SEQ1, inv));
    assertEquals(Order.Same,
                 Sequences.compare(INT_SEQ1, INT_SEQ2.headSet(-1), inv));
    assertEquals(Order.Ascending,
                 Sequences.compare(INT_SEQ1, INT_SEQ2, inv));
    assertEquals(Order.Descending,
                 Sequences.compare(INT_SEQ1, INT_SEQ2, Order.Descending, inv));
    assertEquals(Order.Descending,
                 Sequences.compare(INT_SEQ2, INT_SEQ1, inv));
    assertEquals(Order.Ascending,
                 Sequences.compare(INT_SEQ2, INT_SEQ1, Order.Descending, inv));
    assertEquals(Order.Ascending,
                 Sequences.compare(INT_SEQ1.withAppendedItem(4), INT_SEQ2, inv));

    assertEquals(Order.Same,
                 Sequences.compare(INT_SEQ1, STR_SEQ1, (i, s) -> Order.fromCompare(i.toString().compareTo(s.toString()))));
    assertEquals(Order.Same,
                 Sequences.compare(INT_SEQ1.view(Object::toString), STR_SEQ1));
  }

}
