// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import junit.framework.TestCase;

import java.util.Iterator;
import java.util.PrimitiveIterator;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since February 20, 2023
 */
public class TestLoop
        extends TestCase
{
  @NotNull
  private static Loop<Integer> makeLoop(int count)
  {
    return new Loop<>(IntIndexable.rangeFromSize(count));
  }

  private static Loop<Integer> makeLoop(int from, int len)
  {
    return new Loop<>(IntIndexable.range(from, from + len));
  }

  private static void check(@NotNull IntIndexable idx,
                            @NotNull Loop<Integer> loop)
  {
    assertEquals(idx.size(), loop.size());
    final PrimitiveIterator.OfInt idxIt = idx.intIterator();
    final Iterator<Integer> loopIt = loop.values().iterator();
    while (idxIt.hasNext()) {
      assertEquals(idxIt.nextInt(), loopIt.next().intValue());
    }
  }

  public void testSwap1_1_1()
  {
    final Loop<Integer> loop = makeLoop(1);
    loop.getFirstItem().swapPlaces(loop.getFirstItem());
    check(IntIndexable.rangeFromSize(1), loop);
  }

  public void testSwap2_1_2()
  {
    final Loop<Integer> loop = makeLoop(2);
    final Loop.Item<Integer> firstItem = loop.getFirstItem();
    firstItem.swapPlaces(firstItem.getNext());
    check(IntIndexable.rangeFromSize(2).rotated(1), loop);
  }

  public void testSwap3_1_2()
  {
    final Loop<Integer> loop = makeLoop(3);
    final Loop.Item<Integer> firstItem = loop.getFirstItem();
    firstItem.swapPlaces(firstItem.getNext());
    check(IntIndexable.rangeFromSize(3).withSwappedValuesAt(0, 1), loop);
  }

  public void testSwap3_2_3()
  {
    final Loop<Integer> loop = makeLoop(3);
    final Loop.Item<Integer> firstItem = loop.getFirstItem();
    firstItem.getPrevious().swapPlaces(firstItem.getNext());
    check(IntIndexable.rangeFromSize(3).withSwappedValuesAt(1, 2), loop);
  }

  public void testSwap3_3_2()
  {
    final Loop<Integer> loop = makeLoop(3);
    final Loop.Item<Integer> firstItem = loop.getFirstItem();
    firstItem.getNext().swapPlaces(firstItem.getPrevious());
    check(IntIndexable.rangeFromSize(3).withSwappedValuesAt(1, 2), loop);
  }

  public void testSwap4_1_2()
  {
    final Loop<Integer> loop = makeLoop(4);
    final Loop.Item<Integer> firstItem = loop.getFirstItem();
    firstItem.swapPlaces(firstItem.getNext());
    check(IntIndexable.rangeFromSize(4).withSwappedValuesAt(0, 1), loop);
  }

  public void testSwap4_2_1()
  {
    final Loop<Integer> loop = makeLoop(4);
    final Loop.Item<Integer> firstItem = loop.getFirstItem();
    firstItem.getNext().swapPlaces(firstItem);
    check(IntIndexable.rangeFromSize(4).withSwappedValuesAt(0, 1), loop);
  }

  public void testSwap4_1_4()
  {
    final Loop<Integer> loop = makeLoop(4);
    final Loop.Item<Integer> firstItem = loop.getFirstItem();
    firstItem.swapPlaces(firstItem.getPrevious());
    check(IntIndexable.rangeFromSize(4).withSwappedValuesAt(0, 3), loop);
  }

  public void testSwap4_4_1()
  {
    final Loop<Integer> loop = makeLoop(4);
    final Loop.Item<Integer> firstItem = loop.getFirstItem();
    firstItem.getPrevious().swapPlaces(firstItem);
    check(IntIndexable.rangeFromSize(4).withSwappedValuesAt(0, 3), loop);
  }

  public void testSwap4_2_4()
  {
    final Loop<Integer> loop = makeLoop(4);
    final Loop.Item<Integer> firstItem = loop.getFirstItem();
    firstItem.getNext().swapPlaces(firstItem.getPrevious());
    check(IntIndexable.rangeFromSize(4).withSwappedValuesAt(1, 3), loop);
  }

  public void testSwap4_4_2()
  {
    final Loop<Integer> loop = makeLoop(4);
    final Loop.Item<Integer> firstItem = loop.getFirstItem();
    firstItem.getPrevious().swapPlaces(firstItem.getNext());
    check(IntIndexable.rangeFromSize(4).withSwappedValuesAt(1, 3), loop);
  }

  public void testSwap4_0_1()
  {
    final Loop<Integer> loop = makeLoop(5);
    final Loop.Item<Integer> out = loop.getFirstItem().getPrevious();
    out.remove();
    check(IntIndexable.rangeFromSize(4), loop);

    out.swapPlaces(loop.getFirstItem());
    check(IntIndexable.rangeFromSize(4).withExchangedValueAt(0, 4), loop);
  }

  public void testSwap4_1_0()
  {
    final Loop<Integer> loop = makeLoop(5);
    final Loop.Item<Integer> out = loop.getFirstItem().getPrevious();
    out.remove();
    check(IntIndexable.rangeFromSize(4), loop);

    loop.getFirstItem().swapPlaces(out);
    check(IntIndexable.rangeFromSize(4).withExchangedValueAt(0, 4), loop);
  }

  public void testSwap4_0_2()
  {
    final Loop<Integer> loop = makeLoop(5);
    final Loop.Item<Integer> out = loop.getFirstItem().getPrevious();
    out.remove();
    check(IntIndexable.rangeFromSize(4), loop);

    out.swapPlaces(loop.getFirstItem().getNext());
    check(IntIndexable.rangeFromSize(4).withExchangedValueAt(1, 4), loop);
  }

  public void testSwap4_2_0()
  {
    final Loop<Integer> loop = makeLoop(5);
    final Loop.Item<Integer> out = loop.getFirstItem().getPrevious();
    out.remove();
    check(IntIndexable.rangeFromSize(4), loop);

    loop.getFirstItem().getNext().swapPlaces(out);
    check(IntIndexable.rangeFromSize(4).withExchangedValueAt(1, 4), loop);
  }

  public void testSwap44_0_0()
  {
    final Loop<Integer> loop1 = makeLoop(4);
    final Loop<Integer> loop2 = makeLoop(17, 4);

    loop1.getFirstItem().swapPlaces(loop2.getFirstItem());

    check(IntIndexable.rangeFromSize(4).withExchangedValueAt(0, 17), loop1);
    check(IntIndexable.range(17, 21).withExchangedValueAt(0, 0), loop2);
  }

  public void testSwap44_0_1()
  {
    final Loop<Integer> loop1 = makeLoop(4);
    final Loop<Integer> loop2 = makeLoop(17, 4);

    loop1.getFirstItem().swapPlaces(loop2.getFirstItem().getNext());

    check(IntIndexable.rangeFromSize(4).withExchangedValueAt(0, 18), loop1);
    check(IntIndexable.range(17, 21).withExchangedValueAt(1, 0), loop2);
  }

  public void testSwap44_1_0()
  {
    final Loop<Integer> loop1 = makeLoop(4);
    final Loop<Integer> loop2 = makeLoop(17, 4);

    loop1.getFirstItem().getNext().swapPlaces(loop2.getFirstItem());

    check(IntIndexable.rangeFromSize(4).withExchangedValueAt(1, 17), loop1);
    check(IntIndexable.range(17, 21).withExchangedValueAt(0, 1), loop2);
  }

  public void testSwap44_1_1()
  {
    final Loop<Integer> loop1 = makeLoop(4);
    final Loop<Integer> loop2 = makeLoop(17, 4);

    loop1.getFirstItem().getNext().swapPlaces(loop2.getFirstItem().getNext());

    check(IntIndexable.rangeFromSize(4).withExchangedValueAt(1, 18), loop1);
    check(IntIndexable.range(17, 21).withExchangedValueAt(1, 1), loop2);
  }
}
