// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.generics.tuple.Tuple;
import de.caff.generics.tuple.Tuple4;
import de.caff.generics.util.Counter;
import de.caff.generics.util.SimpleCounter;
import junit.framework.Assert;
import junit.framework.TestCase;

import java.util.Arrays;
import java.util.List;

/**
 * Tests for the Indexable class.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 15, 2021
 */
public class TestIndexable
        extends TestCase
{
  public void testArrayView()
  {
    final Integer[] array = { 0, 1, 2, 3, 4 };
    final Indexable<Integer> indexable = Indexable.viewArray(array);

    assertEquals(array.length, indexable.size());

    assertTrue(Arrays.equals(array, indexable.toArray()));

    final Counter counter = new SimpleCounter();
    indexable.forEach(e -> counter.add1());
    assertEquals(array.length, counter.getValue());

    counter.setValue(0);
    indexable.forEachEntry((i, e) -> counter.add1());
    assertEquals(array.length, counter.getValue());

    indexable.forEachEntry(Assert::assertEquals);

    final Indexable<Integer> sub = indexable.subSet(1, 4);
    assertEquals(3, sub.size());
    sub.forEachEntry((idx, elem) -> assertEquals(idx + 1, elem.intValue()));

    final Indexable<Integer> frozen = sub.frozen();

    array[2] = -2;
    assertEquals(array[2], indexable.get(2));
    assertEquals(array[2], sub.get(1));
    assertEquals(2, frozen.get(1).intValue());
  }

  public void testListView()
  {
    final List<String> list = Arrays.asList("A", "BB", "CCC", "DDDD", "EEEEE");
    final Indexable<String> indexable = Indexable.viewList(list);

    assertEquals(list.size(), indexable.size());
    assertEquals(list, indexable.asList());

    final Counter counter = new SimpleCounter();
    indexable.forEach(e -> counter.add1());
    assertEquals(list.size(), counter.getValue());

    counter.setValue(0);
    indexable.forEachEntry((i, e) -> counter.add1());
    assertEquals(list.size(), counter.getValue());

    indexable.forEachEntry((idx, elem) -> assertEquals(list.get(idx), elem));

    final Indexable<String> sub = indexable.subSet(1, 4);
    assertEquals(3, sub.size());
    sub.forEachEntry((idx, elem) -> assertEquals(list.get(idx + 1), elem));

    final Indexable<String> frozen = sub.frozen();

    list.set(2, "FOO");
    assertEquals(list.get(2), indexable.get(2));
    assertEquals(list.get(2), sub.get(1));
    assertEquals("CCC", frozen.get(1));
  }

  public void testReverse()
  {
    final String[] array = { "A", "B", "C", "D", "E" };
    final Indexable<String> indexable = Indexable.viewArray(array);
    final Indexable<String> frozen = indexable.frozen();
    final Indexable<String> reverse = indexable.reverse();

    assertEquals(indexable.size(), reverse.size());
    indexable.forEachEntry((idx, elem) -> assertEquals(elem, reverse.gyt(-1 - idx)));

    assertEquals(indexable, reverse.reverse());

    final Indexable<String> reverseFrozen = reverse.frozen();
    array[3] = "DDDD";
    assertEquals(array[3], indexable.get(3));
    assertEquals(array[3], reverse.gyt(-4));
    assertEquals("D", reverseFrozen.gyt(-4));

    final Indexable<String> sorted = frozen.sorted(String.CASE_INSENSITIVE_ORDER.reversed());
    assertEquals(reverseFrozen, sorted);
  }

  public void testEqualsAndHash()
  {
    final Indexable<Double> indexable1 = Indexable.viewArray(1.1, Math.PI, Double.NEGATIVE_INFINITY, -42.0, 0.0);
    final Indexable<Double> indexable2 = Indexable.viewArray(1.1, Math.PI, Double.NEGATIVE_INFINITY, -42.0, 0.0);
    assertEquals(indexable1, indexable2);
    assertEquals(indexable1.hashCode(), indexable2.hashCode());
  }

  public void testSubAndFrozen()
  {
    final String[] array = { "A", "B", "C", "D", "E" };
    final Indexable<String> indexable = Indexable.viewArray(array);
    final Indexable<String> sub = indexable.subSet(1, 4);
    final Indexable<String> frozen = indexable.frozen();
    final Indexable<String> subFrozen = sub.frozen();
    final Indexable<String> frozenSub = frozen.subSet(1, 4);

    assertEquals(subFrozen, frozenSub);

    array[2] = "Sea";
    assertEquals(array[2], indexable.get(2));
    assertEquals(array[2], sub.get(1));
    assertEquals("C", frozen.get(2));
    assertEquals("C", subFrozen.get(1));
    assertEquals("C", frozenSub.get(1));
  }

  public void testToString()
  {
    final String[] array = { "A", "B", "C", "D", "E" };
    final Indexable<String> indexable = Indexable.viewArray(array);
    final Indexable<String> sub = indexable.subSet(1, 4);
    assertEquals("[A,B,C,D,E]", indexable.toString());
    assertEquals("[B,C,D]", sub.toString());
    final Indexable<String> frozen = indexable.frozen();
    final Indexable<String> subFrozen = sub.frozen();
    array[2] = "Zeh";
    assertEquals("[A,B,Zeh,D,E]", indexable.toString());
    assertEquals("[B,Zeh,D]", sub.toString());
    assertEquals("[A,B,C,D,E]", frozen.toString());
    assertEquals("[B,C,D]", subFrozen.toString());
  }

  public void testAddToArray()
  {
    final String[] array = { "A", "B", "C", "D", "E", "F" };
    final String[] backup = array.clone();

    final Indexable<String> indexable = Indexable.viewArray(array);

    final String[] directResult = indexable.toArray(String.class);
    assertTrue(Arrays.equals(array, directResult));

    final String[] indirectResult = new String[array.length];
    assertEquals(array.length, indexable.addToArray(indirectResult, 0));
    assertTrue(Arrays.equals(array, indirectResult));

    final Indexable<Indexable<String>> parts =
            Indexable.viewArray(indexable.headSet(2),
                                indexable.subSet(2, 4),
                                indexable.tailSet(-2));

    final String[] partResult = new String[array.length];
    int pos = 0;
    for (Indexable<String> idx : parts) {
      pos = idx.addToArray(partResult, pos);
    }
    assertEquals(partResult.length, pos);
    assertTrue(Arrays.equals(array, partResult));

    final Indexable<String> frozen = indexable.frozen();
    final Indexable<Indexable<String>> frozenParts =
            Indexable.viewArray(frozen.headSet(2),
                                frozen.subSet(2, 4),
                                frozen.tailSet(-2));

    array[3] = "Donnerwetter";
    assertTrue(Arrays.equals(array, indexable.toArray()));
    final String[] midArray = new String[2];
    System.arraycopy(array, 2, midArray,0, 2);
    assertTrue(Arrays.equals(midArray, parts.get(1).toArray()));

    assertTrue(Arrays.equals(backup, frozen.toArray()));
    System.arraycopy(backup, 2, midArray, 0, 2);
    assertTrue(Arrays.equals(midArray, frozenParts.get(1).toArray()));
  }

  public void testInserting()
  {
    final Indexable<String> idx = Indexable.viewArray("one", "two", "three", "four");
    assertEquals(Indexable.viewArray("zero", "one", "two", "three", "four"),
                 idx.withInsertedItemAt(0, "zero"));
    assertEquals(Indexable.viewArray("one", "two", "three", "four", "five"),
                 idx.withInsertedItemAt(idx.size(), "five"));
    assertEquals(Indexable.viewArray("one", "two", "three", "pi", "four"),
                 idx.withInsertedItemAt(3, "pi"));
  }

  public void testTuple()
  {
    final Number[] array = { 0, 1.0, 2L, 3f };
    final Tuple4<Integer, Double, Long, Float> tuple = Tuple.of(0, 1.0, 2L, 3f);
    assertEquals(Indexable.viewArray(array), Indexable.viewTuple(tuple));
  }

  public void testDownCast()
  {
    final Indexable<Double> indexable = Indexable.viewArray(1.0, 2e1, 3e-2, -3.14);
    final Indexable<Number> numbers = Indexable.downCast(indexable);
    assertSame(indexable, numbers);
    final Indexable<Object> objects = Indexable.downCast(numbers);
    assertSame(indexable, objects);
  }
}
