// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2020-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import junit.framework.TestCase;

import java.util.Random;

import static de.caff.generics.ExpandableIntIndexableImpl.PART_SIZE;

/**
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since October 02, 2020
 */
public class TestExpandableIntIndexable
        extends TestCase
{
  private ExpandableIntIndexable createSequence(final int size)
  {
    final int[] array = new int[size];
    for (int i = 0;  i < size;  ++i) {
      array[i] = i;
    }
    final long start = System.currentTimeMillis();
    final ExpandableIntIndexable indexable = ExpandableIntIndexable.from(array);
    System.out.printf("Duration for %d entries: %d ms\n", size, System.currentTimeMillis() - start);
    
    return indexable;
  }

  private ExpandableIntIndexable testCreationFromArray(final int size)
  {
    final ExpandableIntIndexable sequence = createSequence(size);
    assertEquals(size, sequence.size());

    for (int i = 0;  i < size;  ++i) {
      assertEquals(i, sequence.get(i));
    }
    return sequence;
  }

  public void testInitFromArrays()
  {
    final ExpandableIntIndexable indexable = testCreationFromArray(0);
    assertTrue(indexable.isEmpty());

    testCreationFromArray(1);

    int size = PART_SIZE;
    while (size < Integer.MAX_VALUE / PART_SIZE) {
      testCreationFromArray(size - 1);
      testCreationFromArray(size);
      testCreationFromArray(size + 1);

      size *= PART_SIZE;
    }
  }

  public void testInsertion()
  {
    ExpandableIntIndexable indexable = ExpandableIntIndexable.empty();
    indexable = indexable.add(-1);
    assertEquals(1, indexable.size());
    assertEquals(-1, indexable.get(0));

    final Random random = new Random(0);
    int size = PART_SIZE;
    while (size < Integer.MAX_VALUE / PART_SIZE) {
      indexable = createSequence(size - PART_SIZE / 2);

      final long start = System.currentTimeMillis();
      for (int tries = 0;  tries < PART_SIZE;  ++tries) {
        final int index = random.nextInt(indexable.size());
        final int value = -tries - 1;
        final int oldSize = indexable.size();
        indexable = indexable.add(index, value);
        assertEquals(oldSize + 1, indexable.size());
        assertEquals(value, indexable.get(index));
      }
      System.out.printf("%d insertions (elements now: %d) took %d ms.\n", PART_SIZE, indexable.size(), System.currentTimeMillis() - start);

      size *= PART_SIZE;
    }
  }

}
