// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright ©2023-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import junit.framework.TestCase;

import java.util.*;

import static de.caff.generics.Primitives.boxed;

/**
 * Test what we think about Java {@code double} and {@link java.lang.Double} is true.
 * The designers of {@code Double} and {@code Float} made some decisions which are, to say the least,
 * questionable. E.g. they differ between {@code +0.0} and {@code -0.0} which are two valid
 * IEEE 754 values, but the standard also states that they represent the same value {@code 0.0},
 * so they are equal. For primitive {@code double} values this is indeed the case, but
 * {@code Double} disagrees.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since February 12, 2023
 */
public class TestDoubles
        extends TestCase
{
  public void testZeros()
  {
    final double nz = -0.0;
    final double pz = 0.0;

    final Double abNZ = nz; // automatically boxed negative zero (i.e. Double.valueOf())
    final Double abPZ = pz; // automatically boxed positive zero (i.e. Double.valueOf())

    final Double ebNZ = boxed(nz);  // explicitly boxed negative zero from Primitives.boxed()
    final Double ebPZ = boxed(pz);  // explicitly boxed positive zero from Primitives.boxed()

    assert nz == pz;

    assert !abNZ.equals(abPZ);
    assert ebNZ.equals(ebPZ);

    assert Double.hashCode(nz) != Double.hashCode(pz);
    assert Double.hashCode(nz) == abNZ.hashCode();
    assert Double.hashCode(pz) == abPZ.hashCode();
    assert ebNZ.hashCode() == ebPZ.hashCode();

    assert !(nz < pz);
    assert abNZ.compareTo(abPZ) < 0;
    assert !(ebNZ.compareTo(ebPZ) < 0);

    assert abNZ == pz  &&  !abNZ.equals(pz);  // same, but not equal
    assert ebNZ == pz  &&  ebNZ.equals(pz);

    final Set<Double> hashSet = new HashSet<>();
    final Set<Double> treeSet = new TreeSet<>();

    hashSet.add(nz);
    treeSet.add(nz);

    assert !hashSet.contains(pz);
    assert !treeSet.contains(pz);

    hashSet.clear();
    treeSet.clear();

    hashSet.add(ebNZ);
    treeSet.add(ebNZ);

    assert hashSet.contains(pz);
    assert treeSet.contains(pz);
  }

  public void testNaN()
  {
    final double nan = Double.NaN;
    final Double bNaN = nan;

    assert bNaN != nan  &&
           bNaN.equals(nan);

  }

  public void testSubjectivity()
  {
    final double nan = Double.NaN;
    final double anotherNaN = Double.longBitsToDouble(Double.doubleToLongBits(nan) + 1);
    final double nz = -0.0;
    final double pz = 0.0;

    final double[] p1 = { nan, nz };
    final double[] p2 = { anotherNaN, pz };
    assert !Arrays.equals(p1, p2);
    assert Primitives.areEqual(p1, p2);

    final Double[] b1 = { nan, nz };
    final Double[] b2 = { anotherNaN, pz };
    assert !Arrays.equals(b1, b2);
    assert Primitives.areEqual(b1, b2);

    final Object[] a1 = { p1, b1 };
    final Object[] a2 = { p2, b2 };
    assert !Arrays.equals(a1, a2);
    assert Primitives.areEqual(a1, a2);

    final String[] fb = { "foo", "bar" };
    final Object[][] o1 = { fb, a1, { p1, p2 }, b1};
    final Object[][] o2 = { fb, a2, { p2, p1 }, b2};
    assert !Arrays.deepEquals(o1, o2);
    assert Primitives.areEqual(o1, o2); 
  }

  public void testSorting()
  {
    // Java sorting by Arrays.sort is not agreeing with Double.compare()
    final double nan = Double.NaN;
    final double anotherNaN = Double.longBitsToDouble(Double.doubleToLongBits(nan) + 1);
    final double yetAnotherNaN = Double.longBitsToDouble(Double.doubleToLongBits(nan) + 2);
    final double nz = -0.0;
    final double pz = 0.0;


    final double[] array = { pz, nz, yetAnotherNaN, nan, pz, anotherNaN, nan };
    Arrays.sort(array);
    for (int i = 1;  i < array.length;  ++i) {
      assert Double.compare(array[i - 1], array[i]) <= 0;
    }

  }

}
