// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics;

import de.caff.annotation.NotNull;
import de.caff.generics.function.Predicate1;
import junit.framework.TestCase;

import java.util.*;

/**
 * A test with lots of constructs using the Java Generic Framework.
 * <p>
 * It has to be compiled without problems.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public class CompileTest
        extends TestCase
{
  private static final String[] LINES = {
          "Just a few strings.",
          "Foo bar baz.",
          "To compile or not to Compile."
  };


  private static final int[] INTS = {
          1, 2, -5, 42, 121
  };

  // Types.asList() is a bit more efficient than standard
  // Arrays.asList(), as it avoids copying the whole list
  @NotNull
  public static List<String> asList(String ... texts)
  {
    return Types.asList(texts);
  }

  // The is similar functionality for primitive types,
  // under the name toList()
  @NotNull
  public static List<Integer> asList(int ... values)
  {
    return Types.toList(values);
  }

  /**
   * Get the words contained in a bunch of strings.
   * @param lines lines of file
   * @return words, sorted by natural order (case-insensitive)
   */
  public static Collection<String> getWords(String[] lines) {
    return Types.mapX(new TreeSet<>(String.CASE_INSENSITIVE_ORDER), lines, Types.WORD_SPLIT);
  }

  /**
   * Show a word statistics.
   * @param lines lines of file
   */
  @NotNull
  public static CountingTreeSet<String> getWordStats(String[] lines) {
    return Types.mapX(new CountingTreeSet<>(String.CASE_INSENSITIVE_ORDER), lines, Types.WORD_SPLIT);
  }

  public void testLineSplitting()
  {
    Collection<String> words = getWords(LINES);
    assertTrue(11 == words.size());
    assertTrue("a".equals(words.iterator().next()));

    CountingSet<String> set = getWordStats(LINES);
    assertTrue(11 == set.size());
    assertTrue(0 == set.getCount("Hullo"));
    assertTrue(1 == set.getCount("strings"));
    assertTrue(2 == set.getCount("compile"));
    assertTrue(2 == set.getCount("Compile"));
    assertTrue(2 == set.getCount("COMPILE"));
  }

  // apply a list of integers into their string representation:
  @NotNull
  public static List<String> intToString(@NotNull  List<Integer> intList) {
    return Types.map(intList, Types.TO_STRING);
  }

  // apply a list of strings into integers, throwing an exception on problems:
  public static List<Integer> stringToInt1(List<String> strList) throws TypeConverterException {
    return Types.mapE(strList, Types.STR_TO_INT);
  }

  // apply a list of strings into integers, discarding unconvertable strings:
  public static List<Integer> stringToInt2(List<String> strList) {
    // toNonFragile() avoids throwing an exception, but is (with this signature) returning null instead,
    // which -- thanks to the last parameter -- is defined as the delete mark
    return Types.map(strList, Types.toNonFragile(Types.STR_TO_INT), null);
  }

  // same as before, but allowing null on input, which shall be put into output as well:
  @NotNull
  @SuppressWarnings("deprecation") // here we explicitly want an Integer which has a reference unlike any other.
  public static List<Integer> stringToInt3(@NotNull List<String> strList) {
    // the following should be a constant
    final Integer mark = new Integer(42); // KEEP BOXING! No need for something special, only references are compared
    // toNonFragile() avoids throwing an exception, but is (with this signature) returning 'mark' instead,
    // which -- thanks to the last parameter -- is defined as the delete mark
    return Types.map(strList, Types.toNonFragile(Types.STR_TO_INT, mark), mark);
  }

  // something else: flatten a list of collections of strings:
  @NotNull
  public static List<String> getFlattened(@NotNull  List<Collection<String>> incoming)
  {
    return Types.flatten(incoming); // list with all strings from all the collections in incoming
  }

  /**
   * Get a string representation of a collection, separated by commas.
   * @param objects objects to output
   * @return string of objects toString() results, separated by commas
   */
  @NotNull
  public static String toString(@NotNull Iterable<?> objects)
  {
    // using view() instead of map() avoid copying the whole collection
    return Types.join(",", Types.view(objects, Types.TO_STRING));
  }

  // Filter out all strings which start with "#":
  private static final Predicate1<String> STARTS_WITH_HASH =
          arg -> {
            return arg.startsWith("#");  // assuming non-null
          };

  @NotNull
  public List<String> stringsWithHash(@NotNull Collection<String> strings)
  {
    return Types.filter(strings, STARTS_WITH_HASH);
  }


  public void testIntToString() throws TypeConverterException
  {
    List<String> strings = intToString(Types.toList(INTS));
    assertEquals(INTS.length, strings.size());
    assertEquals('-', strings.get(2).charAt(0));

    List<Integer> ints = stringToInt1(strings);
    assertTrue(Types.areEqual(Types.toList(INTS), ints));

    ints = stringToInt2(strings);
    assertTrue(Types.areEqual(Types.toList(INTS), ints));

    ints = stringToInt3(strings); // note that there is a 42 contained in the set!
    assertTrue(Types.areEqual(Types.toList(INTS), ints));
  }

  public void testFlatten()
  {
    List<Collection<String>> start = new ArrayList<>(3);
    start.add(Types.asList(LINES)); //  3 elements
    start.add(getWords(LINES));     // 11 elements
    start.add(getWordStats(LINES)); // 11 elements

    List<String> flat = getFlattened(start);
    assertEquals(25, flat.size());
  }

  public void testCounting()
  {
    Number[] array = { 1, Math.PI, 7L };
    double result = Types.foldLeft(array, 0.0, (arg1, arg2) -> arg1 + arg2.doubleValue());
    assertEquals(1 + Math.PI + 7L, result);

    Collection<Number> numbers = Types.<Number>asList(1, Math.PI, 7L);
    result = Types.foldLeft(numbers, 0.0,
                            (arg1, arg2) -> arg1 + arg2.doubleValue());
    assertEquals(1 + Math.PI + 7L, result);


    DoubleIndexable indexable = DoubleIndexable.viewArray(Number::doubleValue, array);
    result = indexable.foldLeft(0, Double::sum);
    assertEquals(1 + Math.PI + 7L, result);
  }

}
