// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.version;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Indexable;
import de.caff.generics.Types;

import java.util.Comparator;
import java.util.Objects;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * Semantic version.
 * <p>
 * This follows <a href="https://semver.org/spec/v2.0.0.html">Semantic Versioning 2.0.0</a>.
 * <p>
 * Please note that camparison and hash/equals are not always in sync what they consider equal.
 * While comparison ignores {@link #getBuildString() the build metadata}, {@link #equals(Object)} and
 * {@link #hashCode()} take it into account. Use the {@link #compareTo(SemVer, Comparator)} with
 * a string comparator which considers different strings different (e.g. {@link String#compareTo(String)})
 * to get a way of sorting consistent to hash/equals. Or use {@link #equalsIgnoreBuild(SemVer)} which
 * ignrores the build number.
 * <p>
 * This class is deliberately not serializable. Use an intermediate string representation instead.
 * <p>
 * This class is immutable.
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 24, 2021
 */
public class SemVer
        implements Comparable<SemVer>
{
  /** Regexp group id for major version number. Always necessary. */
  private static final String RE_GRP_MAJOR = "major";
  /** Regexp group id for minor version number. Always necessary. */
  private static final String RE_GRP_MINOR = "minor";
  /** Regexp group id for patch version number. Always necessary. */
  private static final String RE_GRP_PATCH = "patch";
  /** Regexp group id for prerelease indicator. Optional. */
  private static final String RE_GRP_PRERELEASE = "prerelease";
  /** Regexp group id for build metadata. Optional. */
  private static final String RE_GRP_BUILD = "build";
  /**
   * Pattern for a single numeric version number.
   * Different to semver.org we don't use the \\d shortcut, as it may fit Unicode digits outside the range 0-9.
   */
  private static final String RE_VERSION = "0|[1-9][0-9]*";
  /** Regular expression for a single build metadata item. */
  private static final String RE_BUILD_ITEM = "[0-9a-zA-Z-]";
  /** Regular expression for build metadata. */
  private static final String RE_BUILD = RE_BUILD_ITEM + "+(?:\\." + RE_BUILD_ITEM + "+)*";
  /** Regular expression for prerelease. */
  private static final String RE_PRERELEASE = "(?:" + RE_VERSION + "|[0-9]*[a-zA-Z-][0-9a-zA-Z-]*)(?:\\.(?:" + RE_VERSION + "|[0-9]*[a-zA-Z-][0-9a-zA-Z-]*))*";
  /** Standard Java-compatible regular expression for a semantic version. */
  private static final String REGEXP = "^(?<" + RE_GRP_MAJOR + ">" + RE_VERSION + ")\\.(?<" + RE_GRP_MINOR + ">" + RE_VERSION + ")\\.(?<" + RE_GRP_PATCH + ">" + RE_VERSION + ")" +
                                       "(?:-(?<" + RE_GRP_PRERELEASE + ">" + RE_PRERELEASE + "))?" +
                                       "(?:\\+(?<" + RE_GRP_BUILD + ">"+ RE_BUILD + "))?$";
  /** Regular expression pattern matching a complete semantic version. */
  private static final Pattern PATTERN = Pattern.compile(REGEXP);
  /** Pattern for a complete single version number. */
  private static final Pattern PATTERN_VERSION = Pattern.compile("^" + RE_VERSION + "$");
  /** Pattern for a complete prerelease. */
  private static final Pattern PATTERN_PRERELEASE = Pattern.compile("^" + RE_PRERELEASE + "$");
  /** Pattern for a single build item. */
  private static final Pattern PATTERN_BUILD_ITEM = Pattern.compile("^" + RE_BUILD_ITEM + "$");
  /** Pattern for a build metadata. */
  private static final Pattern PATTERN_BUILD = Pattern.compile("^" + RE_BUILD + "$");

  private static final String DOT = ".";
  /** Separator pattern for splitting at dots. */
  static final Pattern SEP_DOT = Pattern.compile(Pattern.quote(DOT));

  public static final SemVer VERSION_0_0_1 = new SemVer(0, 0, 1);

  /** Major version. */
  private final int major;
  /** Minor version. */
  private final int minor;
  /** Patch level. */
  private final int patch;
  /** Prerelease versioning, not used if {@code null}. */
  @Nullable
  private final Prerelease prerelease;
  /** Build metadata, not used if empty. */
  @NotNull
  private final Indexable<String> buildMetaData;

  /**
   * Constructor.
   * This will create a regular semantic version with neither prerelease not build metadata.
   * @param major non-negative major version
   * @param minor non-negative minor version
   * @param patch non-negative patch version
   * @throws IllegalArgumentException if any version number is negative
   */
  public SemVer(int major, int minor, int patch)
  {
    this(major, minor, patch, null, Indexable.emptyIndexable());
  }

  /**
   * Constructor including prerelease data.
   * @param major non-negative major version
   * @param minor non-negative minor version
   * @param patch non-negative patch version
   * @param prerelease prerelease data, none if {@code null}
   * @throws IllegalArgumentException if any version number is negative
   */
  public SemVer(int major, int minor, int patch, @Nullable Prerelease prerelease)
  {
    this(major, minor, patch, prerelease, Indexable.emptyIndexable());
  }

  /**
   * Constructor including build metadata.
   * @param major non-negative major version
   * @param minor non-negative minor version
   * @param patch non-negative patch version
   * @param buildString build metadata, may only contain ASCII letters, ASCII digits and both '.' and '-'
   * @throws IllegalArgumentException if any version number is negative, or {@code build} contains illegal characters
   */
  public SemVer(int major, int minor, int patch, @Nullable String buildString)
  {
    this(major, minor, patch, null, buildString);
  }

  /**
   * Constructor.
   * @param major non-negative major version
   * @param minor non-negative minor version
   * @param patch non-negative patch version
   * @param prerelease prerelease data, none if {@code null}
   * @param buildString build metadata, may only contain ASCII letters, ASCII digits and both '.' and '-'
   */
  public SemVer(int major, int minor, int patch, @Nullable Prerelease prerelease, @Nullable String buildString)
  {
    this(major, minor, patch, prerelease, parseBuild(buildString), false);
  }

  /**
   * Constructor.
   * @param major non-negative major version
   * @param minor non-negative minor version
   * @param patch non-negative patch version
   * @param prerelease prerelease data, none if {@code null}
   * @param buildMetadata build metadata, may only contain ASCII letters, ASCII digits and both and '-'
   */
  private SemVer(int major, int minor, int patch,
                 @Nullable Prerelease prerelease,
                 @NotNull Indexable<String> buildMetadata)
  {
    this(major, minor, patch, prerelease, buildMetadata, true);
  }

  /**
   * Constructor.
   * @param major non-negative major version
   * @param minor non-negative minor version
   * @param patch non-negative patch version
   * @param prerelease prerelease data, none if {@code null}
   * @param buildMetadata build metadata, may only contain ASCII letters, ASCII digits and both and '-'
   * @param checkBuildItems check the format of the single items in buildMetaData
   */
  private SemVer(int major, int minor, int patch,
                 @Nullable Prerelease prerelease,
                 @NotNull Indexable<String> buildMetadata,
                 boolean checkBuildItems)
  {
    if (major < 0) {
      throw new IllegalArgumentException("major has to be 0 or positive, but is " + major);
    }
    if (minor < 0) {
      throw new IllegalArgumentException("minor has to be 0 or positive, but is " + minor);
    }
    if (patch < 0) {
      throw new IllegalArgumentException("patch has to be 0 or positive, but is " + patch);
    }
    this.major = major;
    this.minor = minor;
    this.patch = patch;
    this.prerelease = prerelease;
    if (checkBuildItems) {
      this.buildMetaData = buildMetadata.frozen();
      this.buildMetaData.forEach(item -> {
        if (!PATTERN_BUILD_ITEM.matcher(item).matches()) {
          throw new IllegalArgumentException("Illegal build metadata item: \"" + item + '"');
        }
      });
    }
    else {
      this.buildMetaData = buildMetadata;
    }
  }

  /**
   * Parse a combined build string.
   * @param build combined string
   * @return separated build metadata
   */
  @NotNull
  private static Indexable<String> parseBuild(@Nullable String build)
  {
    if (build == null) {
      return Indexable.emptyIndexable();
    }
    if (!PATTERN_BUILD.matcher(build).matches()) {
      throw new IllegalArgumentException("Illegal build metadata definition: \"" + build + '"');
    }
    return Indexable.viewArray(SEP_DOT.split(build));
  }

  /**
   * Get the major number.
   * The major number is most significant and is usually changed.
   * Version {@code 0} is considered unstable, any changes might occur in the versioned API.
   * Major number is increased when there are incompatible API changes.
   * 
   * @return positive major number, or {@code 0} indicating an unstable initial development version
   */
  public int getMajor()
  {
    return major;
  }

  /**
   * Get the minor number.
   * The minor number is increased when there are API additions or new deprecation markings,
   * but the API stays backward-compatible.
   * @return minor number, positive or {@code 0}
   */
  public int getMinor()
  {
    return minor;
  }

  /**
   * Get the patch number.
   * The patch number is increased on internal bugfixes, when not API changes are implemented.
   * @return patch number
   */
  public int getPatch()
  {
    return patch;
  }

  /**
   * Get the prerelease number.
   * Any semantic version with a non-null prerelease is before the same version with no
   * prerelease.
   * @return prerelease, or {@code null} if this is no prerelease
   */
  @Nullable
  public Prerelease getPrerelease()
  {
    return prerelease;
  }

  /**
   * Get the build metadata.
   * The build metadata is related to, but not part of the semantic version.
   * It has no influence on sorting.
   * @return build metadata, empty if no metadata is attached
   * @see #getBuildString()
   */
  @NotNull
  public Indexable<String> getBuildMetaData()
  {
    return buildMetaData;
  }

  /**
   * Get the build metadata as one string.
   * The build metadata is related to, but not part of the semantic version.
   * It has no influence on sorting.
   * @return build metadata, or {@code null} if no metadata is attached
   * @see #getBuildMetaData()
   */
  @Nullable
  public String getBuildString()
  {
    return buildMetaData.isEmpty()
            ? null
            : Types.join(DOT, buildMetaData);
  }

  /**
   * Get the next semantic version number when increasing the major version.
   * @return next major version, with zero minor and zero patch version, and neither prerelease indicator nor build metadata
   * @throws ArithmeticException if the increasing would overflow an integer
   */
  @NotNull
  public SemVer nextMajor()
  {
    if (major == Integer.MAX_VALUE) {
      throw new ArithmeticException("Major version cannot be increased bcause of integer restrictions!");
    }
    return new SemVer(major + 1, 0, 0);
  }

  /**
   * Get the next semantic version number when increasing the minor version.
   * @return next minor version, with same major version and zero patch version, and neither prerelease indicator nor build metadata
   * @throws ArithmeticException if the increasing would overflow an integer
   */
  @NotNull
  public SemVer nextMinor()
  {
    if (minor == Integer.MAX_VALUE) {
      throw new ArithmeticException("Minor version cannot be increased bcause of integer restrictions!");
    }
    return new SemVer(major, minor + 1, 0);
  }

  /**
   * Get the next semantic version number when increasing the patch version.
   * @return next patch version, with same major and minor version, and neither prerelease indicator nor build metadata
   * @throws ArithmeticException if the increasing would overflow an integer
   */
  @NotNull
  public SemVer nextPatch()
  {
    if (patch == Integer.MAX_VALUE) {
      throw new ArithmeticException("Patch version cannot be increased bcause of integer restrictions!");
    }
    return new SemVer(major, minor, patch + 1);
  }

  /**
   * Get the same version as this. but with a different prerelease.
   * @param prerelease prerelease
   * @return this version with the given prerelease, same major, minor and patch version, but no build metadata
   */
  @NotNull
  public SemVer withPrerelease(@Nullable Prerelease prerelease)
  {
    if (buildMetaData.isEmpty()  &&  Objects.equals(this.prerelease, prerelease)) {
      return this;
    }
    return new SemVer(major, minor, patch, prerelease);
  }

  /**
   * Get the same version as this, but with a different prerelease.
   * @param prerelease prerelease as string
   * @return this version with the given prerelease
   */
  @NotNull
  public SemVer withPrereleaseFromString(@Nullable String prerelease)
  {
    if (prerelease == null) {
      return withPrerelease(null);
    }
    final Prerelease pre = Prerelease.parse(prerelease);
    if (pre == null) {
      throw new IllegalArgumentException("Invalid prerelease string: \""+prerelease+'"');
    }
    return withPrerelease(pre);
  }

  /**
   * Get the same version as this, but with no build metadata.
   * @return this version without the build metadata
   */
  @NotNull
  public SemVer withNoBuild()
  {
    return buildMetaData.isEmpty()
            ? this
            : new SemVer(major, minor, patch, prerelease);
  }

  /**
   * Get the same version as this. but with different build metadata.
   * @param build build metadata, must contain only ASCII letters, digits, and both {@code '.'} and {@code '-'}
   * @return this version with the given build metadata, and same major, minor and patch version,
   *         and the same prerelease
   * @throws IllegalArgumentException if {@code build} contains illegal characters
   */
  @NotNull
  public SemVer withBuildString(@NotNull String build)
  {
    return new SemVer(major, minor, patch, prerelease, build);
  }

  /**
   * Get the same version as this. but with different build metadata.
   * @param buildMetaData build metadata, must contain only ASCII letters, digits, and both {@code '.'} and {@code '-'}
   * @return this version with the given build metadata, and same major, minor and patch version,
   *         and the same prerelease
   * @throws IllegalArgumentException if {@code build} contains illegal characters
   */
  public SemVer withBuild(@NotNull Indexable<String> buildMetaData)
  {
    if (buildMetaData.equals(this.buildMetaData)) {
      return this;
    }
    return new SemVer(major, minor, patch, prerelease, buildMetaData, true);
  }

  @Override
  public boolean equals(Object o)
  {
    if (this == o) {
      return true;
    }
    if (o == null || getClass() != o.getClass()) {
      return false;
    }
    final SemVer semVer = (SemVer)o;
    return major == semVer.major &&
           minor == semVer.minor &&
           patch == semVer.patch &&
           Objects.equals(prerelease, semVer.prerelease) &&
           Objects.equals(buildMetaData, semVer.buildMetaData);
  }

  /**
   * Equality check ignoring the build metadata.
   * This is similar to {@link #equals(Object)}, but does not care for
   * the {@link #getBuildString() build metadata}.
   * @param other other version
   * @return {@code true} if this and the other version have the same major, minor and patch version, and same prerelease data<br>
   *         {@code false} otherwise
   */
  public boolean equalsIgnoreBuild(SemVer other)
  {
    if (this == other) {
      return true;
    }
    if (other == null) {
      return false;
    }
    return major == other.major &&
           minor == other.minor &&
           patch == other.patch &&
           Objects.equals(prerelease, other.prerelease);
  }

  @Override
  public int hashCode()
  {
    return Objects.hash(major, minor, patch, prerelease, buildMetaData);
  }

  /**
   * Get a hash code ignoring the build metadata.
   * This is similar to {@link #hashCode()}, but does not use the
   * {@link #getBuildString() build metadata}.
   * @return hash code
   */
  public int hashCodeIgnoreBuild()
  {
    return Objects.hash(major, minor, patch, prerelease);
  }

  /**
   * Compare this semantic version with another.
   * According to the definition this completely ignores the {@link #getBuildString() build metadata}.
   * If you need sorting which includes it use {@link #compareTo(SemVer, Comparator)} instead.
   * @param o other semantic version
   * @return negative if {@code this < o},<br>
   *         positive if {@code this > o}, or<br>
   *         {@code 0} if (@code this == o}
   */
  @Override
  public int compareTo(@NotNull SemVer o)
  {
    int result = Integer.compare(major, o.major);
    if (result != 0) {
      return result;
    }

    result = Integer.compare(minor, o.minor);
    if (result != 0) {
      return result;
    }

    result = Integer.compare(patch, o.patch);
    if (result != 0) {
      return result;
    }

    if (prerelease == null) {
      return o.prerelease == null
              ? 0
              : 1;
    }
    return o.prerelease == null
            ? -1
            : prerelease.compareTo(o.prerelease);
  }

  /**
   * Compare this with another version, but don't ignore the build metadata.
   * The {@link #compareTo(SemVer) compare method} ignores the
   * {@link #getBuildString() build metadata}, while this falls back to compare it
   * if all other properties are equal.
   * @param otherVersion other version to compare
   * @param buildCompare comparator called if all other properties are equal,
   *                     will not receive {@code null} values but empty strings
   *                     instead
   * @return negative if {@code this < o},<br>
   *         positive if {@code this > o}, or<br>
   *         {@code 0} if (@code this == o}
   */
  public int compareTo(@NotNull SemVer otherVersion,
                       @NotNull Comparator<? super Indexable<String>> buildCompare)
  {
    final int result = compareTo(otherVersion);
    return result != 0
            ? result
            : buildCompare.compare(buildMetaData,
                                   otherVersion.buildMetaData);
  }

  /**
   * Is this version newer than the other version?
   * @param otherVersion other version
   * @return {@code true} if this version is newer, {@code false} if this is older or the same version
   */
  public boolean isNewerThan(@NotNull SemVer otherVersion)
  {
    return compareTo(otherVersion) > 0;
  }

  @Override
  public String toString()
  {
    String result = major + DOT + minor + DOT + patch;
    if (prerelease != null) {
      result += "-" + prerelease;
    }
    if (!buildMetaData.isEmpty()) {
      result += '+' + Types.join(DOT, buildMetaData);
    }
    return result;
  }

  /**
   * Parse a semantic version.
   * @param versionString string representation of a semantic version
   * @return semantic version, or {@code null} if {@code versionString} does not represent a valid semantic version
   */
  @Nullable
  public static SemVer parse(@NotNull String versionString)
  {
    final Matcher matcher = PATTERN.matcher(versionString);
    if (!matcher.matches()) {
      return null;
    }
    try {
      final int major = Integer.parseInt(matcher.group(RE_GRP_MAJOR));
      final int minor = Integer.parseInt(matcher.group(RE_GRP_MINOR));
      final int patch = Integer.parseInt(matcher.group(RE_GRP_PATCH));
      final String matchPrerelease = matcher.group(RE_GRP_PRERELEASE);
      final String build = matcher.group(RE_GRP_BUILD);
      final Prerelease prerelease;
      if (matchPrerelease != null) {
        prerelease = Prerelease.parseInternal(matchPrerelease);
      }
      else {
        prerelease = null;
      }
      return new SemVer(major, minor, patch, prerelease, build);
    } catch (NumberFormatException x) {
      // although regexp ensures numbers are all digits, they might still overflow
      throw new IllegalArgumentException("Version number integer overflow!", x);
    }
  }

  /**
   * Prerelease indicator.
   * <p>
   * This is the prerelease part of a semantic version.
   */
  public static final class Prerelease
          implements Indexable<Prerelease.Identifier>,
                     Comparable<Prerelease>
  {
    private static final String CONST_ALPHA = "alpha";
    private static final String CONST_BETA = "beta";
    private static final String CONST_RC = "rc";

    /** Alpha prerelease. */
    public static final Prerelease ALPHA = new Prerelease(new AlphanumericIdentifier("alpha"));
    /** Beta prerelease. */
    public static final Prerelease BETA = new Prerelease(new AlphanumericIdentifier("beta"));
    /** Release candidate prerelease. */
    public static final Prerelease RC = new Prerelease(new AlphanumericIdentifier("rc"));

    @NotNull
    private final Indexable<Identifier> parts;

    /**
     * Constructor for a simple one-identifier prerelease.
     * @param ident identifier
     */
    public Prerelease(@NotNull Identifier ident)
    {
      this(Indexable.singleton(ident));
    }

    private Prerelease(@NotNull Indexable<Identifier> parts)
    {
      this.parts = parts;
    }

    /**
     * Constructor for a multiple-identifier prerelease.
     * @param firstIdent first identifier
     * @param moreIdents second identifier
     */
    public Prerelease(@NotNull Identifier firstIdent, @NotNull  Identifier ... moreIdents)
    {
      this(new Indexable.Base<Identifier>()
      {
        @Override
        public Identifier get(int index)
        {
          return index == 0 ? firstIdent : moreIdents[index - 1];
        }

        @Override
        public int size()
        {
          return moreIdents.length + 1;
        }
      }.frozen()); // make frozen to decouple from possible moreIdents array
    }

    /**
     * Get the identifier in the given position.
     * @param index index between {@code 0} and {@code size() - 1}, from most significant to least
     * @return identifier at the given position
     */
    @Override
    public Identifier get(int index)
    {
      return parts.get(index);
    }

    /**
     * Get the number of identifiers in this prerelease definition.
     * @return number of identifiers, at least 1
     */
    @Override
    public int size()
    {
      return parts.size();
    }

    /**
     * Create a prerelase from its string representation.
     * @param prereleaseString prerelease string
     * @return parese prerelease, or {@code null} if {@code prereleaseString} is not valid
     */
    @Nullable
    public static Prerelease parse(@NotNull String prereleaseString)
    {
      return PATTERN_PRERELEASE.matcher(prereleaseString).matches()
              ? parseInternal(prereleaseString)
              : null;
    }

    @Override
    public boolean equals(Object o)
    {
      if (this == o) {
        return true;
      }
      if (o == null || getClass() != o.getClass()) {
        return false;
      }
      Prerelease that = (Prerelease)o;
      return parts.equals(that.parts);
    }

    @Override
    public int hashCode()
    {
      return Objects.hash(parts);
    }

    @Override
    public int compareTo(@NotNull Prerelease o)
    {
      return Types.lexicalCompare(parts, o.parts);
    }

    @NotNull
    @Override
    public String toString()
    {
      return Types.join(DOT, parts.view(Object::toString));
    }

    /**
     * Internal parsing.
     * This does not check the format of the incoming string.
     * @param prereleaseString textual representation of a prerelease, expected to be valid
     * @return Prerelease constructed from the string
     */
    @NotNull
    private static Prerelease parseInternal(@NotNull String prereleaseString)
    {
      final String[] parts = SEP_DOT.split(prereleaseString);
      if (parts.length == 1) {
        switch (parts[0]) {
        case CONST_ALPHA:
          return ALPHA;
        case CONST_BETA:
          return BETA;
        case CONST_RC:
          return RC;
        }
      }
      final Identifier[] idents = new Identifier[parts.length];
      for (int i = 0;  i < parts.length;  ++i) {
        final String part = parts[i];
        final Matcher matcher = PATTERN_VERSION.matcher(part);
        if (matcher.matches()) {
          try {
            idents[i] = new NumericIdentifier(Integer.parseInt(part));
            continue;
          } catch (NumberFormatException x) {
            // to large for an integer?
            Logger.getAnonymousLogger().warning("Switching to alpha-numeric prerelease because integer version is too large: "+part);
          }
        }
        idents[i] = new AlphanumericIdentifier(part);
      }
      return new Prerelease(Indexable.viewArray(idents));
    }

    /**
     * Identifier.
     */
    public interface Identifier
            extends Comparable<Identifier>
    {
      /**
       * Compare this to a numeric identifier.
       * @param identifier numeric identifier
       * @return negative if {@code this < identifier},<br>
       *         positive if {@code this > identifier}, and<br>
       *         {@code 0} if (@code this == identifier}
       */
      int compareTo(@NotNull NumericIdentifier identifier);

      /**
       * Compare this to an alphanumeric identifier.
       * @param identifier alphanumeric identifier
       * @return negative if {@code this < identifier},<br>
       *         positive if {@code this > identifier}, and<br>
       *         {@code 0} if (@code this == identifier}
       */
      int compareTo(@NotNull AlphanumericIdentifier identifier);
    }

    /**
     * Numeric identifier.
     */
    public static class NumericIdentifier
            implements Identifier
    {
      private final int id;

      /**
       * Constructor.
       * @param id numeric id
       */
      public NumericIdentifier(int id)
      {
        if (id < 0) {
          throw new IllegalArgumentException("Numeric identifier hss to be non-negative!");
        }
        this.id = id;
      }

      public int getId()
      {
        return id;
      }

      @Override
      public int compareTo(@NotNull NumericIdentifier identifier)
      {
        return Integer.compare(id, identifier.id);
      }

      @Override
      public int compareTo(@NotNull AlphanumericIdentifier identifier)
      {
        return -1;
      }

      @Override
      public int compareTo(@NotNull Identifier o)
      {
        return -o.compareTo(this);
      }

      @Override
      public boolean equals(Object o)
      {
        if (this == o) {
          return true;
        }
        if (o == null || getClass() != o.getClass()) {
          return false;
        }
        NumericIdentifier that = (NumericIdentifier)o;
        return id == that.id;
      }

      @Override
      public int hashCode()
      {
        return Objects.hash(id);
      }

      @Override
      public String toString()
      {
        return Integer.toString(id);
      }
    }

    /** Alphanumeric identifier. */
    public static class AlphanumericIdentifier
            implements Identifier
    {
      @NotNull
      private final String id;

      /**
       * Constructor.
       * @param id alphanumeric ID
       */
      public AlphanumericIdentifier(@NotNull String id)
      {
        this.id = id;
      }

      @NotNull
      public String getId()
      {
        return id;
      }

      @Override
      public int compareTo(@NotNull NumericIdentifier identifier)
      {
        return 1;
      }

      @Override
      public int compareTo(@NotNull AlphanumericIdentifier identifier)
      {
        return id.compareTo(identifier.id);
      }

      @Override
      public int compareTo(@NotNull Identifier o)
      {
        return -o.compareTo(this);
      }

      @Override
      public boolean equals(Object o)
      {
        if (this == o) {
          return true;
        }
        if (o == null || getClass() != o.getClass()) {
          return false;
        }
        final AlphanumericIdentifier that = (AlphanumericIdentifier)o;
        return id.equals(that.id);
      }

      @Override
      public int hashCode()
      {
        return Objects.hash(id);
      }

      @Override
      public String toString()
      {
        return id;
      }
    }
  }
}
