// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.version;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.Empty;
import de.caff.generics.Indexable;
import de.caff.generics.Types;

import java.io.*;
import java.nio.charset.StandardCharsets;
import java.nio.file.Files;
import java.text.SimpleDateFormat;
import java.util.Date;
import java.util.regex.Pattern;

/**
 * Helper class providing static methods to create and access
 * classes implementing {@link ModuleVersionService}.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since Dezember 28, 2021
 */
public final class ModuleVersionTool
{
  /** Format used for timestamp, included in semantic vesion as build string. */
  public static final SimpleDateFormat TIMESTAMP_FORMAT = new SimpleDateFormat("yyyyMMdd-HHmmss");

  /** Don't create. */
  private ModuleVersionTool()
  {}

  /**
   * Write a Java file which implements this service.
   * This will create the package directory if necessary, then write a Java file which contains
   * a service class implementing this interface with the given name which package/name combination which
   * returns the {@code moduleName} and {@code version} as expected.
   * @param fullClassName full class name including package
   * @param moduleName    name of module, will be returned by {@link ModuleVersionService#getModuleName()} of the created service
   * @param version       version of module, will be returned by {@link ModuleVersionService#getModuleVersion()} of the created service
   * @param releaseDate   release date
   * @param srcDir        root of the source tree, has to be an existing directory
   * @throws IOException on write or directory creation arrors
   */
  public static void createJavaImpl(@NotNull String fullClassName,
                                    @NotNull String moduleName,
                                    @NotNull SemVer version,
                                    @NotNull Date releaseDate,
                                    @NotNull File srcDir)
          throws IOException
  {
    if (!srcDir.isDirectory()) {
      throw new IOException("Source directory does not exist: " +srcDir);
    }
    final Indexable<String> parts = Indexable.viewArray(fullClassName.split(Pattern.quote(".")));
    final Indexable<String> pack = parts.headSet(-1);
    final String packageLine;
    File packageDir = srcDir;
    if (!pack.isEmpty()) {
      // make sure package directory exits
      for (String dir : pack) {
        packageDir = new File(packageDir, dir);
        if (packageDir.exists()) {
          if (!packageDir.isDirectory()) {
            throw new IOException("Required package directory already exists, but is no directory: " + packageDir);
          }
        }
        else {
          if (!packageDir.mkdir()) {
            throw new IOException("Cannot create package directory: "+packageDir);
          }
        }
      }
      packageLine = "package " + Types.join(".", pack) + "\n";
    }
    else {
      packageLine = Empty.STRING;
    }
    final String className = parts.gyt(-1);
    final File javaFile = new File(packageDir, className + ".java");
    try (OutputStream fos = Files.newOutputStream(javaFile.toPath())) {
      fos.write(String.format(JAVA_SERVICE_TEMPLATE,
                              packageLine,
                              className,
                              moduleName,
                              version.withBuildString(TIMESTAMP_FORMAT.format(releaseDate)),
                              ModuleVersionService.class.getTypeName())
                        .getBytes(StandardCharsets.UTF_8));
    }
  }

  /**
   * Extract the semantic version from a Java service implementation.
   * This is a simple-minded helper method which works with Java files created by
   * {@link #createJavaImpl(String, String, SemVer, Date, File)}, but not necessarily
   * with other files.
   * @param javaImplFile Java implementation or a module version service
   * @return semantic module version
   * @throws IOException on read errors or when no version was found
   */
  @NotNull
  public static SemVer extractFromJavaImpl(@NotNull File javaImplFile)
          throws IOException
  {
    try (BufferedReader reader = new BufferedReader(new FileReader(javaImplFile))) {
      String line;
      final String lookup = "SemVer.parse(\"";
      final int lookupLength = lookup.length();
      while ((line = reader.readLine()) != null) {
        final int pos = line.indexOf(lookup);
        if (pos >= 0) {
          final int start = pos + lookupLength;
          final int end = line.indexOf('"', start);
          if (end > 0) {
            final String versionString = line.substring(start, end);
            final SemVer version = SemVer.parse(versionString);
            if (version == null) {
              throw new IOException("Invalid semantic version string: " + versionString);
            }
            return version;
          }
        }
      }
    }
    throw new IOException("No semantic version found!");
  }

  /**
   * Extract the semantic version from a Java service implementation.
   * @param fullClassName full class name of the service implementation including package
   * @param srcDir    root directory of the Java source tree
   * @return semantic version, or {@code null} if implementation does not exist
   * @throws IOException on read errors or if the file exists but does not contain an extractable version,
   *                     compare {@link #extractFromJavaImpl(File)}
   */
  @Nullable
  public static SemVer extractFromJavaImpl(@NotNull String fullClassName,
                                           @NotNull File srcDir)
          throws IOException
  {
    if (!srcDir.isDirectory()) {
      throw new IOException("Source directory does not exist: " +srcDir);
    }
    final Indexable<String> parts = Indexable.viewArray(fullClassName.split(Pattern.quote(".")));
    final Indexable<String> pack = parts.headSet(-1);
    File packageDir = srcDir;
    for (String dir : pack) {
      packageDir = new File(packageDir, dir);
      if (!packageDir.isDirectory()) {
        return null;
      }
    }
    final File javaFile = new File(packageDir, parts.gyt(-1) + ".java");
    if (javaFile.isFile()) {
      return extractFromJavaImpl(javaFile);
    }
    return null;
  }

  /**
   * Service class template.
   * This is used as a format string for {@code String.format()}, where it expects the following arguments:
   * <ol>
   *   <li>The package of the created class, inluding the {@code package} keyword .</li>
   *   <li>The class name w/o package.</li>
   *   <li>The name of the module for which the service is created.</li>
   *   <li>The semantic version for which this is created.</li>
   *   <li>The creating class.</li>
   * </ol>
   */
  public static final String JAVA_SERVICE_TEMPLATE =
          "// This class is automatically created by %5$s.\n" +
          "%1$s;" +
          "\n" +
          "import de.caff.annotation.NotNull;\n" +
          "import de.caff.generics.Types;\n" +
          "import de.caff.version.ModuleVersionService;\n" +
          "import de.caff.version.SemVer;\n" +
          "\n" +
          "import java.util.Objects;\n" +
          "\n" +
          "/**\n" +
          " * Module version information.\n" +
          " * This automatically created class provides version information for the module in which it is contained.\n" +
          " * Note that creating this class is only the first step, it is necessary to include it in the\n" +
          " * META-INF/services directory in the module's jar.\n" +
          " */\n" +
          "public class %2$s\n" +
          "        implements ModuleVersionService\n" +
          "{\n" +
          "  /** Module name. */\n" +
          "  @NotNull\n" +
          "  public static final String MODULE_NAME = \"%3$s\";\n" +
          "  /** Module version as a constant. */\n" +
          "  @NotNull\n" +
          "  public static final SemVer VERSION = Objects.requireNonNull(SemVer.parse(\"%4$s\"));\n" +
          "\n" +
          "  @NotNull\n" +
          "  @Override\n" +
          "  public String getModuleName()\n" +
          "  {\n" +
          "    return MODULE_NAME;\n" +
          "  }\n" +
          "\n" +
          "  @NotNull\n" +
          "  @Override\n" +
          "  public SemVer getModuleVersion()\n" +
          "  {\n" +
          "    return VERSION;\n" +
          "  }\n" +
          "\n" +
          "  /**\n" +
          "   * Get the release date.\n" +
          "   * This assumes that the release date is included in {@link #VERSION} as build string.\n" +
          "   * @return build string of the version, empty if there is none\n" +
          "   */\n" +
          "  @NotNull\n" +
          "  public static String getReleaseDate()\n" +
          "  {\n" +
          "    return Types.notNull(VERSION.getBuildString());\n" +
          "  }\n" +
          "\n" +
          "  @NotNull\n" +
          "  @Override\n" +
          "  public String toString()\n" +
          "  {\n" +
          "    return String.format(\"%%s: %%s\", MODULE_NAME, VERSION);\n" +
          "  }\n" +
          "}\n";

}
