// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.version;

import de.caff.annotation.NotNull;
import de.caff.generics.FilteringIteratorWrapper;
import de.caff.generics.Indexable;
import de.caff.generics.Types;

import java.util.Iterator;
import java.util.ServiceLoader;

/**
 * Service interface used for retrieving module versions.
 * <p>
 * It is expected that there is a class in each module which implements this service.
 * There is no Java-intrinsic automatism how this implementations are created, and how
 * they are announced in META-INF/services.
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since December 28, 2021
 */
public interface ModuleVersionService
{
  /**
   * Get the name of the module for which the version is given.
   * @return module name
   */
  @NotNull
  String getModuleName();

  /**
   * Get the current version of the module for which this service is responsible.
   * @return module version
   */
  @NotNull
  SemVer getModuleVersion();

  /**
   * Get an iterator over all module version services.
   * This reloads everything, so under normal circumstances (e.g. no dynamic class reloading)
   * the result should be cached.
   * @return iterator over all provided module version services
   */
  @NotNull
  static Iterator<ModuleVersionService> iterator()
  {
    final ServiceLoader<ModuleVersionService> loader = ServiceLoader.load(ModuleVersionService.class);
    return loader.iterator();
  }

  /**
   * Get an iterator over all module version services with one excluded.
   * This is useful to collect all modules on which the excluded one depends.
   * @param excludedModule excluded module
   * @return all modules but the excluded one
   */
  @NotNull
  static Iterator<ModuleVersionService> iterator(@NotNull String excludedModule)
  {
    return new FilteringIteratorWrapper<>(iterator(), mvs -> !excludedModule.equals(mvs.getModuleName()));
  }

  /**
   * Collect the module version services and cache them in an indexable.
   * This takes a snapshot of the currently known services. If modules are
   * added dynamically later on the versions of these modules will not be contained.
   * @return currently available module version services
   */
  @NotNull
  static Indexable<ModuleVersionService> collectServices()
  {
    return Indexable.fromIterator(iterator());
  }

  /**
   * Collect the module version services and cache them in an indexable.
   * This is useful to collect all modules on which the excluded one depends.
   * This takes a snapshot of the currently known services. If modules are
   * added dynamically later on the versions of these modules will not be contained.
   * @param excludedModule excluded module
   * @return currently available module version services
   */
  @NotNull
  static Indexable<ModuleVersionService> collectServices(@NotNull String excludedModule)
  {
    return Indexable.fromIterator(iterator(excludedModule));
  }

  /**
   * Create a string for depending module versions.
   * .
   * @param prefix          prefix of the output, determines indentation of further lines
   * @param excludedModule  module which is depending on others
   * @return multi-line string
   */
  @NotNull
  static String makeDependsOutput(@NotNull String prefix,
                                  @NotNull String excludedModule)
  {
    final StringBuilder sb = new StringBuilder(prefix.length());
    for (int i = 0;  i < prefix.length();  ++i) {
      final char ch = prefix.charAt(i);
      if (ch == '\n') {
        sb.setLength(0);
        continue;
      }
      sb.append(Character.isSpaceChar(ch) ? ch : ' ');
    }
    final String join = "\n" + sb;
    return prefix + Types.join(join, collectServices(excludedModule).view(Object::toString));
  }
}
