// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2021-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================
package de.caff.generics.util.combi;

import de.caff.annotation.NotNull;
import de.caff.annotation.Nullable;
import de.caff.generics.*;

import java.math.BigInteger;
import java.util.*;

/**
 * Helper class for permutations.
 * <p>
 * This class takes care of permutating a set of generic values.
 * Evaluation is done lazy using the Steinhaus-Johnson-Trotter algorithm.
 * As the number of permutations becomes large very fast, this class does not
 * provide a size which would tell how many elements will be returned from the iterator
 * until it finally stops
 * (obviously {@code n!} when the number of elements to be rearranged is {@code n}),
 * <p>
 * There are various subclasses which take care of permutations of primitive values
 * avoiding boxing and unboxing overhead:
 * <ul>
 *   <li>
 *     {@link OfRange} is the most basic and takes care of rearranging a
 *     range of consecutive indices.
 *     Indeed all other permutations are based on {@link RangeIterator}
 *     which is used without overhead only by {@code OfRange}.
 *   </li>
 *   <li>{@link OfInt} rearranges arbitrary {@code int} values.</li>
 *   <li>{@link OfLong} rearranges arbitrary {@code long} values.</li>
 *   <li>{@link OfShort} rearranges arbitrary {@code short} values.</li>
 *   <li>{@link OfByte} rearranges arbitrary {@code byte} values.</li>
 *   <li>{@link OfDouble} rearranges arbitrary {@code double} values.</li>
 *   <li>{@link OfFloat} rearranges arbitrary {@code float} values.</li>
 *   <li>{@link OfChar} rearranges arbitrary {@code char} values.</li>
 *   <li>{@link OfString} rearranges the characters of a {@code java.lang.CharSequence}.</li>
 *   <li>{@link OfBoolean} rearranges arbitrary {@code byte} values.</li>
 * </ul>
 *
 * @param <T> type pf elements which will be permuted
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 17, 2021
 */
public class Permutations<T>
        implements Iterable<Indexable<T>>
{
  @NotNull
  private final Indexable<T> elements;

  /**
   * Constructor.
   * @param set elements which will be permuted
   */
  public Permutations(@NotNull Indexable<T> set)
  {
    this.elements = set;
  }

  /**
   * Constructor.
   * @param list elements which will be permuted
   */
  public Permutations(@NotNull List<T> list)
  {
    this(Indexable.viewList(list));
  }

  /**
   * Constructor.
   * @param set elements which will be permuted
   */
  public Permutations(@NotNull Countable<T> set)
  {
    this(set.toList());
  }

  /**
   * Constructor.
   * @param set elements which will be permuted
   */
  public Permutations(@NotNull Collection<T> set)
  {
    this(Countable.viewCollection(set));
  }

  /**
   * Constructor.
   * @param elems elements which will be permuted
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  public Permutations(@NotNull T... elems)
  {
    this(Indexable.viewArray(elems));
  }

  /**
   * Get the iterator which provides the permutations.
   * @return permutations
   */
  @Override
  @NotNull
  public Iterator<Indexable<T>> iterator()
  {
    return new IteratorConverter<>(new RangeIterator(elements.size()),
                                   rangeIdx -> rangeIdx.view(elements::get));
  }

  /**
   * The permutations of an index range.
   * <p>
   * The iterator will return an indexable with integer values, and you can
   * use these values as indices into any set you have, although in most cases
   * using the enclosing class {@link Permutations} is simpler because it already
   * handles this for you.
   */
  public static class OfRange
          implements Iterable<IntIndexable>
  {
    private final int size;

    /**
     * Constructor.
     * This permutation will rearrange the numbers from {@code 0} to {@code size - 1}.
     * @param size size of the range, non-negative
     */
    public OfRange(int size)
    {
      if (size < 0) {
        throw new IllegalArgumentException("size has to be non-negative!");
      }
      this.size = size;
    }

    /**
     * Get the size of the range.
     * @return size of this range
     */
    public int getSize()
    {
      return size;
    }

    @Override
    public Iterator<IntIndexable> iterator()
    {
      return new RangeIterator(size);
    }
  }

  /**
   * The permutations of integer values.
   */
  public static class OfInt
          implements Iterable<IntIndexable>
  {
    @NotNull
    private final IntIndexable values;

    /**
     * Constructor.
     * @param values int indexable which will be permuted
     */
    public OfInt(@NotNull IntIndexable values)
    {
      this.values = values;
    }

    /**
     * Constructor.
     * @param values int values which will be permuted
     */
    public OfInt(@NotNull int ... values)
    {
      this(IntIndexable.viewArray(values));
    }

    @Override
    public Iterator<IntIndexable> iterator()
    {
      return new IteratorConverter<>(new RangeIterator(values.size()),
                                     rangeIdx -> new IntIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public int get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * The permutations of byte values.
   */
  public static class OfByte
          implements Iterable<ByteIndexable>
  {
    @NotNull
    private final ByteIndexable values;

    /**
     * Constructor.
     * @param values byte indexable which will be permuted
     */
    public OfByte(@NotNull ByteIndexable values)
    {
      this.values = values;
    }

    /**
     * Constructor.
     * @param values byte values which will be permuted
     */
    public OfByte(@NotNull byte ... values)
    {
      this(ByteIndexable.viewArray(values));
    }

    @Override
    public Iterator<ByteIndexable> iterator()
    {
      return new IteratorConverter<>(new RangeIterator(values.size()),
                                     rangeIdx -> new ByteIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public byte get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * The permutations of long values.
   */
  public static class OfLong
          implements Iterable<LongIndexable>
  {
    @NotNull
    private final LongIndexable values;

    /**
     * Constructor.
     * @param values int indexable which will be permuted
     */
    public OfLong(@NotNull LongIndexable values)
    {
      this.values = values;
    }

    /**
     * Constructor.
     * @param values int values which will be permuted
     */
    public OfLong(@NotNull long ... values)
    {
      this(LongIndexable.viewArray(values));
    }

    @Override
    public Iterator<LongIndexable> iterator()
    {
      return new IteratorConverter<>(new RangeIterator(values.size()),
                                     rangeIdx -> new LongIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public long get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * The permutations of short values.
   */
  public static class OfShort
          implements Iterable<ShortIndexable>
  {
    @NotNull
    private final ShortIndexable values;

    /**
     * Constructor.
     * @param values int indexable which will be permuted
     */
    public OfShort(@NotNull ShortIndexable values)
    {
      this.values = values;
    }

    /**
     * Constructor.
     * @param values int values which will be permuted
     */
    public OfShort(@NotNull short ... values)
    {
      this(ShortIndexable.viewArray(values));
    }

    @Override
    public Iterator<ShortIndexable> iterator()
    {
      return new IteratorConverter<>(new RangeIterator(values.size()),
                                     rangeIdx -> new ShortIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public short get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * The permutations of char values.
   * @see OfString
   */
  public static class OfChar
          implements Iterable<CharIndexable>
  {
    @NotNull
    private final CharIndexable values;

    /**
     * Constructor.
     * @param values char indexable which will be permuted
     */
    public OfChar(@NotNull CharIndexable values)
    {
      this.values = values;
    }

    /**
     * Constructor.
     * @param values char values which will be permuted
     */
    public OfChar(@NotNull char ... values)
    {
      this(CharIndexable.viewArray(values));
    }

    /**
     * Constructor.
     * @param text text which chars will be permuted
     */
    public OfChar(@NotNull String text)
    {
      this(CharIndexable.viewString(text));
    }

    @Override
    public Iterator<CharIndexable> iterator()
    {
      return new IteratorConverter<>(new RangeIterator(values.size()),
                                     rangeIdx -> new CharIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public char get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * The permutations of the characters of a string, as strings.
   * @see OfChar
   */
  public static class OfString implements Iterable<String>
  {
    @NotNull
    private final CharSequence str;

    /**
     * Constructor.
     * @param string string which will be permuted
     */
    public OfString(@NotNull CharSequence string)
    {
      str = string;
    }

    @Override
    public Iterator<String> iterator()
    {
      final int size = str.length();
      return new IteratorConverter<>(new RangeIterator(size),
                                     rangeIdx -> {
                                       final StringBuilder sb = new StringBuilder(size);
                                       for (PrimitiveIterator.OfInt it = rangeIdx.intIterator();  it.hasNext(); ) {
                                         final int idx = it.nextInt();
                                         sb.append(str.charAt(idx));
                                       }
                                       return sb.toString();
                                     });
    }
  }

  /**
   * The permutations of boolean values.
   */
  public static class OfBoolean
          implements Iterable<BooleanIndexable>
  {
    @NotNull
    private final BooleanIndexable values;

    /**
     * Constructor.
     * @param values boolean indexable which will be permuted
     */
    public OfBoolean(@NotNull BooleanIndexable values)
    {
      this.values = values;
    }

    /**
     * Constructor.
     * @param values int values which will be permuted
     */
    public OfBoolean(@NotNull boolean ... values)
    {
      this(BooleanIndexable.viewArray(values));
    }

    @Override
    public Iterator<BooleanIndexable> iterator()
    {
      return new IteratorConverter<>(new RangeIterator(values.size()),
                                     rangeIdx -> new BooleanIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public boolean get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * The permutations of double values.
   */
  public static class OfDouble
          implements Iterable<DoubleIndexable>
  {
    @NotNull
    private final DoubleIndexable values;

    /**
     * Constructor.
     * @param values int indexable which will be permuted
     */
    public OfDouble(@NotNull DoubleIndexable values)
    {
      this.values = values;
    }

    /**
     * Constructor.
     * @param values double values which will be permuted
     */
    public OfDouble(@NotNull double ... values)
    {
      this(DoubleIndexable.viewArray(values));
    }

    @Override
    public Iterator<DoubleIndexable> iterator()
    {
      return new IteratorConverter<>(new RangeIterator(values.size()),
                                     rangeIdx -> new DoubleIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public double get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * The permutations of float values.
   */
  public static class OfFloat
          implements Iterable<FloatIndexable>
  {
    @NotNull
    private final FloatIndexable values;

    /**
     * Constructor.
     * @param values flaot indexable which will be permuted
     */
    public OfFloat(@NotNull FloatIndexable values)
    {
      this.values = values;
    }

    /**
     * Constructor.
     * @param values float values which will be permuted
     */
    public OfFloat(@NotNull float ... values)
    {
      this(FloatIndexable.viewArray(values));
    }

    @Override
    public Iterator<FloatIndexable> iterator()
    {
      return new IteratorConverter<>(new RangeIterator(values.size()),
                                     rangeIdx -> new FloatIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public float get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * Iterator using the Steinhaus-Johnson-Trotter algorithm for an index range.
   * All other permutation iterators can make use of this.
   */
  public static class RangeIterator implements Iterator<IntIndexable>
  {
    @NotNull
    private final MutableIndexable<IndexAndDir> indexes;

    @Nullable
    private MutableIntIndexable nextPermutation;

    /**
     * Constructor.
     * @param numElements number of elements
     */
    public RangeIterator(int numElements)
    {
      final IntIndexable range = IntIndexable.rangeFromSize(numElements);
      indexes = MutableIndexable.fromIndexable(range.view(IndexAndDir::new));
      nextPermutation = MutableIntIndexable.fromIndexable(indexes, idx -> idx.index);
    }

    @Override
    public boolean hasNext()
    {
      return nextPermutation != null;
    }

    @Override
    public IntIndexable next()
    {
      if (nextPermutation == null) {
        throw new NoSuchElementException();
      }

      final IntIndexable result = nextPermutation.frozen();

      // find the largest mobile integer k
      int largestMobileIndex = -1;
      int largestMobile = -1;
      final int size = indexes.size();
      for (int i = 0; i < size; ++i) {
        final IndexAndDir kd = indexes.get(i);
        final int index = kd.index;
        final boolean dir = kd.dir;
        if ((dir  &&  i < size - 1  && index > indexes.get(i + 1).index) ||
            (!dir  &&  i > 0  && index > indexes.get(i - 1).index)) {
          if (index > largestMobile) { // NOPMD
            largestMobile = index;
            largestMobileIndex = i;
          }
        }
      }
      if (largestMobile < 0) {
        // finished
        nextPermutation = null;
      }
      else {
        // calculate next permutation
        indexes.swap(largestMobileIndex, largestMobileIndex + indexes.get(largestMobileIndex).getOffset());

        int idx = 0;
        for (IndexAndDir id : indexes) {
          if (id.index > largestMobile) {
            id.dir = !id.dir;
          }
          nextPermutation.set(idx++, id.index);
        }
      }
      return result;
    }

    private static class IndexAndDir
    {
      final int index;
      boolean dir;

      IndexAndDir(int index)
      {
        this.index = index;
        dir = false;
      }

      public int getOffset()
      {
        return dir ? 1 : -1;
      }

      @Override
      public String toString()
      {
        return String.format("{%d,%s}", index, dir);
      }
    }
  }

  /**
   * Get the number of permutations for a set of size n.
   * @param n set size. at least {@code 0}
   * @return number of permutations, equal to n!
   */
  @NotNull
  public static BigInteger count(int n)
  {
    if (n < 0) {
      throw new IllegalArgumentException("Set size of "+n+" is not possible!");
    }
    return factorial(n);
  }

  @NotNull
  static BigInteger factorial(int n)
  {
    if (n < FIRST_FACTORIALS.size()) {
      return FIRST_FACTORIALS.get(n);
    }
    BigInteger f = FIRST_FACTORIALS.gyt(-1);
    for (int i = FIRST_FACTORIALS.size();  i <= n;  ++i) {
      f = f.multiply(BigInteger.valueOf(i));
    }
    return f;
  }

  /** Precalculated set of factorials (big int). */
  static final Indexable<BigInteger> FIRST_FACTORIALS;
  /** Precalculated set of factorials (long). */
  static final LongIndexable FIRST_FACTORIALS_LONG;
  static {
    final long[] factorials = new long[21]; // 20! is the largest factorial possible for long
    factorials[0] = factorials[1] = 1;
    for (int n = 2;  n < factorials.length;  ++n) {
      factorials[n] = factorials[n - 1] * n;
    }
    FIRST_FACTORIALS_LONG = LongIndexable.viewArray(factorials);
    FIRST_FACTORIALS = FIRST_FACTORIALS_LONG.view(BigInteger::valueOf).frozen();
  }

  /**
   * Test code.
   * @param args a number
   */
  public static void main(@NotNull String[] args)
  {
    final int size = args.length > 0
            ? Integer.parseInt(args[0])
            : 4;
    if (true) {
      final Indexable<String> testSet =
              IntIndexable.range('A', 'A' + size - 1).view(i -> Character.toString((char)i));
      final Set<Indexable<String>> check = new HashSet<>();
      int count = 0;
      for (Indexable<String> perm : new Permutations<>(testSet)) {
        System.out.println(perm);
        if (check.contains(perm)) {
          System.err.printf("Duplicate: %s\n", perm);
        }
        check.add(perm);
        ++count;
      }
      if (count != factorial(size).intValue()) {
        System.err.println("Incorrect count!");
      }
    }
    else {
      for (Iterator<IntIndexable> it = new RangeIterator(size); it.hasNext(); ) {
        System.out.println(it.next());
      }
    }
  }
}
