// ============================================================================
// File:               IntPartition
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2023-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            5/25/23 5:30 PM
//=============================================================================
package de.caff.generics.util.combi;

import de.caff.annotation.NotNull;
import de.caff.generics.Indexable;

import java.util.*;

/**
 * Partition of a positive integer value.
 * <p>
 * An <a href="https://en.wikipedia.org/wiki/Partition_(number_theory)">integer partition</a>
 * contains all sets of numbers which add up to this value. This class allows to iterate over all
 * unique partitions sequencing from higher to lower numbers.
 * <p>
 * E.g. for the number 4 this will return the following sequence of indexables of {@link Part}s:
 * <table border="1" summary="Example Partition">
 *   <tr><th>Result (count*number,...)</th><th>Meaning</th></tr>
 *   <tr><td>{@code 1*4}</td><td>{@code 4 = 4}</td></tr>
 *   <tr><td>{@code 1*3,1*1}</td><td>{@code 4 = 3 + 1}</td></tr>
 *   <tr><td>{@code 2*2}</td><td>{@code 4 = 2 + 2}</td></tr>
 *   <tr><td>{@code 1*2,2*2}</td><td>{@code 4 = 2 + 1 + 1}</td></tr>
 *   <tr><td>{@code 4*1}</td><td>{@code 4 = 1 + 1 + 1 + 1}</td></tr>
 * </table>
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since May 25, 2023
 */
public class IntPartitions
{
  /**
   * One part of a partition.
   * This is just the given number and how often it is contained.
   */
  public static class Part
  {
    /** Number. A positive number. */
    public final int number;
    /** How often the number is contained in the partition. */
    public final int count;

    /**
     * Constructor.
     * @param number the number
     * @param count  how often it is contained
     */
    Part(int number, int count)
    {
      this.number = number;
      this.count = count;
    }

    /**
     * Get the number.
     * @return a positive number which is part of the partition
     */
    public int getNumber()
    {
      return number;
    }

    /**
     * Get how often the {@link #getNumber()} is contained in the partition.
     * @return count of the number in the partition
     */
    public int getCount()
    {
      return count;
    }

    @Override
    public boolean equals(Object o)
    {
      if (this == o) {
        return true;
      }
      if (!(o instanceof Part)) {
        return false;
      }
      final Part that = (Part)o;
      return number == that.number && count == that.count;
    }

    @Override
    public int hashCode()
    {
      return Objects.hash(number, count);
    }

    @Override
    public String toString()
    {
      return String.format("%d*%d", count, number);
    }
  }

  /** One partition. */
  public static class Partition
          implements Indexable<Part>
  {
    @NotNull
    private final Part[] parts;
    private final int sumOfCounts;

    private Partition(@NotNull Stack<IntPartitionIterator.Unit> units)
    {
      parts = new Part[units.size()];
      int i = 0;
      int sum = 0;
      for (IntPartitionIterator.Unit unit : units) {
        parts[i++] = unit.part();
        sum += unit.count;
      }
      sumOfCounts = sum;
    }

    /**
     * Get the sum over all counts.
     * @return sum of the {@link Part#getCount() counts} of all parts used in this partition
     */
    public int getSumOfCounts()
    {
      return sumOfCounts;
    }

    @Override
    public Part get(int index)
    {
      return parts[index];
    }

    @Override
    public int size()
    {
      return parts.length;
    }

    @NotNull
    @Override
    public Partition frozen()
    {
      return this;
    }
  }

  /**
   * Helper class performing the iteration over the partitions.
   */
  private static class IntPartitionIterator
          implements Iterator<Partition>
  {
    /**
     * One unit.
     * Basically the same as {@link Part}, but mutable.
     * Use {@link #part()} to convert it to an immutable part.
     */
    private static class Unit
    {
      int number;
      int count;

      Unit(int number, int count)
      {
        this.number = number;
        this.count = count;
      }

      @NotNull
      IntPartitions.Part part()
      {
        return new Part(number, count);
      }
    }

    /** Stack of currently active units. */
    private final Stack<Unit> stack = new Stack<>();
    /** Are there any more partitions? */
    private boolean hasNext = true;

    /**
     * Constructor.
     * @param n number for which partitions are requested
     */
    private IntPartitionIterator(int n)
    {
      stack.push(new Unit(n, 1));
    }

    @Override
    public boolean hasNext()
    {
      return hasNext;
    }

    @Override
    public Partition next()
    {
      if (!hasNext) {
        throw new NoSuchElementException("No more partitions left!");
      }
      // get partition and freeze it before next partition is prepared
      final Partition result = new Partition(stack);
      hasNext = prepareNext();
      return result;
    }

    /**
     * Prepare the next partition.
     * @return whether there are more partitions
     */
    private boolean prepareNext()
    {
      if (stack.get(0).number == 1) {
        return false;
      }
      int p = 0;
      while (true) {
        final Unit part = stack.peek();
        p += part.number;

        --part.count;
        final int is = part.number - 1;
        if (is == 0) {
          p += part.count;
          stack.pop();
        }
        else {
          if (part.count == 0) {
            stack.pop();
          }
          final int js = p / is;
          if (js != 0) {
            stack.push(new Unit(is, js));
            p -= is*js;
            if (p == 0) {
              return true;
            }
          }
          stack.push(new Unit(p, 1));
          return true;
        }
      }
    }
  }

  /**
   * Get an iterable iterating over the integer partitions of the given number.
   * @param n a positive number
   * @return an iterable which will iterate over the given partitions
   */
  @NotNull
  public static Iterable<Partition> of(int n)
  {
    if (n <= 0) {
      throw new IllegalArgumentException("Cannot only create partitions for positive values, not "+n);
    }
    return () -> new IntPartitionIterator(n);
  }

  /**
   * Test partitioning.
   * @param args at least one positive number
   */
  public static void main(@NotNull String[] args)
  {
    if (args.length == 0) {
      System.err.println("Need at least one number!");
      System.exit(1);
    }
    for (String arg: args) {
      System.out.printf("%s:\n", arg);
      final int n = Integer.parseInt(arg);
      for (Indexable<IntPartitions.Part> parts : of(n)) {
        System.out.println(parts);
      }
      System.out.println();
    }
  }
}
