// ============================================================================
// File:               Combinations
//
// Project:            CAFF
//
// Purpose:            
//
// Author:             Rammi
//
// Copyright Notice:   © 2021-2024  Rammi (rammi@caff.de)
//                     The usage of this source code in commercial or open 
//                     source projects is not allowed without explicit 
//                     permission.
//
// Created:            18.11.21 12:32
//=============================================================================
package de.caff.generics.util.combi;

import de.caff.annotation.NotNull;
import de.caff.generics.*;

import java.math.BigInteger;
import java.util.*;

/**
 * Create all combinations.
 * This class takes care of creating all combination of <i>n choose k</i> lazily
 * from a given set of objects.
 * <p>
 * Combinations returned will be ordered when considering the index into the incoming elements:
 * <ul>
 *   <li>
 *     Each single combination will appear in the order of the incoming elements.
 *     i.e. the each element has in the base set is ascending.
 *   </li>
 *   <li>
 *     Steps will be ordered lexicographic by their indexes into the incoming elements
 *     when viewed inverted (from end to start). I.e. a choice of 2 from the string
 *     {@code "ABCD"} will return
 *     <ul>
 *       <li>{@code "AB"}</li>
 *       <li>{@code "AC"}</li>
 *       <li>{@code "BC"}</li>
 *       <li>{@code "AD"}</li>
 *       <li>{@code "BD"}</li>
 *       <li>{@code "CD"}</li>
 *     </ul>
 *     When you read each result from <b>right to left</b> they are in strict lexical order.
 *   </li>
 * </ul>
 *
 * There are various subclasses which take care of combinations of primitive values
 * avoiding boxing and unboxing overhead:
 * <ul>
 *   <li>
 *     {@link OfRange} is the most basic and takes care of combinations of consecutive indices.
 *     Indeed all other combinations classes are based on {@link #createRangeIterator(int, int)}
 *     which is used without overhead only by {@code OfRange}.
 *   </li>
 *   <li>{@link OfInt} handle combinations of arbitrary {@code int} values.</li>
 *   <li>{@link OfLong} handles combinations of arbitrary {@code long} values.</li>
 *   <li>{@link OfShort} handles combinatons of arbitrary {@code short} values.</li>
 *   <li>{@link OfByte} handles combinations of arbitrary {@code byte} values.</li>
 *   <li>{@link OfDouble} handles combinations of arbitrary {@code double} values.</li>
 *   <li>{@link OfFloat} handles combintaions of arbitrary {@code float} values.</li>
 *   <li>{@link OfChar} handles combinations of arbitrary {@code char} values.</li>
 *   <li>{@link OfString} handles combinations of the characters of a {@code java.lang.CharSequence}.</li>
 *   <li>{@link OfBoolean} handles combintaions of arbitrary {@code byte} values.</li>
 * </ul>
 *
 * @param <T> element type of the incoming set
 *
 * @author <a href="mailto:rammi@caff.de">Rammi</a>
 * @since November 18, 2021
 */
public class Combinations<T>
        implements Iterable<Indexable<T>>
{
  /** Elements of the set from which is chosen. */
  @NotNull
  private final Indexable<T> elements;
  /** Number of elements to choose from the incoming set. */
  private final int k;

  /**
   * Constructor.
   * @param k   number of elements to choose, has to be non-negative
   *            and not more the size of the {@code elements} set
   * @param elements set from which elements are chosen
   */
  public Combinations(int k, @NotNull Indexable<T> elements)
  {
    checkBinomial(elements.size(), k);
    this.elements = elements;
    this.k = k;
  }

  /**
   * Constructor.
   *
   * @param k   number of elements to choose, has to be non-negative
   *            and not more the size of the {@code list}
   * @param list list from which elements will be cosen
   */
  public Combinations(int k, @NotNull List<T> list)
  {
    this(k, Indexable.viewList(list));
  }

  /**
   * Constructor.
   * @param k   number of elements to choose, has to be non-negative
   *            and not more the size of the {@code set}
   * @param set set from which elements will be chosen
   */
  public Combinations(int k, @NotNull Countable<T> set)
  {
    this(k, set.toList());
  }

  /**
   * Constructor.
   * @param k   number of elements to choose, has to be non-negative
   *            and not more the size of the {@code set}
   * @param set set from which elements will be chosen
   */
  public Combinations(int k, @NotNull Collection<T> set)
  {
    this(k, Countable.viewCollection(set));
  }

  /**
   * Constructor.
   * @param k   number of elements to choose, has to be non-negative
   *            and not more the number of {@code elements}
   * @param elems set from which elements will be chosen
   */
  @SafeVarargs
  @SuppressWarnings("varargs")
  public Combinations(int k, @NotNull T... elems)
  {
    this(k, Indexable.viewArray(elems));
  }

  @Override
  @NotNull
  public Iterator<Indexable<T>> iterator()
  {
    return new IteratorConverter<>(createRangeIterator(elements.size(), k),
                                   rangeIdx -> rangeIdx.view(elements::get));
  }

  /**
   * The combinations of an index range.
   * <p>
   * The iterator will return an indexable with integer values, and you can
   * use these values as indices into any set you have, although in most cases
   * using the enclosing class {@link Combinations} is simpler because it already
   * handles this for you.
   */
  public static class OfRange
          implements Iterable<IntIndexable>
  {
    private final int k;
    private final int n;

    /**
     * Constructor.
     * @param n size of the range, non-negative
     * @param k size of the subset to be chosen, non-negative and not more than {@code n}
     */
    public OfRange(int n, int k)
    {
      checkBinomial(n, k);
      this.n = n;
      this.k = k;
    }

    /**
     * Get the size of the range.
     * @return size of this range
     */
    public int getSize()
    {
      return n;
    }

    @Override
    @NotNull
    public Iterator<IntIndexable> iterator()
    {
      return createRangeIterator(n, k);
    }
  }

  /**
   * The combinations of integer values.
   */
  public static class OfInt
          implements Iterable<IntIndexable>
  {
    private final int k;
    @NotNull
    private final IntIndexable values;

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values set of numbers from which there will be chosen
     */
    public OfInt(int k, @NotNull IntIndexable values)
    {
      checkBinomial(values.size(), k);
      this.k = k;
      this.values = values;
    }

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values set of numbers from which there will be chosen
     */
    public OfInt(int k, @NotNull int ... values)
    {
      this(k, IntIndexable.viewArray(values));
    }

    @Override
    public Iterator<IntIndexable> iterator()
    {
      return new IteratorConverter<>(createRangeIterator(values.size(), k),
                                     rangeIdx -> new IntIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public int get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * Combinations of byte values.
   */
  public static class OfByte
          implements Iterable<ByteIndexable>
  {
    private final int k;
    @NotNull
    private final ByteIndexable values;

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values byte indexable from which there will be chosen
     */
    public OfByte(int k, @NotNull ByteIndexable values)
    {
      checkBinomial(values.size(), k);
      this.k = k;
      this.values = values;
    }

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values byte indexable from which there will be chosen
     */
    public OfByte(int k, @NotNull byte ... values)
    {
      this(k, ByteIndexable.viewArray(values));
    }

    @Override
    public Iterator<ByteIndexable> iterator()
    {
      return new IteratorConverter<>(createRangeIterator(values.size(), k),
                                     rangeIdx -> new ByteIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public byte get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * Combinations of long values.
   */
  public static class OfLong
          implements Iterable<LongIndexable>
  {
    private final int k;
    @NotNull
    private final LongIndexable values;

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values long indexable from which there will be chosen
     */
    public OfLong(int k, @NotNull LongIndexable values)
    {
      checkBinomial(values.size(), k);
      this.k = k;
      this.values = values;
    }

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values long values from which there will be chosen
     */
    public OfLong(int k,
                  @NotNull long ... values)
    {
      this(k, LongIndexable.viewArray(values));
    }

    @Override
    public Iterator<LongIndexable> iterator()
    {
      return new IteratorConverter<>(createRangeIterator(values.size(), k),
                                     rangeIdx -> new LongIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public long get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * Combinations of short values.
   */
  public static class OfShort
          implements Iterable<ShortIndexable>
  {
    private final int k;
    @NotNull
    private final ShortIndexable values;

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values short indexable from which there will be chosen
     */
    public OfShort(int k, @NotNull ShortIndexable values)
    {
      checkBinomial(values.size(), k);
      this.k = k;
      this.values = values;
    }

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values short values from which there will be chosen
     */
    public OfShort(int k, @NotNull short ... values)
    {
      this(k, ShortIndexable.viewArray(values));
    }

    @Override
    public Iterator<ShortIndexable> iterator()
    {
      return new IteratorConverter<>(createRangeIterator(values.size(), k),
                                     rangeIdx -> new ShortIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public short get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * Combinations of char values.
   * @see OfString
   */
  public static class OfChar
          implements Iterable<CharIndexable>
  {
    private final int k;
    @NotNull
    private final CharIndexable values;

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values char indexable from which there will be chosen
     */
    public OfChar(int k, @NotNull CharIndexable values)
    {
      checkBinomial(values.size(), k);
      this.k = k;
      this.values = values;
    }

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values char values from which there will be chosen
     */
    public OfChar(int k, @NotNull char ... values)
    {
      this(k, CharIndexable.viewArray(values));
    }

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param text text which chars will be permutated
     */
    public OfChar(int k, @NotNull String text)
    {
      this(k, CharIndexable.viewString(text));
    }

    @Override
    public Iterator<CharIndexable> iterator()
    {
      return new IteratorConverter<>(createRangeIterator(values.size(), k),
                                     rangeIdx -> new CharIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public char get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * The combinations of the characters of a string, as strings.
   * @see OfChar
   */
  public static class OfString implements Iterable<String>
  {
    private final int k;
    @NotNull
    private final CharSequence str;

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code string}
     * @param string string which will be chosen
     */
    public OfString(int k, @NotNull CharSequence string)
    {
      checkBinomial(string.length(), k);
      this.k = k;
      str = string;
    }

    @Override
    public Iterator<String> iterator()
    {
      return new IteratorConverter<>(createRangeIterator(str.length(), k),
                                     rangeIdx -> {
                                       final StringBuilder sb = new StringBuilder(k);
                                       for (PrimitiveIterator.OfInt it = rangeIdx.intIterator();  it.hasNext(); ) {
                                         final int idx = it.nextInt();
                                         sb.append(str.charAt(idx));
                                       }
                                       return sb.toString();
                                     });
    }
  }

  /**
   * Combinations of boolean values.
   */
  public static class OfBoolean
          implements Iterable<BooleanIndexable>
  {
    private final int k;
    @NotNull
    private final BooleanIndexable values;

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values boolean indexable from which there will be chosen
     */
    public OfBoolean(int k, @NotNull BooleanIndexable values)
    {
      checkBinomial(values.size(), k);
      this.k = k;
      this.values = values;
    }

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values int values which will be permutated
     */
    public OfBoolean(int k, @NotNull boolean ... values)
    {
      this(k, BooleanIndexable.viewArray(values));
    }

    @Override
    public Iterator<BooleanIndexable> iterator()
    {
      return new IteratorConverter<>(createRangeIterator(values.size(), k),
                                     rangeIdx -> new BooleanIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public boolean get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * Combinations of double values.
   */
  public static class OfDouble
          implements Iterable<DoubleIndexable>
  {
    private final int k;
    @NotNull
    private final DoubleIndexable values;

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values double indexable from which there will be chosen
     */
    public OfDouble(int k, @NotNull DoubleIndexable values)
    {
      checkBinomial(values.size(), k);
      this.k = k;
      this.values = values;
    }

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values double values from which there will be chosen
     */
    public OfDouble(int k, @NotNull double ... values)
    {
      this(k, DoubleIndexable.viewArray(values));
    }

    @Override
    public Iterator<DoubleIndexable> iterator()
    {
      return new IteratorConverter<>(createRangeIterator(values.size(), k),
                                     rangeIdx -> new DoubleIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public double get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * Combinations of float values.
   */
  public static class OfFloat
          implements Iterable<FloatIndexable>
  {
    private final int k;
    @NotNull
    private final FloatIndexable values;

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values float indexable from which there will be chosen
     */
    public OfFloat(int k, @NotNull FloatIndexable values)
    {
      checkBinomial(values.size(), k);
      this.k = k;
      this.values = values;
    }

    /**
     * Constructor.
     * @param k   number of elements to choose, has to be non-negative
     *            and not more the size of {@code values}
     * @param values float values from which there will be chosen
     */
    public OfFloat(int k, @NotNull float ... values)
    {
      this(k, FloatIndexable.viewArray(values));
    }

    @Override
    public Iterator<FloatIndexable> iterator()
    {
      return new IteratorConverter<>(createRangeIterator(values.size(), k),
                                     rangeIdx -> new FloatIndexable.Base()
                                     {
                                       @Override
                                       public int size()
                                       {
                                         return rangeIdx.size();
                                       }

                                       @Override
                                       public float get(int index)
                                       {
                                         return values.get(rangeIdx.get(index));
                                       }
                                     });
    }
  }

  /**
   * Create an iterator which iterates of the subsets of size {@code k}
   * from the set {@code {0, 1, 2, ..., n-1}}.
   * <p>
   * As the latter can be used as indices this iterator can be used
   * as a base for chosing from any set.
   * @param n number of elements in the set from which is chosen
   * @param k number of elements in the chosen sub sets
   * @return iterator which returns one possible choice from each set,
   *         with ordered numbers
   */
  @NotNull
  public static Iterator<IntIndexable> createRangeIterator(int n, int k)
  {
    if (n < 0) {
      throw new IllegalArgumentException("n has to be non-negative!");
    }
    if (k < 0  ||  k > n) {
      throw new IllegalArgumentException(String.format("k has to be >=0 and <= %d, but is %d!",
                                                       n, k));
    }
    if (k == 0) {
      // no elements
      return new SingletonIterator<>(IntIndexable.EMPTY);
    }
    if (k == n) {
      // all elements
      return new SingletonIterator<>(IntIndexable.rangeFromSize(n));
    }
    return new RangeIterator(n, k);
  }

  /**
   * N choose K iterator for the set {@code {0, 1, ..., n-1}}.
   * This follows Donald Knuth: &quot;A Draft of Section 7.2.1.3: Creating all Combinations&quot;.
   * It will not work for the degenerate cases <i>n choose 0</i> and <i>n choose n</i>
   * therefore its private, and these special cases are handled
   * by {@link #createRangeIterator(int, int)}, as is the general sanity of both parameters.
   */
  private static class RangeIterator
          implements Iterator<IntIndexable>
  {
    /** Combinations, stored in c[1] to c[k], while c[k+1] and c[k+2] are used by the algorithm.   */
    @NotNull
    private final int[] c;
    /** Number of elements to be chosen. */
    private final int k;
    /** See algorithm description for variable of the same name. */
    private int j;
    /** {@code true} while there are more elements.  */
    private boolean goOn = true;

    /**
     * Constructor.
     * @param n size of the basic set
     * @param k size of the chosen subsets
     */
    RangeIterator(int n, int k) {
      this.k = k;
      c = new int[k + 3];

      // Step T1 of the algorithm
      for (int i = 1; i <= k; ++i) {
        c[i] = i - 1;
      }
      c[k + 1] = n;
      c[k + 2] = 0;
      j = k;
    }

    public boolean hasNext() {
      return goOn;
    }

    @NotNull
    public IntIndexable next() {
      if (!goOn) {
        throw new NoSuchElementException();
      }

      final IntIndexable result = IntIndexable.viewArray(c, 1, k).frozen();

      // T2/T6
      int x = 0;
      if (j > 0) {
        x = j;
        c[j] = x;
        --j;
        return result;
      }
      // T3
      if (c[1] + 1 < c[2]) {
        ++c[1];
        return result;
      }
      else {
        j = 2;
      }
      // T4: find next j
      while (true) {
        c[j - 1] = j - 2;
        x = c[j] + 1;
        if (x == c[j + 1]) {
          ++j;
        }
        else {
          break;
        }
      }
      // T5
      if (j > k) {
        // finished
        goOn = false;
        return result;
      }
      // T6
      c[j] = x;
      --j;
      return result;
    }
  }

  /**
   * Check the parameters of a &quot;n choice k&quot; for sanity.
   * @param n n, size of the set from which is chosen
   * @param k k, number of chosen elements
   * @throws IllegalArgumentException if parameters are invalid
   */
  static void checkBinomial(int n, int k)
  {
    if (n < 0) {
      throw new IllegalArgumentException(String.format("n must not be negative, but is %d!", n));
    }
    if (k < 0  || k > n) {
      throw new IllegalArgumentException(String.format("k has to be >=0 and <= %d, but is %d!",
                                                       n, k));
    }
  }

  /**
   * Get the number of combinations for n choose k.
   * @param n n, size of the set from which is chosen
   * @param k k, number of chosen elements
   * @return number of all possible combinations
   * @throws IllegalArgumentException if parameters are invalid
   */
  @NotNull
  public static BigInteger count(int n, int k)
  {
    checkBinomial(n, k);

    if (k == 0  ||  k == n) {
      return BigInteger.ONE;
    }

    if (n < Permutations.FIRST_FACTORIALS_LONG.size()) {
      // use faster long math
      final LongIndexable f = Permutations.FIRST_FACTORIALS_LONG;
      return BigInteger.valueOf(f.get(n) / (f.get(k) * f.get(n - k)));
    }

    k = Math.max(k, n - k);

    if ((n + k) / 2 < Permutations.FIRST_FACTORIALS.size()) {
      // use the precalculated factorials
      return Permutations.factorial(n)
              .divide(Permutations.factorial(k).multiply(Permutations.factorial(n - k)));
    }

    // calculate direct: n!/k! = (k + 1) * (k + 2) * ... * n
    BigInteger result = BigInteger.ONE;

    for (int i = k + 1;  i <= n;  ++i) {
      result = result.multiply(BigInteger.valueOf(i));
    }
    // apply missing division of (n-k)!
    return result.divide(Permutations.factorial(n - k));
  }

  public static void main(String[] args)
  {
    int count = 0;
    final Set<String> check = new HashSet<>();
    for (String s : new OfString(5, "0123456789")) {
      System.out.println(s);
      if (check.contains(s)) {
        System.err.printf("Duplicate: %s\n", s);
      }
      check.add(s);
      ++count;
    }
    if (count != count(10, 5).intValue()) {
      System.err.println("Incorrect!");
    }
  }
}
