// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.generics.util;

import de.caff.annotation.NotNull;

import java.util.concurrent.atomic.AtomicLong;

/**
 *  Changeable long integer wrapper, to be used as counter especially as map values.
 *  Note that as this is a mutable value, it is a very bad idea to use it as a map key.
 * <p>
 *  This class is thread safe!
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class ThreadSafeLongCounter
        implements LongCounter
{
  private final AtomicLong value;

  /**
   *  Default constructor. Initialize value with 0.
   */
  public ThreadSafeLongCounter() {
    value = new AtomicLong();
  }

  /**
   *  Constructor. Initialize value with {@code val}.
   *  @param   val  init value
   */
  public ThreadSafeLongCounter(int val) {
    value = new AtomicLong(val);
  }

  /**
   *  Copy constructor. Initialize value with given counter.
   *  @param  count  counter to be copied
   */
  public ThreadSafeLongCounter(@NotNull LongCounter count) {
    value = new AtomicLong(count.getValue());
  }

  @Override
  public long add(long add)
  {
    return value.addAndGet(add);
  }

  @Override
  public long add(@NotNull LongCounter counter)
  {
    return add(counter.getValue());
  }

  @Override
  public long add1()
  {
    return add(1);
  }

  @Override
  public long subtract(long sub)
  {
    return add(-sub);
  }

  @Override
  public long subtract(@NotNull LongCounter counter)
  {
    return add(-counter.getValue());
  }

  @Override
  public long subtract1()
  {
    return add(-1);
  }

  @Override
  public long getValue() {
    return value.get();
  }

  @Override
  public void setValue(long val)
  {
    value.set(val);
  }

  @Override
  @NotNull
  public String toString()
  {
    return Long.toString(value.get());
  }
}
