// ============================================================================
// COPYRIGHT NOTICE
// ----------------------------------------------------------------------------
// (This is the open source ISC license, see
// http://en.wikipedia.org/wiki/ISC_license
// for more info)
//
// Copyright © 2012-2024  Andreas M. Rammelt <rammi@caff.de>
//
// Permission to use, copy, modify, and/or distribute this software for any
// purpose with or without fee is hereby granted, provided that the above
// copyright notice and this permission notice appear in all copies.
//
// THE SOFTWARE IS PROVIDED "AS IS" AND THE AUTHOR DISCLAIMS ALL WARRANTIES
// WITH REGARD TO THIS SOFTWARE INCLUDING ALL IMPLIED WARRANTIES OF
// MERCHANTABILITY AND FITNESS. IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR
// ANY SPECIAL, DIRECT, INDIRECT, OR CONSEQUENTIAL DAMAGES OR ANY DAMAGES
// WHATSOEVER RESULTING FROM LOSS OF USE, DATA OR PROFITS, WHETHER IN AN
// ACTION OF CONTRACT, NEGLIGENCE OR OTHER TORTIOUS ACTION, ARISING OUT OF
// OR IN CONNECTION WITH THE USE OR PERFORMANCE OF THIS SOFTWARE.
//=============================================================================
// Latest version on https://caff.de/projects/decaff-commons/
//=============================================================================

package de.caff.generics.util;

import de.caff.annotation.NotNull;

import java.util.concurrent.atomic.AtomicInteger;

/**
 *  Changeable integer wrapper, to be used as counter especially as map values.
 *  Note that as this is a mutable value, it is a very bad idea to use it as a map key.
 * <p>
 *  This class is thread safe!
 *  @author <a href="mailto:rammi@caff.de">Rammi</a>
 */
public final class ThreadSafeCounter
        implements Counter
{
  private final AtomicInteger value;

  /**
   *  Default constructor. Initialize value with 0.
   */
  public ThreadSafeCounter() {
    value = new AtomicInteger();
  }

  /**
   *  Constructor. Initialize value with {@code val}.
   *  @param   val  init value
   */
  public ThreadSafeCounter(int val) {
    value = new AtomicInteger(val);
  }

  /**
   *  Copy constructor. Initialize value with given counter.
   *  @param  count  counter to be copied
   */
  public ThreadSafeCounter(@NotNull Counter count) {
    value = new AtomicInteger(count.getValue());
  }

  /**
   * Add something to the counter.
   *
   * @param add value to add
   * @return counter value
   */
  @Override
  public int add(int add)
  {
    return value.addAndGet(add);
  }

  /**
   * Add something to the counter.
   *
   * @param counter counter which value is added
   * @return counter value
   */
  @Override
  public int add(@NotNull Counter counter)
  {
    return add(counter.getValue());
  }

  /**
   * Add 1 to counter.
   *
   * @return counter value
   */
  @Override
  public int add1()
  {
    return add(1);
  }

  /**
   * Subtract something from the counter.
   *
   * @param sub value to subtract
   * @return counter value
   */
  @Override
  public int subtract(int sub)
  {
    return add(-sub);
  }

  /**
   * Subtract something from the counter.
   *
   * @param counter counter which value is subtracted
   * @return counter value
   */
  @Override
  public int subtract(@NotNull Counter counter)
  {
    return add(-counter.getValue());
  }

  /**
   * Subtract 1 from counter.
   *
   * @return counter value
   */
  @Override
  public int subtract1()
  {
    return add(-1);
  }

  /**
   *  Get the value.
   *  @return  accumulated value
   */
  @Override
  public int getValue() {
    return value.get();
  }

  /**
   * Set the value.
   *
   * @param val new value
   */
  @Override
  public void setValue(int val)
  {
    value.set(val);
  }

  @Override
  public String toString()
  {
    return Integer.toString(value.get());
  }
}
